﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.Resources;
    using Projects;
    using ToolDevelopmentKit;

    internal class WaveSoundOutputFactory : ComponentSetup<SoundArchiveContext, WaveSoundSetBase>
    {
        private FileManager fileManager;

        public WaveSoundOutputFactory(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveSoundSetBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            if (context.Traits.IsWaveSound2BinaryEnabled)
            {
                // WSD ごとに出力する場合
                foreach (var waveSound in component.Children.Cast<WaveSound>().Where(waveSound => waveSound.IsConvertTarget()))
                {
                    Ensure.Operation.True(waveSound.IsWaveSound2BinaryRequired(), "WaveSound must has WaveSoundResource.");

                    IOutput outputTarget = this.fileManager.GetOutput(waveSound);
                    waveSound.SetOutputTarget(outputTarget);

                    context.AddFile(waveSound, outputTarget);

                    HashSet<string> usingKeys = new HashSet<string>();

                    // 波形アーカイブ設定だけは、WSDSET のものを使う
                    foreach (KeyValuePair<string, WaveArchiveBase> item in component.GetWaveArchiveDictionary())
                    {
                        usingKeys.Add(item.Key);

                        if (item.Key.Length == 0)
                        {
                            this.AddOutput(context, outputTarget, waveSound);
                        }
                        else
                        {
                            GroupBase group = context.Groups.GetValue(item.Key);
                            this.AddGroupDependedOutput(context, outputTarget, waveSound, group);
                        }
                    }

                    // HACK : 読み込んだ .depend ファイルが不要な出力アイテムを保持している可能性があるので、
                    //        とりあえずここで削除しています。他に良い場所があれば、移しても構いません。
                    foreach (string key in outputTarget.ItemDictionary.Keys.ToArray())
                    {
                        if (usingKeys.Contains(key))
                        {
                            continue;
                        }

                        outputTarget.ItemDictionary[key].Clean();
                        outputTarget.ItemDictionary.Remove(key);
                    }
                }
            }
            else
            {
                // WSDSET で１つ出力する場合
                IOutput outputTarget = this.fileManager.GetOutput(component);
                component.SetOutputTarget(outputTarget);

                context.AddFile(component, outputTarget);

                HashSet<string> usingKeys = new HashSet<string>();

                foreach (KeyValuePair<string, WaveArchiveBase> item in component.GetWaveArchiveDictionary())
                {
                    usingKeys.Add(item.Key);

                    if (item.Key.Length == 0)
                    {
                        this.AddOutput(context, outputTarget, component);
                    }
                    else
                    {
                        GroupBase group = context.Groups.GetValue(item.Key);
                        this.AddGroupDependedOutput(context, outputTarget, component, group);
                    }
                }

                // HACK : 読み込んだ .depend ファイルが不要な出力アイテムを保持している可能性があるので、
                //        とりあえずここで削除しています。他に良い場所があれば、移しても構いません。
                foreach (string key in outputTarget.ItemDictionary.Keys.ToArray())
                {
                    if (usingKeys.Contains(key))
                    {
                        continue;
                    }

                    outputTarget.ItemDictionary[key].Clean();
                    outputTarget.ItemDictionary.Remove(key);
                }
            }
        }

        private void AddOutput(SoundArchiveContext context, IOutput outputTarget, Component component)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(component);

            string fileName = CreateFileName(context, component);

            if (fileName.Length >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, fileName),
                        component));
            }

            this.fileManager.RegisterCacheItem(outputTarget, string.Empty, fileName);
        }

        private void AddGroupDependedOutput(SoundArchiveContext context, IOutput outputTarget, Component component, GroupBase group)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(group);

            string fileName = CreateFileNameForGroup(context, component, group);

            if (fileName.Length >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, fileName),
                        component));
            }

            this.fileManager.RegisterCacheItem(outputTarget, group.Name, fileName);
        }

        /// <summary>
        /// コンポーネントのファイル名を作成します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileName(SoundArchiveContext context, Component component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(this.fileManager.BinaryOutputTraits.WaveSoundBinaryFileExtension);
            Assertion.Argument.NotNull(this.fileManager.BinaryOutputTraits.WaveSoundBinary2FileExtension);
            Assertion.Argument.StringNotEmpty(this.fileManager.BinaryOutputTraits.WaveSoundBinaryFileExtension);
            Assertion.Argument.StringNotEmpty(this.fileManager.BinaryOutputTraits.WaveSoundBinary2FileExtension);

            if (component.Name.Length == 0)
            {
                throw new Exception("internal error : invalid name.");
            }

            var extension = context.Traits.IsWaveSound2BinaryEnabled
                    ? this.fileManager.BinaryOutputTraits.WaveSoundBinary2FileExtension
                    : this.fileManager.BinaryOutputTraits.WaveSoundBinaryFileExtension;

            return component.Name + "." + extension;
        }

        /// <summary>
        /// コンポーネントのファイル名を作成します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <param name="group">コンポーネントが含まれるグループを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileNameForGroup(SoundArchiveContext context, Component component, GroupBase group)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(this.fileManager.BinaryOutputTraits.WaveSoundBinaryFileExtension);
            Assertion.Argument.NotNull(this.fileManager.BinaryOutputTraits.WaveSoundBinary2FileExtension);
            Assertion.Argument.StringNotEmpty(this.fileManager.BinaryOutputTraits.WaveSoundBinaryFileExtension);
            Assertion.Argument.StringNotEmpty(this.fileManager.BinaryOutputTraits.WaveSoundBinary2FileExtension);

            if (component.Name.Length == 0)
            {
                throw new Exception("internal error : invalid name.");
            }

            var extension = context.Traits.IsWaveSound2BinaryEnabled
                    ? this.fileManager.BinaryOutputTraits.WaveSoundBinary2FileExtension
                    : this.fileManager.BinaryOutputTraits.WaveSoundBinaryFileExtension;

            return string.Format("{0}.{1}{2}.{3}",
                component.Name,
                group.Name,
                SoundArchiveContext.AutoGeneratedNamePostfix,
                extension);
        }
    }
}
