﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;

    internal class WaveArchiveProcessor : ComponentProcessor<SoundArchiveContext, WaveArchiveBase>
    {
        public WaveArchiveProcessor(WaveArchiveBase component, IOutputItem outputItem)
            : base(component, outputItem)
        {
        }

        protected override bool PreProcessInternal(SoundArchiveContext context)
        {
            return base.PreProcessInternal(context);
        }

        protected override void ProcessInternal(SoundArchiveContext context)
        {
            using (Stream stream = this.OutputTargetItem.OpenWrite())
            {
                this.Write(context, context.CreateBinaryWriter(stream));
            }
        }

        protected override void OutputLog(SoundArchiveContext context, Component[] components)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(components);

            context.Logger.AddLine(new InformationLine(
                string.Format("[WARC] {0} > {1}",
                components[0].Name,
                Path.GetFileName(this.OutputTargetItem.Path)
                )));
        }

        private void Write(SoundArchiveContext context, BinaryWriter writer)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(writer);

            List<Stream> waveStreams = new List<Stream>();

            try
            {
                foreach (IOutput outputTarget in
                    (this.TargetComponent as WaveArchiveBase).GetItemOutputTargets())
                {
                    waveStreams.Add(outputTarget.ItemDictionary[string.Empty].OpenRead());
                }

                WaveArchiveBinary waveFile = new WaveArchiveFileBuilder(
                    context.Traits.BinaryFileInfo.WaveArchiveSignature,
                    context.Traits.BinaryFileInfo.WaveArchiveVersion).
                    Build(waveStreams);

                DomElement waveFileElement = new DomBuilder().Build(waveFile);
                new DomWriter(writer).Run(new DomObjectWriter(), waveFileElement);
            }
            finally
            {
                foreach (Stream stream in waveStreams)
                {
                    stream.Close();
                }
            }
        }
    }
}
