﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using Projects;
    using ToolDevelopmentKit;

    /// <summary>
    /// シーケンスサウンドバイナリを作成します。
    /// </summary>
    public class SequenceSoundProcessor : ConversionProcessor<SoundArchiveContext>, IComponentProcessor
    {
        private HashSet<Component> components = new HashSet<Component>();
        private HashSet<Component> relatedComponents = new HashSet<Component>();
        private string inputPath;
        private IOutputItem outputItem;

        public SequenceSoundProcessor(SequenceSoundBase component, IOutputItem outputItem)
        {
            Ensure.Argument.NotNull(component);
            Ensure.Argument.NotNull(outputItem);

            this.components.Add(component);
            this.outputItem = outputItem;

            switch (component.FileType)
            {
                case SequenceSoundFileType.Smf:
                    this.inputPath = this.GetIntemediateFilePath(component);
                    break;

                case SequenceSoundFileType.Text:
                    this.inputPath = component.FilePath;
                    break;

                default:
                    throw new Exception("internal error : invalid sequence file type.");
            }
        }

        /// <summary>
        /// 依存するコンポーネントのコレクションを取得します。
        /// </summary>
        public ICollection<Component> Components
        {
            get { return this.components; }
        }

        /// <summary>
        /// 関連するコンポーネントのコレクションを取得します。
        /// </summary>
        public ICollection<Component> RelatedComponents
        {
            get { return this.relatedComponents; }
        }

        public IOutputItem OutputTargetItem
        {
            get { return this.outputItem; }
        }

        public string ConverterExePath { get; set; }

        protected override bool PreRunInternal(SoundArchiveContext context)
        {
            return this.OutputTargetItem.IsDirty;
        }

        protected override bool RunInternal(SoundArchiveContext context)
        {
            if (this.ConverterExePath == null)
            {
                throw new Exception("internal error : converter exe path not initialized.");
            }

            SequenceSoundTextConverter converter =
                new SequenceSoundTextConverter(this.ConverterExePath)
                {
                    InputPath = this.inputPath,
                    OutputPath = this.outputItem.Path,
                    TargetComponents = this.components.Concat(this.relatedComponents).ToArray(),
                    LogPrefix = "[SEQ] ",
                };

            return converter.Run(context);
        }

        protected override void PostRunInternal(SoundArchiveContext context)
        {
            this.ExtractStartInfo(context);
            this.SetBinaryPaths(context);
        }

        private void ExtractStartInfo(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(outputItem);

            using (Stream fileStream = this.outputItem.OpenRead())
            {
                BinaryReader fileReader = context.CreateBinaryReader(fileStream);

                SequenceSoundBinaryReader sequenceSoundReader = new SequenceSoundBinaryReader();
                sequenceSoundReader.Parse(fileReader);

                foreach (SequenceSoundBase sequenceSound in this.Components)
                {
                    StartPositionInfo startPositionInfo;
                    try
                    {
                        startPositionInfo =
                            sequenceSoundReader.GetStartPositionInfo(
                                                                     fileReader,
                                                                     sequenceSound.StartPosition
                                                                     );
                    }
                    catch (KeyNotFoundException e)
                    {
                        string message = Resources.MessageResource.Message_SequenceStartLabelNotFound;
                        message = string.Format(message, e.Message);

                        throw new ConversionException(new ErrorLine(message, sequenceSound));
                    }

                    sequenceSound.SetStartOffset((uint)startPositionInfo.OffsetFromDataBlockBody);
                    sequenceSound.SetAllocateTrackFlags(startPositionInfo.AllocateTrackFlags);
                }
            }
        }

        private void SetBinaryPaths(SoundArchiveContext context)
        {
            if (!context.Settings.IsConvertParts)
            {
                return;
            }

            foreach (SequenceSoundBase sequenceSound in this.Components)
            {
                // bxseq ファイルパスを設定します。
                sequenceSound.SetBinaryFilePathForPartsConvert(this.OutputTargetItem.Path);
            }
        }

        private string GetIntemediateFilePath(SequenceSoundBase component)
        {
            Assertion.Argument.NotNull(component);

            try
            {
                return component.GetOutputTarget().
                    ItemDictionary[SequenceSoundProcessorFactory.OutputID_Intermediate].Path;
            }
            catch (Exception exception)
            {
                throw new Exception("internal error : intermediate output item not found.", exception);
            }
        }
    }
}
