﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// コンポーネントを処理する ConversionProcessor です。
    /// </summary>
    /// <typeparam name="TComponent">コンポーネントの型を指定します。</typeparam>
    internal abstract class ComponentProcessor<TContext, TComponent> :
        ConversionProcessor<TContext>, IComponentProcessor
        where TContext : ConversionContext
        where TComponent : Component
    {
        private HashSet<Component> components = new HashSet<Component>();
        private HashSet<Component> relatedComponents = new HashSet<Component>();
        private IOutputItem outputItem;

        public ComponentProcessor(TComponent component, IOutputItem outputItem)
        {
            Ensure.Argument.NotNull(component);
            Ensure.Argument.NotNull(outputItem);

            this.components.Add(component);
            this.outputItem = outputItem;
        }

        /// <summary>
        /// 依存するコンポーネントのコレクションを取得します。
        /// </summary>
        public ICollection<Component> Components
        {
            get { return this.components; }
        }

        /// <summary>
        /// 関連するコンポーネントのコレクションを取得します。
        /// </summary>
        public ICollection<Component> RelatedComponents
        {
            get { return this.relatedComponents; }
        }

        public IOutputItem OutputTargetItem
        {
            get { return this.outputItem; }
        }

        protected Component TargetComponent
        {
            get { return this.components.First(); }
        }

        protected sealed override bool PreRunInternal(TContext context)
        {
            Assertion.Argument.NotNull(context);
            return this.PreProcessInternal(context);
        }

        protected sealed override bool RunInternal(TContext context)
        {
            Assertion.Argument.NotNull(context);

            Component[] components = this.components.Concat(this.relatedComponents).ToArray();
            this.OutputLog(context, components);

            try
            {
                this.ProcessInternal(context);
            }
            catch (IOException exception)
            {
                context.Logger.AddLine(
                    new ErrorLine(exception.Message, components)
                    );
                return false;
            }

            return true;
        }

        protected sealed override void PostRunInternal(TContext context)
        {
            Assertion.Argument.NotNull(context);
            this.PostProcessInternal(context);
        }

        protected virtual bool PreProcessInternal(TContext context)
        {
            return this.OutputTargetItem.IsDirty;
        }

        protected abstract void ProcessInternal(TContext context);

        protected virtual void PostProcessInternal(TContext context)
        {
        }

        protected virtual void OutputLog(TContext context, Component[] components)
        {
        }
    }
}
