﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;
    using FileFormats.NintendoSdk;
    using System.Text.RegularExpressions;
    using System.Linq;

    /// <summary>
    /// バンクバイナリの作成に必要なコンテキストを構築します。
    /// </summary>
    internal class BankBuilder
    {
        private FileManager fileManager;

        private SoundProjectService _soundProjectService;
        private BankService bankService;
        private WaveArchiveBase waveArchive;
        private string sourceName;

        public BankBuilder(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        public bool IsInitialized
        {
            get { return this.bankService != null; }
        }

        public void Initialize(SoundProjectService soundProjectService, BankService bankService, WaveArchiveBase waveArchive, string sourceName)
        {
            Ensure.Argument.NotNull(bankService);
            Ensure.Argument.NotNull(bankService.BankDocument);

            _soundProjectService = soundProjectService;
            this.bankService = bankService;
            this.waveArchive = waveArchive;
            this.sourceName = sourceName;
        }

        public bool Build(BankContext context, IOutput output, IOutputItem outputItem, IOutputItem includeOutputItem)
        {
            Ensure.Argument.NotNull(context);
            Ensure.Argument.NotNull(outputItem);

            if (!IsInitialized)
            {
                throw new Exception("internal error : not initialized.");
            }

            this.ValidateNames(context);
            this.ValidateFilePaths(context);
            this.ValidateProgramNos(context);

            if (context.IsFailed)
            {
                return false;
            }

            this.CreateOutputItems(context);
            this.CreateComponentProcessors(context);

            this.LinkDependedFiles(context);

            this.CreateBankProcessor(context, output, outputItem, includeOutputItem);

            return true;
        }

        //-----------------------------------------------------------------
        // 名前の検証
        //-----------------------------------------------------------------

        private void ValidateNames(BankContext context)
        {
            Assertion.Argument.NotNull(context);
            ValidateNameDuplication(context);
        }

        private void ValidateNameDuplication(BankContext context)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Operation.True(this.bankService != null);
            Assertion.Operation.True(this.bankService.ComponentDictionary != null);

            foreach (string name in this.bankService.ComponentDictionary.DuplicatedNames)
            {
                Component[] components = this.bankService.ComponentDictionary[name];

                foreach (Component component in components)
                {
                    bool doesEnableComponentFound = false;

                    if (!(component is KeyRegion || component is VelocityRegion || !component.IsConvertTarget()))
                    {
                        if (doesEnableComponentFound)
                        {
                            context.Logger.AddLine(
                                new ErrorLine(
                                    string.Format(Resources.MessageResource.Message_NameAlreadyExists, name,
                                    components[0].Name), components
                                    ));
                            break;
                        }

                        doesEnableComponentFound = true;
                    }
                }
            }
        }

        //-----------------------------------------------------------------
        // ファイルパスの検証
        //-----------------------------------------------------------------

        private void ValidateFilePaths(BankContext context)
        {
            Assertion.Argument.NotNull(context);

            ParameterValidatorSelector selector = new ParameterValidatorSelector();

            foreach (VelocityRegion velocityRegion in this.bankService.VelocityRegions)
            {
                //  Instrument が無効なら処理しません。
                if (!velocityRegion.Instrument.IsConvertTarget())
                {
                    continue;
                }

                selector.Run(context, velocityRegion);
            }
        }

        //-----------------------------------------------------------------
        // プログラム番号の検証
        //-----------------------------------------------------------------

        private void ValidateProgramNos(BankContext context)
        {
            Assertion.Argument.NotNull(context);

            HashSet<int> existedProgramNos = new HashSet<int>();

            foreach (Instrument instrument in this.bankService.Instruments)
            {
                if (!instrument.IsConvertTarget())
                {
                    continue;
                }

                if (instrument.ProgramNo < 0 || Instrument.ProgramNoMax < instrument.ProgramNo)
                {
                    context.Logger.AddLine(
                        new ErrorLine(
                            string.Format(
                                Resources.MessageResource.Message_ProgramNoOutOfRange,
                                instrument.ProgramNo),
                            instrument));
                }

                if (existedProgramNos.Contains(instrument.ProgramNo))
                {
                    context.Logger.AddLine(
                        new ErrorLine(
                            string.Format(
                                Resources.MessageResource.Message_ProgramNoAlreadyExisting,
                                instrument.ProgramNo),
                            instrument));
                }
                else
                {
                    existedProgramNos.Add(instrument.ProgramNo);
                }
            }
        }

        //-----------------------------------------------------------------
        // 出力アイテムの生成
        //-----------------------------------------------------------------

        private void CreateOutputItems(BankContext context)
        {
            Assertion.Argument.NotNull(context);

            OutputItemFactorySelector runner = new OutputItemFactorySelector(this.fileManager);

            foreach (VelocityRegion velocityRegion in this.bankService.VelocityRegions)
            {
                //  Instrument が無効なら処理しません。
                if (!velocityRegion.Instrument.IsConvertTarget())
                {
                    continue;
                }

                runner.Run(context, velocityRegion);
            }
        }

        //-----------------------------------------------------------------
        // ConversionProcessor の生成
        //-----------------------------------------------------------------

        private void CreateComponentProcessors(BankContext context)
        {
            Assertion.Argument.NotNull(context);

            ComponentProcessorFactorySelector runner = new ComponentProcessorFactorySelector();

            foreach (VelocityRegion velocityRegion in this.bankService.VelocityRegions)
            {
                //  Instrument が無効なら処理しません。
                if (!velocityRegion.Instrument.IsConvertTarget())
                {
                    continue;
                }

                runner.Run(context, velocityRegion);
            }
        }

        //-----------------------------------------------------------------
        // ファイル依存関係の構築
        //-----------------------------------------------------------------

        private void LinkDependedFiles(BankContext context)
        {
            Assertion.Argument.NotNull(context);

            DependedFileLinkerSelector runner = new DependedFileLinkerSelector(_soundProjectService, this.fileManager);

            foreach (VelocityRegion velocityRegion in this.bankService.VelocityRegions)
            {
                //  Instrument が無効なら処理しません。
                if (!velocityRegion.Instrument.IsConvertTarget())
                {
                    continue;
                }

                runner.Run(context, velocityRegion);
            }
        }

        //-----------------------------------------------------------------
        // バンクプロセッサ
        //-----------------------------------------------------------------

        private void CreateBankProcessor(
            BankContext context,
            IOutput output,
            IOutputItem outputItem,
            IOutputItem includeOutputItem)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(output);
            Assertion.Argument.NotNull(outputItem);

            var bankProcessor =
                new BankProcessor(this.bankService.Bank, this.waveArchive, this.sourceName, outputItem, includeOutputItem);

            context.AddComponentProcessor(output, string.Empty, bankProcessor);
        }
    }
}
