﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.Codecs;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Resources;
    using NintendoWare.ToolDevelopmentKit;
    using WaveFormats = NintendoWare.SoundFoundation.FileFormats.Wave;

    public class WaveConverter
    {
        private const int MaxPath = 256;

        private List<string> warningMessages;

        private readonly string signature;
        private readonly BinaryVersion version;
        private readonly bool isLittleEndian = true;

        public WaveConverter(string signature, BinaryVersion version, bool isLittleEndian)
        {
            Ensure.Argument.StringNotEmpty(signature);

            this.signature = signature;
            this.version = version;
            this.isLittleEndian = isLittleEndian;

            warningMessages = new List<string>();
        }

        public void Convert(string filePath, string outputFilePath, WaveEncoding waveEncoding,
                            int loopStartFrame, int loopEndFrame)
        {
            bool loop = false;

            if (loopStartFrame >= 0 || loopEndFrame >= 0)
            {
                loop = true;
            }

            ConversionContext context = new ConversionContext(
                new ConversionTraits(this.isLittleEndian, MaxPath));

            WaveEncoding encoding = waveEncoding;

            using (WaveFormats.WaveFileReader reader =
                   WaveFormats.WaveFileReader.CreateInstance(filePath))
            {
                reader.LoopEndFrameModified += OnLoopEndFrameModified;

                WaveFormats.WaveFile srcFile = reader.Open(filePath);

                if (loopStartFrame < 0)
                {
                    if ((int)srcFile.LoopStartFrame < 0)
                    {
                        loopStartFrame = 0;
                    }
                    else
                    {
                        loopStartFrame = (int)srcFile.LoopStartFrame;
                    }
                }
                if (loopEndFrame < 0)
                {
                    if ((int)srcFile.LoopEndFrame < 0)
                    {
                        loopEndFrame = (int)srcFile.FrameCount;
                    }
                    else
                    {
                        loopEndFrame = (int)srcFile.LoopEndFrame;
                    }
                }
                if ((int)srcFile.FrameCount < loopEndFrame)
                {
                    loopEndFrame = (int)srcFile.FrameCount;
                }

                loop = loop || (srcFile.IsLoop &&
                                ((int)srcFile.LoopStartFrame != (int)srcFile.LoopEndFrame));

                if (srcFile.IsLoop && ((int)srcFile.LoopStartFrame > (int)srcFile.LoopEndFrame))
                {
                    throw new Exception("internal error : loop start frame must be before loop end frame.");
                }

                WaveFormat waveFormat = new WaveFormat()
                {
                    BitsPerSample = srcFile.SampleBit,
                    ChannelCount = srcFile.ChannelCount,
                    Encoding = Encoding.Pcm,
                    HasLoop = loop,
                    IsLittleEndian = srcFile is WaveFormats.WaveFileWav,
                    IsSigned = !(
                                     (srcFile is WaveFormats.WaveFileWav) &&
                                     srcFile.SampleBit == 8
                                     ),
                    OriginalLoopStartFrame = loopStartFrame,
                    OriginalLoopEndFrame = loopEndFrame,
                    LoopStartFrame = loopStartFrame,
                    LoopEndFrame = loopEndFrame,
                    SamplingRate = srcFile.SampleRate,
                };

                NintendoWareWaveEncoder encoder = this.CreateEncoder(encoding);

                using (var waveStream = new WaveStream(waveFormat, reader.OpenDataStream()))
                {
                    CodecOutput[] codecOutputs = encoder.Run(waveStream);

                    foreach (CodecOutput codecOutput in codecOutputs)
                    {
                        if (codecOutput.Format.HasLoop)
                        {
                            continue;
                        }

                        codecOutput.Format = new WaveFormat(codecOutput.Format)
                        {
                            OriginalLoopStartFrame = 0,
                            OriginalLoopEndFrame = 0,
                            LoopStartFrame = 0,
                            LoopEndFrame = (int)srcFile.FrameCount,
                        };
                    }

                    WaveBinary file = new WaveFileBuilder(this.signature, this.version).Build(encoding, codecOutputs);

                    DomElement fileElement = new DomBuilder().Build(file);

                    Stream stream = File.Open(outputFilePath, FileMode.Create, FileAccess.Write);
                    BinaryWriter writer = context.CreateBinaryWriter(stream);

                    new DomWriter(writer).Run(new DomObjectWriter(), fileElement);
                }
            }

            foreach (string message in this.warningMessages)
            {
                Console.Error.WriteLine(new WarningLine(message));
            }
            this.warningMessages.Clear();
        }

        private NintendoWareWaveEncoder CreateEncoder(WaveEncoding encoding)
        {
            switch (encoding)
            {
                case WaveEncoding.ImaAdpcm:
                    return new NintendoWareImaAdpcmEncoder();

                case WaveEncoding.Adpcm:
                    return new NintendoWareDspAdpcmEncoder();

                case WaveEncoding.Pcm16:
                    return new NintendoWareLinearPcmEncoder()
                    {
                        BitsPerSample = 16,
                        IsLittleEndian = this.isLittleEndian,
                    };

                case WaveEncoding.Pcm8:
                    return new NintendoWareLinearPcmEncoder()
                    {
                        BitsPerSample = 8,
                    };
            }

            throw new Exception("internal error : invalid wave encoding.");
        }

        private void OnLoopEndFrameModified(object send, LoopEndFrameModifiedEventArgs e)
        {
            string format = string.Format(
                MessageResource.Message_LoopEndFrameModified,
                e.LoopEndFrame,
                e.FrameCount,
                e.FilePath);

            this.warningMessages.Add(format);
        }
    }
}
