﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Management;
    using System.Text;
    using NintendoWare.SoundFoundation.Core.Diagnostics;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// サウンドプロジェクトをコンバートし、サウンドアーカイブバイナリを生成します。
    /// </summary>
    public class SoundProjectConverter : ISoundProjectConverter2
    {
        private const string OutputListFilePathEnvironmentVariableName = "OUTPUT_LIST_FILE_PATH";

        private static uint processorCoreCount = uint.MaxValue;

        private readonly object runLock = new object();

        private SoundProjectConversionTraits traits;
        private SoundProjectConversionSettings settings;
        private SoundArchiveContext context;
        private Logger logger;
        private FileManager fileManager;
        private SoundProjectService projectService;
        private SoundSet addonSoundSet;

        private ConsoleProcess consoleProcess = null;
        private ConversionProcessScheduler conversionProcessScheduler;
        private Stopwatch stopWatch = new Stopwatch();
        private bool canPostRun = false;
        private bool doGarbageCollection = true;
        private bool isCanceling = false;
        private bool isCanceled = false;

        public SoundProjectConverter(SoundProjectConversionTraits traits)
        {
            Ensure.Argument.NotNull(traits);
            this.traits = traits;

            this.settings = new SoundProjectConversionSettings();

            this.logger = new Logger();
            this.logger.LineAdded += OnLineAdded;
        }

        public event EventHandler<OutputLineEventArgs> LineOutput;

        public event EventHandler ConversionCompleted;

        public event EventHandler<CustomConversionEventArgs> PreConvert;

        public event EventHandler<CustomConversionEventArgs> PostConvert;

        public SoundProjectConversionTraits Traits
        {
            get { return this.traits; }
        }

        public SoundProjectConversionSettings Settings
        {
            get { return this.settings; }
        }

        public bool IsSucceeded
        {
            get
            {
                if (this.conversionProcessScheduler != null)
                {
                    return this.conversionProcessScheduler.IsSucceeded;
                }

                return false;
            }
        }

        public bool IsFailed
        {
            get
            {
                if (this.conversionProcessScheduler != null)
                {
                    return this.conversionProcessScheduler.IsFailed;
                }

                return false;
            }
        }

        public bool IsCanceled
        {
            get
            {
                if (this.IsSucceeded)
                {
                    return false;
                }

                if (this.isCanceled)
                {
                    return true;
                }

                if (this.conversionProcessScheduler == null)
                {
                    return false;
                }

                return this.conversionProcessScheduler.IsCanceled;
            }
        }

        public int ProgressMax
        {
            get
            {
                if (this.conversionProcessScheduler == null)
                {
                    return 0;
                }

                return this.conversionProcessScheduler.TotalProcessCount;
            }
        }

        public int ProgressCurrent
        {
            get
            {
                if (this.conversionProcessScheduler == null)
                {
                    return 0;
                }

                return this.conversionProcessScheduler.CompletedProcessCount;
            }
        }

        /// <summary>
        /// コンバート対象のサウンドプロジェクトファイルパスを取得します。
        /// </summary>
        private string TargetProjectFilePath
        {
            get
            {
                if (this.projectService == null)
                {
                    return string.Empty;
                }

                return this.projectService.ProjectDocument.Resource.Key;
            }
        }

        /// <summary>
        /// 作業ディレクトリパスを取得します。
        /// </summary>
        private string WorkDirectoryPath
        {
            get { return Path.GetDirectoryName(TargetProjectFilePath); }
        }

        public void Prepare(SoundProjectService projectService)
        {
            Ensure.Argument.NotNull(projectService);

            this.logger.AddLine(new InformationLine(
                string.Format(Resources.MessageResource.Message_BeginConvert, projectService.Project.Name)
                ));

            this.projectService = projectService;
        }

        public void Cleanup()
        {
            if (this.projectService == null)
            {
                return;
            }

            this.context = null;
            this.addonSoundSet = null;
            this.projectService = null;
        }

        public void Run(IEnumerable<BankService> bankServices)
        {
            this.Run(null, bankServices, null, false);
        }

        public void Run(IEnumerable<BankService> bankServices, Func<string, bool> isCacheUse)
        {
            this.Run(null, bankServices, isCacheUse, false);
        }

        public void Run(IEnumerable<BankService> bankServices, Func<string, bool> isCacheUse, bool isForced)
        {
            this.Run(null, bankServices, isCacheUse, isForced);
        }

        public void Run(SoundSet addonSoundSet, IEnumerable<BankService> bankServices, Func<string, bool> isCacheUse, bool isForced)
        {
            Ensure.Argument.NotNull(bankServices);
            Ensure.Operation.ObjectNotNull(this.projectService);

            uint executionCountMax = this.Settings.ParallelConversionCountMax;

            if (executionCountMax == 0)
            {
                executionCountMax = this.GetProcessorCoreCount();
            }

            lock (this.runLock)
            {
                this.conversionProcessScheduler = new ConversionProcessScheduler()
                {
                    ExecutionCountMax = executionCountMax,
                };

                this.stopWatch.Reset();
                this.stopWatch.Start();

                SoundArchiveBuilder builder = null;

                try
                {
                    this.context = new SoundArchiveContext(this.traits, this.Settings, projectService.ProjectFilePath)
                    {
                        Logger = this.logger,
                    };

                    this.addonSoundSet = addonSoundSet;
                    this.doGarbageCollection = true;
                    this.isCanceled = false;

                    if (!this.PreRun(this.context))
                    {
                        this.ErrorExit(projectService.Project, false);
                        return;
                    }

                    if (this.isCanceling || this.IsCanceled)
                    {
                        this.CancelExit(projectService.Project);
                        this.OnConversionCompleted(EventArgs.Empty);
                        return;
                    }

                    this.fileManager = new FileManager(
                        this.traits.IntermediateOutputTraits,
                        this.traits.BinaryOutputTraits);

                    string dependFilePath = this.projectService.ProjectDocument.GetDependFilePath(this.addonSoundSet);

                    // 注意
                    // CacheDiretoryPath には、DependFilePath のディレクトリからの相対パスを渡します
                    // OutputDirectoryPath には、CacheDiretoryPath からの相対パスを渡します
                    this.fileManager.Initialize(
                        new FileManager.InitializeArgs()
                        {
                            DependFilePath = dependFilePath,
                            OutputDirectoryPath = "../" + PathEx.MakeRelative(
                                this.projectService.ProjectOutputPath,
                                Path.GetDirectoryName(dependFilePath)),
                            CacheDiretoryPath = "./" + this.GetSoundArchiveName() + "/",
                            IsCacheUseFunc = isCacheUse,
                            IsAllCachesClean = isForced,
                            IsKeepAllCaches = false,
                            IsWaveSound2Enabled = context.Traits.IsWaveSound2BinaryEnabled,
                        });

                    if (isForced)
                    {
                        this.fileManager.Clean();
                    }

                    builder = new SoundArchiveBuilder(this.fileManager);
                    builder.Initialize(projectService, addonSoundSet, bankServices);

                    if (!builder.Build(this.context))
                    {
                        this.ErrorExit(projectService.Project, false);
                        return;
                    }
                }
                catch
                {
                    if (this.context != null)
                    {
                        if (builder != null)
                        {
                            builder.Cleanup(this.context);
                        }
                    }

                    this.ErrorExit(projectService.Project, false);
                    throw;
                }

                try
                {
                    this.canPostRun = true;

                    this.conversionProcessScheduler.Completed += (sender, e) => builder.Cleanup(this.context);
                    this.conversionProcessScheduler.Completed += OnProcessCompleted;

                    this.logger.AddLine(new InformationLine(
                        string.Format(Resources.MessageResource.Message_ConvertingFrom, this.GetSoundArchiveName())
                        ));

                    this.conversionProcessScheduler.IsForced = isForced;
                    this.conversionProcessScheduler.Run(this.context);
                }
                catch
                {
                    if (this.context != null)
                    {
                        if (builder != null)
                        {
                            builder.Cleanup(this.context);
                        }
                    }

                    this.ErrorExit(projectService.Project);
                    throw;
                }
            }
        }

        public void RunParts(
            IEnumerable<BankService> bankServices,
            IEnumerable<SoundSetItem> soundSetItems,
            string outputDirectoryPath,
            bool doGarbageCollection)
        {
            Ensure.Argument.NotNull(bankServices);
            Ensure.Argument.NotNull(soundSetItems);
            Ensure.Argument.StringNotEmpty(outputDirectoryPath);
            Ensure.Operation.ObjectNotNull(this.projectService);

            uint executionCountMax = this.Settings.ParallelConversionCountMax;

            if (executionCountMax == 0)
            {
                executionCountMax = this.GetProcessorCoreCount();
            }

            lock (this.runLock)
            {
                this.conversionProcessScheduler = new ConversionProcessScheduler()
                {
                    ExecutionCountMax = executionCountMax,
                };

                this.stopWatch.Reset();
                this.stopWatch.Start();

                SoundArchiveBuilder builder = null;

                try
                {
                    this.context = new SoundArchiveContext(this.traits, this.Settings, projectService.ProjectFilePath)
                    {
                        Logger = this.logger,
                    };

                    this.doGarbageCollection = doGarbageCollection;
                    this.isCanceled = false;

                    if (this.isCanceling || this.IsCanceled)
                    {
                        this.CancelExit(projectService.Project);
                        this.OnConversionCompleted(EventArgs.Empty);
                        return;
                    }

                    this.fileManager = new FileManager(
                        this.traits.IntermediateOutputTraits,
                        this.traits.BinaryOutputTraits);

                    string dependFilePath = this.projectService.ProjectDocument.GetDependFilePath();

                    this.fileManager.Initialize(
                        new FileManager.InitializeArgs()
                        {
                            DependFilePath = Path.Combine(outputDirectoryPath, Path.GetFileName(dependFilePath)),
                            OutputDirectoryPath = "../output",
                            CacheDiretoryPath = "./cache",
                            IsAllCachesClean = false,
                            IsKeepAllCaches = !doGarbageCollection,
                            IsWaveSound2Enabled = context.Traits.IsWaveSound2BinaryEnabled,
                        });

                    builder = new SoundArchiveBuilder(this.fileManager);
                    builder.Initialize(projectService, null, bankServices);

                    if (!builder.BuildParts(this.context, soundSetItems))
                    {
                        this.ErrorExit(projectService.Project, false);
                        return;
                    }
                }
                catch
                {
                    if (this.context != null)
                    {
                        if (builder != null)
                        {
                            builder.Cleanup(this.context);
                        }
                    }

                    this.ErrorExit(projectService.Project, false);
                    throw;
                }

                try
                {
                    this.canPostRun = false;

                    this.conversionProcessScheduler.Completed += (sender, e) => builder.Cleanup(this.context);
                    this.conversionProcessScheduler.Completed += OnProcessCompleted;

                    this.conversionProcessScheduler.IsForced = false;
                    this.conversionProcessScheduler.Run(this.context);
                }
                catch
                {
                    if (this.context != null)
                    {
                        if (builder != null)
                        {
                            builder.Cleanup(this.context);
                        }
                    }

                    this.ErrorExit(projectService.Project);
                    throw;
                }
            }
        }

        public bool ExecutePreConvertCommands()
        {
            if (this.Settings.IsPreConvertCommandsIgnored) { return true; }

            if (!this.projectService.Project.IsPreConvertCommandsEnabled) { return true; }
            if (0 == this.projectService.Project.PreConvertCommands.Count) { return true; }

            List<string> commandLines = new List<string>();

            foreach (string line in this.projectService.Project.PreConvertCommands)
            {
                if (line.Length == 0)
                {
                    continue;
                }

                commandLines.Add(line);
            }

            if (commandLines.Count == 0)
            {
                return true;
            }

            this.logger.AddLine(new InformationLine(Resources.MessageResource.Message_ExecutePreConvertCommands));

            return this.ExecuteCommands(commandLines, this.WorkDirectoryPath, this.logger, null);
        }

        public bool ExecutePostConvertCommands(IEnumerable<string> outputFilePaths)
        {
            if (this.Settings.IsPostConvertCommandsIgnored) { return true; }

            if (!this.projectService.Project.IsPostConvertCommandsEnabled) { return true; }
            if (0 == this.projectService.Project.PostConvertCommands.Count) { return true; }

            var outputListFilePath = this.CreateOutputList(outputFilePaths);

            try
            {
                List<string> commandLines = new List<string>();

                foreach (string line in this.projectService.Project.PostConvertCommands)
                {
                    if (line.Length == 0)
                    {
                        continue;
                    }

                    commandLines.Add(line);
                }

                if (commandLines.Count == 0)
                {
                    return true;
                }

                this.logger.AddLine(new InformationLine(Resources.MessageResource.Message_ExecutePostConvertCommands));

                return this.ExecuteCommands(
                    commandLines,
                    this.WorkDirectoryPath,
                    this.logger,
                    new Tuple<string, string>[]
                    {
                        new Tuple<string, string>(OutputListFilePathEnvironmentVariableName, outputListFilePath)
                    });
            }
            finally
            {
                File.Delete(outputListFilePath);
            }
        }

        public void Cancel()
        {
            lock (this.runLock)
            {
                if (this.conversionProcessScheduler == null)
                {
                    this.isCanceling = true;
                    return;
                }

                this.conversionProcessScheduler.Cancel();

                if (this.consoleProcess != null)
                {
                    try
                    {
                        this.consoleProcess.Cancel();
                    }
                    catch (InvalidOperationException)
                    {
                        // プロセスが既に終了しています。
                    }
                }
            }
        }

        public void Wait()
        {
            this.conversionProcessScheduler.Wait();
        }

        protected virtual void OnLineOutput(OutputLineEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.LineOutput != null)
            {
                this.LineOutput(this, e);
            }
        }

        protected virtual void OnConversionCompleted(EventArgs e)
        {
            if (this.ConversionCompleted != null)
            {
                this.ConversionCompleted(this, e);
            }
        }

        protected virtual void OnPreConvert(CustomConversionEventArgs e)
        {
            if (this.PreConvert != null)
            {
                this.PreConvert(this, e);
            }
        }

        protected virtual void OnPostConvert(CustomConversionEventArgs e)
        {
            if (this.PostConvert != null)
            {
                this.PostConvert(this, e);
            }
        }

        private bool PreRun(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            this.OnPreConvert(new CustomConversionEventArgs(context.Logger));

            return !context.IsFailed;
        }

        private bool PostRun(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);

            var args = new CustomConversionEventArgs(context.Logger);

            if (context.SoundArchiveOutput != null &&
                !string.IsNullOrEmpty(context.SoundArchiveOutput.Path))
            {
                args.OutputFilePaths.Add(PathUtility.GetFullPath(context.SoundArchiveOutput.Path));
            }

            if (context.SoundArchiveMapOutput != null &&
                !string.IsNullOrEmpty(context.SoundArchiveMapOutput.Path))
            {
                args.OutputFilePaths.Add(PathUtility.GetFullPath(context.SoundArchiveMapOutput.Path));
            }

            // bfstm, bfgrp
            // HACK : RegisteringOutputs は暫定措置で追加されたプロパティなので、廃止される際には、
            //        bfstm, bfgrp （外部ファイル）パスの取得方法も別の方法に差し替える必要があります。
            foreach (var filePath in context.RegisteringOutputs
                .Select(output => output.TargetOutput.ItemDictionary.First().Value.Path))
            {
                args.OutputFilePaths.Add(PathUtility.GetFullPath(filePath));
            }

            this.OnPostConvert(args);

            return !context.IsFailed;
        }

        private void SuccessExit(SoundProject project)
        {
            Assertion.Argument.NotNull(project);

            this.logger.AddLine(new InformationLine(
                string.Format(
                    this.conversionProcessScheduler.IsForced == false ?
                        Resources.MessageResource.Message_EndConvertSucceeded_WithTime :
                        Resources.MessageResource.Message_EndReConvertSucceeded_WithTime,
                    DateTime.Now.ToString("G"),
                    this.GetSoundArchiveName(),
                    this.GetElapsedTimeMinutes(this.stopWatch),
                    this.GetElapsedTimeSecondsAndMilliseconds(this.stopWatch))
                ));

            this.Exit();
        }

        private void CancelExit(SoundProject project)
        {
            Assertion.Argument.NotNull(project);

            this.logger.AddLine(new InformationLine(
                string.Format(Resources.MessageResource.Message_EndConvertCanceled, project.Name)
                ));

            this.isCanceled = true;
            this.Exit();
        }

        private void ErrorExit(SoundProject project)
        {
            this.ErrorExit(project, true);
        }

        private void ErrorExit(SoundProject project, bool isSaveDependencies)
        {
            Assertion.Argument.NotNull(project);

            this.logger.AddLine(new InformationLine(
                string.Format(Resources.MessageResource.Message_EndConvertFailed, project.Name)
                ));

            this.Exit(isSaveDependencies);
        }

        private void Exit()
        {
            this.Exit(true);
        }

        private void Exit(bool isSaveDependencies)
        {
            if (this.fileManager != null)
            {
                if (isSaveDependencies)
                {
                    try
                    {
                        this.fileManager.SaveDependencies(this.doGarbageCollection);
                    }
                    catch
                    {
                    }
                }

                this.fileManager = null;
            }

            this.isCanceling = false;
            this.consoleProcess = null;
        }

        private int GetElapsedTimeMinutes(Stopwatch stopWatch)
        {
            Assertion.Argument.NotNull(stopWatch);
            return stopWatch.Elapsed.Minutes;
        }

        private float GetElapsedTimeSecondsAndMilliseconds(Stopwatch stopWatch)
        {
            Assertion.Argument.NotNull(stopWatch);
            return ((float)stopWatch.Elapsed.Seconds +
                    ((float)stopWatch.Elapsed.Milliseconds / 1000.0F));
        }

        private float GetElapsedTime(Stopwatch stopWatch)
        {
            Assertion.Argument.NotNull(stopWatch);
            return ((float)stopWatch.ElapsedMilliseconds) / 1000f;
        }

        private bool ExecuteCommands(
            IEnumerable commands,
            string workDirectoryPath,
            ILogger logger,
            IEnumerable<Tuple<string, string>> environmentVariables)
        {
            Assertion.Argument.NotNull(commands);
            Assertion.Argument.NotNull(logger);

            try
            {
                using (this.consoleProcess = new ConsoleProcess())
                {
                    if (environmentVariables != null)
                    {
                        foreach (var environmentVariable in environmentVariables)
                        {
                            this.consoleProcess.EnvironmentVariables.Add(environmentVariable.Item1, environmentVariable.Item2);
                        }
                    }

                    this.consoleProcess.OutputLineReceived += (sender, e) => this.AddInformationLineToLogger(logger, e.Line);
                    this.consoleProcess.ErrorLineReceived += (sender, e) => this.AddErrorLineToLogger(logger, e.Line);

                    if (!this.consoleProcess.Start(workDirectoryPath))
                    {
                        return false;
                    }

                    foreach (string command in commands)
                    {
                        if (this.IsCanceled)
                        {
                            break;
                        }

                        this.consoleProcess.WriteCommandLine(command, true);
                    }

                    consoleProcess.Exit();
                }
            }
            catch (Exception exception)
            {
                logger.AddLine(new ErrorLine(exception.Message));
            }
            finally
            {
                this.consoleProcess = null;
            }

            return true;
        }

        private void AddInformationLineToLogger(ILogger logger, string line)
        {
            Assertion.Argument.NotNull(logger);
            Assertion.Argument.NotNull(line);

            if (line.Length == 0)
            {
                return;
            }

            logger.AddLine(new InformationLine(line));
        }

        private void AddErrorLineToLogger(ILogger logger, string line)
        {
            Assertion.Argument.NotNull(logger);
            Assertion.Argument.NotNull(line);

            if (line.Length == 0)
            {
                return;
            }

            logger.AddLine(new ErrorLine(line));
        }

        private uint GetProcessorCoreCount()
        {
            // WMI クエリ処理に少し時間がかかるので、結果をキャッシュしておきます。
            if (SoundProjectConverter.processorCoreCount != uint.MaxValue)
            {
                return SoundProjectConverter.processorCoreCount;
            }

            uint coreCountTotal = 0;

            try
            {
                ManagementObjectCollection result = new ManagementObjectSearcher("SELECT * FROM Win32_Processor").Get();

                if (result != null)
                {
                    foreach (ManagementObject obj in result)
                    {
                        PropertyData numberOfCores = obj.Properties["NumberOfCores"];

                        if (numberOfCores.Value is uint)
                        {
                            coreCountTotal += (uint)numberOfCores.Value;
                        }
                        else
                        {
                            coreCountTotal++;
                        }
                    }
                }
            }
            catch
            {
            }

            SoundProjectConverter.processorCoreCount = (coreCountTotal == 0) ? 1 : coreCountTotal;

            return SoundProjectConverter.processorCoreCount;
        }

        private string CreateOutputList(IEnumerable<string> filePaths)
        {
            Ensure.Operation.ObjectNotNull(this.context);
            Ensure.Operation.ObjectNotNull(this.context.SoundArchiveOutput);
            Ensure.Operation.ObjectNotNull(this.context.SoundArchiveMapOutput);
            Ensure.Operation.True(!string.IsNullOrEmpty(this.context.SoundArchiveOutput.Path));
            Ensure.Operation.True(!string.IsNullOrEmpty(this.context.SoundArchiveMapOutput.Path));

            string outputListFilePath = Path.GetTempFileName();

            using (StreamWriter writer = new StreamWriter(outputListFilePath, false, Encoding.UTF8))
            {
                writer.WriteLine(this.context.SoundArchiveOutput.Path);
                writer.WriteLine(this.context.SoundArchiveMapOutput.Path);

                Func<ComponentFile, bool> filter = file =>
                {
                    if (file.IsExternal)
                    {
                        return true;
                    }

                    if (file.Components.Count != 1)
                    {
                        return false;
                    }

                    var group = file.Components[0] as GroupBase;

                    if (group == null)
                    {
                        return false;
                    }

                    return group.OutputType == GroupOutputType.UserManagement;
                };

                foreach (var filePath in this.context.Files.
                    Where(file => filter(file)).
                    Select(file => file.OutputTarget.ItemDictionary[string.Empty].Path))
                {
                    writer.WriteLine(filePath);
                }

                foreach (var filePath in filePaths)
                {
                    writer.WriteLine(filePath);
                }
            }

            return outputListFilePath;
        }

        private string GetSoundArchiveName()
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            return this.addonSoundSet == null
                ? this.projectService.Project.Name
                : this.addonSoundSet.Name;
        }

        private void OnLineAdded(object sender, OutputLineEventArgs e)
        {
            Assertion.Argument.NotNull(e);
            this.OnLineOutput(e);
        }

        private void OnProcessCompleted(object sender, EventArgs e)
        {
            Assertion.Argument.NotNull(e);

            try
            {
                if (this.IsSucceeded && this.canPostRun)
                {
                    this.PostRun(this.context);
                }
            }
            catch (Exception exception)
            {
                this.logger.AddLine(
                    new ErrorLine(exception.Message)
                    );
            }

            try
            {
                this.stopWatch.Stop();

                if (this.IsSucceeded)
                {
                    this.SuccessExit(this.context.Project);
                }
                else if (this.IsCanceled)
                {
                    this.CancelExit(this.context.Project);
                }
                else
                {
                    this.ErrorExit(this.context.Project);
                }
            }
            finally
            {
                this.OnConversionCompleted(EventArgs.Empty);
            }
        }
    }
}
