﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements;
    using NintendoWare.ToolDevelopmentKit;
    using System.IO;
    using System.Linq;

    /// <summary>
    /// サウンドID C/C++ヘッダを出力します。
    /// </summary>
    public class SoundIDCppHeaderExporter : ISoundIDCppHeaderExporter
    {
        private const uint InvalidFileID = 0xFFFFFFFF;

        /// <summary>
        /// サウンドID C/C++ヘッダを出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="projectService">プロジェクトサービスを指定します。</param>
        public void Export(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Ensure.Argument.NotNull(writer);
            Ensure.Argument.NotNull(soundArchiveBinaryMapXml);

            this.ExportSoundIDs(writer, soundArchiveBinaryMapXml);
            writer.WriteLine();

            this.ExportSoundGroupIDs(writer, soundArchiveBinaryMapXml);
            writer.WriteLine();

            this.ExportSoundSetBankIDs(writer, soundArchiveBinaryMapXml);
            writer.WriteLine();

            this.ExportPlayerIDs(writer, soundArchiveBinaryMapXml);
            writer.WriteLine();

            this.ExportWaveArchiveIDs(writer, soundArchiveBinaryMapXml);
            writer.WriteLine();

            this.ExportGroupIDs(writer, soundArchiveBinaryMapXml);
        }

        /// <summary>
        /// サウンドIDを出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportSoundIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// SoundIDs");

            foreach (var xml in soundArchiveBinaryMapXml.Sounds)
            {
                this.ExportItemIDDefinition(writer, xml);
            }
        }

        /// <summary>
        /// サウンドグループIDを出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportSoundGroupIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// SoundGroupIDs");

            ExportSoundGroupWaveSoundSetIDs(writer, soundArchiveBinaryMapXml);
            ExportSoundGroupSequenceSoundSetIDs(writer, soundArchiveBinaryMapXml);
        }

        /// <summary>
        /// サウンドグループID(WaveSoundSets)を出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportSoundGroupWaveSoundSetIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// WaveSoundSetIDs");

            foreach (var xml in soundArchiveBinaryMapXml.SoundGroups.OfType<WaveSoundGroupXml>())
            {
                this.ExportItemIDDefinition(writer, xml);
            }
        }

        /// <summary>
        /// サウンドグループID(SequenceSoundSets)を出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportSoundGroupSequenceSoundSetIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// SequenceSoundSetIDs");

            foreach (var xml in soundArchiveBinaryMapXml.SoundGroups.OfType<SequenceSoundGroupXml>())
            {
                this.ExportItemIDDefinition(writer, xml);
            }
        }

        /// <summary>
        /// サウンドセットバンクIDを出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportSoundSetBankIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// BankIDs");

            foreach (var xml in soundArchiveBinaryMapXml.SoundSetBanks
                .Where(soundSetBank => soundSetBank.FileID != InvalidFileID))
            {
                this.ExportItemIDDefinition(writer, xml);
            }
        }

        /// <summary>
        /// プレイヤーIDを出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportPlayerIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// PlayerIDs");

            // 追加サウンドアーカイブの場合は、メインサウンドアーカイブのプレイヤー情報が含まれるが、
            // ローカルなプレイヤー ID でありユーザーに積極的に公開しないため、xsid には出力しない
            if (!soundArchiveBinaryMapXml.IsAddonSoundArchive)
            {
                foreach (var xml in soundArchiveBinaryMapXml.Players)
                {
                    this.ExportItemIDDefinition(writer, xml);
                }
            }
        }

        /// <summary>
        /// 波形アーカイブIDを出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportWaveArchiveIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// WaveArchiveIDs");

            // 自動生成アイテムを除外するために "@" を含むものは対象外とする
            foreach (var xml in soundArchiveBinaryMapXml.WaveArchives.Where(xml => !xml.Name.Contains("@")))
            {
                this.ExportItemIDDefinition(writer, xml);
            }
        }

        /// <summary>
        /// グループIDを出力します。
        /// </summary>
        /// <param name="writer">出力するテキストライタを指定します。</param>
        /// <param name="soundArchiveBinaryMapXml">サウンドアーカイブバイナリマップ XML ルート要素を指定します。</param>
        public void ExportGroupIDs(TextWriter writer, SoundArchiveBinaryXml soundArchiveBinaryMapXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundArchiveBinaryMapXml);

            writer.WriteLine("// GroupIDs");

            foreach (var xml in soundArchiveBinaryMapXml.Groups)
            {
                this.ExportItemIDDefinition(writer, xml);
            }
        }

        private void ExportItemIDDefinition(TextWriter writer, SoundSetItemXml soundSetItemXml)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundSetItemXml);

            writer.WriteLine(string.Format(
                "static const unsigned int {0} = 0x{1:X8};",
                soundSetItemXml.Name,
                soundSetItemXml.ID.Value
                ));
        }
    }
}
