﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// リニアPCMにエンコードします。
    /// </summary>
    internal class NintendoWareLinearPcmEncoder : NintendoWareWaveEncoder
    {
        private SourceStreamReader reader;
        private List<SampleGrabber> sampleGrabbers = new List<SampleGrabber>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public NintendoWareLinearPcmEncoder()
        {
            this.BitsPerSample = 16;
            this.IsLittleEndian = true;
        }

        /// <summary>
        /// 出力のビット深度を取得または設定します。
        /// </summary>
        public int BitsPerSample { get; set; }

        /// <summary>
        /// リトルエンディアンで出力するかどうかを取得または設定します。
        /// </summary>
        public bool IsLittleEndian { get; set; }

        /// <summary>
        /// 波形データをエンコードします。
        /// </summary>
        /// <param name="waveStreams">波形ストリームの配列を指定します。</param>
        /// <returns>出力の配列を返します。</returns>
        public override CodecOutput[] Run(WaveStream[] waveStreams)
        {
            Ensure.Argument.NotNull(waveStreams);
            Ensure.Argument.True(waveStreams.Length == 1);

            this.Initialize(waveStreams[0]);

            this.reader.Run();

            List<CodecOutput> codecOutputs = new List<CodecOutput>();

            foreach (SampleGrabber sampleGrabber in this.sampleGrabbers)
            {
                codecOutputs.Add(
                    new CodecOutput()
                    {
                        Data = sampleGrabber.Samples ?? new byte[0],
                        Format = sampleGrabber.Inputs[0].StreamInformation.Format,
                        FrameCount = sampleGrabber.Inputs[0].StreamInformation.FrameCount,
                    });
            }

            return codecOutputs.ToArray();
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="waveStream">波形ストリームを指定します。</param>
        private void Initialize(WaveStream waveStream)
        {
            Assertion.Argument.NotNull(waveStream);

            this.sampleGrabbers.Clear();

            this.reader = new SourceStreamReader()
            {
                Source = waveStream.Payload,
                SourceFormat = waveStream.Format,
            };

            LinearPcmNormalizer normalizer = new LinearPcmNormalizer();
            this.reader.Outputs[0].ConnectTo(normalizer.Inputs[0]);

            ChannelDivider channelDivider = new ChannelDivider();
            normalizer.Outputs[0].ConnectTo(channelDivider.Inputs[0]);

            for (int i = 0; i < channelDivider.Outputs.Length; i++)
            {
                LoopedSampleAligner loopedSampleAligner = new LoopedSampleAligner();
                channelDivider.Outputs[i].ConnectTo(loopedSampleAligner.Inputs[0]);

                LinearPcmEncoder encoder = new LinearPcmEncoder()
                {
                    BitsPerSample = this.BitsPerSample,
                    IsLittleEndian = this.IsLittleEndian,
                };
                loopedSampleAligner.Outputs[0].ConnectTo(encoder.Inputs[0]);

                SampleGrabber sampleGrabber = new SampleGrabber();
                encoder.Outputs[0].ConnectTo(sampleGrabber.Inputs[0]);

                this.sampleGrabbers.Add(sampleGrabber);
            }
        }
    }
}
