﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core.Parameters;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TreeCtrlCore : OperatableControl, IInplaceOwner
    {
        private ExchangeOperator _ExchangeOperator = null;
        private SelectOperator _SelectOperator = null;
        private MoveOperator _MoveOperator = null;

        private ITreeItem _CaretItem = null;

        private bool _EnabledRubberBand = false;
        private bool _RubberBandToggle = false;
        private Point _RubberBandBeginPoint = new Point();
        private Point _RubberBandEndPoint = new Point();
        private bool _EnabledDragMarker = false;
        private ITreeItem _DragItem = null;
        private TreeItemInPosition _DragItemInPosition = TreeItemInPosition.None;

        private InplaceEditor _ActiveEditor = null;
        private ITreeItem _EditingItem = null;

#if DEBUG
        private int _DrawCounter = 0;
#endif

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeCtrlCore()
        {
            _ExchangeOperator = new ExchangeOperator(this);
            _SelectOperator = new SelectOperator(this);
            _MoveOperator = new MoveOperator(this);
            CurrentOperator = _ExchangeOperator;

            HorizontalScrollBarSmallChange = 1;
            VerticalScrollBarSmallChange = 1;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TreeCtrl OwnerCtrl { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateInfos()
        {
            OwnerCtrl.UpdateInfos();

            //
            if (GetItemInfo(_CaretItem) == null)
            {
                _CaretItem = null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ReadOnly
        {
            get { return OwnerCtrl.ReadOnly; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool AllowDragItem
        {
            get { return OwnerCtrl.AllowDragItem; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ClickedOutsideUnselectItem
        {
            get { return OwnerCtrl.ClickedOutsideUnselectItem; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int IndentLength
        {
            get { return OwnerCtrl.IndentLength; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool MultiSelect
        {
            get { return OwnerCtrl.MultiSelect; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool FullRowSelect
        {
            get { return OwnerCtrl.FullRowSelect; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool FullRowDoubleClickedCaption
        {
            get { return OwnerCtrl.FullRowDoubleClickedCaption; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool FollowSelectedItemToCaret
        {
            get { return OwnerCtrl.FollowSelectedItemToCaret; }
        }

        /// <summary>
        /// 編集中なのか調べる
        /// </summary>
        public bool IsEditing
        {
            get
            {
                return _ActiveEditor != null ? true : false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// スクロールバーの更新
        /// </summary>
        public override void UpdateScrollBar()
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                HorizontalScrollBar.Visible = false;
                VerticalScrollBar.Visible = false;
                return;
            }

            //
            bool redraw = false;
            int itemCount = 0;
            int fullVisibleItemCount = 0;
            int effectiveItemWidth = HorizontalWidth;
            int visibleItemWidth = ClientRectangleWithoutScrollBar.Width;

            itemCount = infos.Length;
            fullVisibleItemCount = GetFullyVisibleItemCount(infos, VerticalScrollBar.Value);

            if (visibleItemWidth < 0) { visibleItemWidth = 0; }

            HorizontalScrollBar.Visible =
                visibleItemWidth >= effectiveItemWidth ? false : true;
            VerticalScrollBar.Visible =
                fullVisibleItemCount >= itemCount ? false : true;

            //
            base.UpdateScrollBar();

            //
            if (AdjustHorizontalScrollBarValue() != false)
            {
                redraw = true;
            }

            if (AdjustVerticalScrollBarValue() != false)
            {
                fullVisibleItemCount = GetFullyVisibleItemCount(infos, VerticalScrollBar.Value);
                redraw = true;
            }

            //
            HorizontalScrollBar.Minimum = 0;
            HorizontalScrollBar.Maximum = effectiveItemWidth;
            HorizontalScrollBar.SmallChange = HorizontalScrollBarSmallChange;
            HorizontalScrollBar.LargeChange = visibleItemWidth;

            VerticalScrollBar.Minimum = 0;
            VerticalScrollBar.Maximum = itemCount == 0 ? 0 : itemCount - 1;
            VerticalScrollBar.SmallChange = VerticalScrollBarSmallChange;
            VerticalScrollBar.LargeChange = fullVisibleItemCount;

            //
            if (redraw != false)
            {
                Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        /// ドラック中、スクロールバーの矢印を押した時の水平方向のスクロールの量の
        /// 取得/設定
        /// </summary>
        public int HorizontalScrollBarSmallChange { get; set; }

        ///--------------------------------
        /// <summary>
        /// ドラック中、スクロールバーの矢印を押した時の垂直方向のスクロールの量の
        /// 取得/設定
        /// </summary>
        public int VerticalScrollBarSmallChange { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool BeginEditBySelectedItem()
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            ITreeItem[] items = null;

            items = selectedItems.GetAll();
            if (items.Length != 1)
            {
                return false;
            }

            return BeginEdit(items[0]);
        }

        /// <summary>
        /// 編集の終了
        /// </summary>
        public void EndEdit()
        {
            if (_ActiveEditor == null)
            {
                return;
            }

            Debug.WriteLine("EndEdit");

            InplaceEditor editor = null;

            editor = _ActiveEditor;
            _ActiveEditor = null;

            editor.Dispose();
            editor = null;

            OnEditEnded();
        }

        ///--------------------------------
        /// <summary>
        /// キャレットの移動
        /// </summary>
        public bool MoveCaret(CaretMove direction)
        {
            switch (direction)
            {
                case CaretMove.Left:
                case CaretMove.Up:
                    return MoveCaretPreviousItem();

                case CaretMove.Right:
                case CaretMove.Down:
                    return MoveCaretNextItem();
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool MoveCaret(ITreeItem item)
        {
            if (item == null)
            {
                return false;
            }

            _CaretItem = item;

            AdjustVerticalScrollBarByItem(_CaretItem);
            OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem));
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ITreeItemsSource ItemsSource
        {
            get
            {
                if (OwnerCtrl != null)
                {
                    return OwnerCtrl.ItemsSource;
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ITreeItemInfo[] ItemInfos
        {
            get
            {
                if (OwnerCtrl != null)
                {
                    if (OwnerCtrl.ItemsSource != null)
                    {
                        return OwnerCtrl.ItemInfos;
                    }
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DemandInvalidate()
        {
            OwnerCtrl.DemandInvalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanDropFiles(string[] filePaths, object dropTarget)
        {
            return OwnerCtrl.CanDropFiles(filePaths, (ITreeItem)dropTarget);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool DropFiles(string[] filePaths, object dropTarget)
        {
            return OwnerCtrl.DropFiles(filePaths, (ITreeItem)dropTarget);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected TreeItemSelectedDictionary GetItemSelecteds()
        {
            return OwnerCtrl.GetItemSelecteds(this);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ITreeItem[] GetSelectedItems()
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            return selectedItems.GetAll();
        }

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToExchange(Point beginPoint)
        {
            Debug.WriteLine("Operator changed to exchange");
            CurrentOperator = _ExchangeOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToSelect(Point beginPoint)
        {
            Debug.WriteLine("Operator changed to Select");
            CurrentOperator = _SelectOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToMove(Point beginPoint)
        {
            Debug.WriteLine("Operator changed to Move");
            CurrentOperator = _MoveOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle ToClientRectangle(int x, int y, int width, int height)
        {
            return new Rectangle(x - ScrollLogicalX, y - ScrollLogicalY, width, height);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle ToClientRectangle(Rectangle rect)
        {
            rect.X -= ScrollLogicalX;
            rect.Y -= ScrollLogicalY;
            return rect;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToLogicalX(int cx)
        {
            return cx + ScrollLogicalX;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToLogicalY(int cy)
        {
            return cy + ScrollLogicalY;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Point ToLogicalPoint(Point point)
        {
            point.X += ScrollLogicalX;
            point.Y += ScrollLogicalY;
            return point;
        }

        ///--------------------------------
        /// <summary>
        /// 2点の座標から正規化された矩形を作成
        /// </summary>
        protected Rectangle NormalizedRectangle(Point begin, Point end)
        {
            Point lp = new Point();
            Point rp = new Point();

            lp.X = begin.X <= end.X ? begin.X : end.X;
            lp.Y = begin.Y <= end.Y ? begin.Y : end.Y;

            rp.X = begin.X >= end.X ? begin.X : end.X;
            rp.Y = begin.Y >= end.Y ? begin.Y : end.Y;

            return new Rectangle(lp.X, lp.Y, rp.X - lp.X, rp.Y - lp.Y);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ScrollLogicalX
        {
            get { return HorizontalScrollBar.Value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ScrollLogicalY
        {
            get
            {
                ITreeItemInfo[] infos = ItemInfos;
                return infos == null ? 0 : infos[VerticalScrollBar.Value].Y;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int HorizontalWidth
        {
            get
            {
                TreeItemDrawDescriptor itemDrawDesc = null;
                ITreePartDrawer partDrawer = null;
                ITreeSubItem subItem = null;
                ITreeItem item = null;
                Graphics gc = null;
                Rectangle bounds = new Rectangle();
                int indentLength = OwnerCtrl.IndentLength;
                int clientWidth = ClientRectangleWithoutScrollBar.Width;
                int maxWidth = 0;
                int width = 0;
                int bx = 0;

                try
                {
                    gc = CreateGraphics();
                    itemDrawDesc = new TreeItemDrawDescriptor(gc, Font, OwnerCtrl.Images);


                    foreach (ITreeItemInfo info in ItemInfos)
                    {
                        item = info.Item;
                        subItem = OwnerCtrl.SubItems[item.SubItemName];

                        bx = indentLength * item.Depth;

                        bounds.X = bx;
                        bounds.Y = 0;
                        bounds.Width = clientWidth;
                        bounds.Height = item.Length;

                        item.GetConstValues(itemDrawDesc.Parameters);
                        itemDrawDesc.Bounds = bounds;

                        foreach (ITreePart part in subItem.GetParts(itemDrawDesc))
                        {
                            if ((partDrawer = OwnerCtrl.PartDrawers[part.Name]) != null)
                            {
                                width = bx + part.Bounds.X + part.Bounds.Width;
                                if (width > maxWidth)
                                {
                                    maxWidth = width;
                                }
                            }
                        }
                    }
                }

                finally { gc.Dispose(); }
                return maxWidth;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 垂直方向のスクロールしてデータを表示する領域の幅の取得
        /// </summary>
        protected virtual int VisibleClientHeight
        {
            get { return ClientRectangleWithoutScrollBar.Height; }
        }

        ///--------------------------------
        /// <summary>
        /// 横スクロールの位置を補正する
        /// 補正された場合には trueを返す
        /// </summary>
        protected bool AdjustHorizontalScrollBarValue()
        {
            int value = HorizontalScrollBar.Value;
            int min = HorizontalScrollBar.Minimum;
            int max = HorizontalScrollBar.Maximum;
            int clientWidth = ClientRectangleWithoutScrollBar.Width;
            int logicalWidth = max - min;
            int offset = 0;

            if (value + clientWidth > max)
            {
                offset = (value + clientWidth) - max;
                value -= offset;
            }

            if (HorizontalScrollBar.Value == value)
            {
                return false;
            }

            if (value < min || value > max)
            {
                value = 0;
            }

            HorizontalScrollBar.Value = value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールの位置を補正する
        /// 補正された場合には trueを返す
        /// </summary>
        protected bool AdjustVerticalScrollBarValue()
        {
            int value = GetAdjustVerticalScrollBarValue();

            if (VerticalScrollBar.Value == value)
            {
                return false;
            }

            VerticalScrollBar.Value = value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int GetAdjustVerticalScrollBarValue()
        {

            ITreeItemInfo[] infos = ItemInfos;
            int index = VerticalScrollBar.Value;
            int min = VerticalScrollBar.Minimum;
            int max = infos.Length;
            int height = VisibleClientHeight;
            int h = 0;

            while (index > min)
            {
                h = 0;
                for (int loop = index - 1; loop < max; loop++)
                {
                    h += infos[loop].Item.Length;
                    if (h > height)
                    {
                        return index;
                    }
                }
                index--;
            }
            return index;
        }

        ///--------------------------------
        /// <summary>
        /// 指定ID番号のアイテムが見えるように垂直スクロールバーの位置の値を調整する
        /// </summary>
        protected void AdjustVerticalScrollBarByItem(ITreeItem item)
        {
            ITreeItemInfo[] infos = ItemInfos;
            int position = VerticalScrollBar.Value;
            int min = 0;
            int max = 0;
            int index = -1;

            if (infos == null)
            {
                return;
            }

            min = position;
            max = position + GetFullyVisibleItemCount(infos, position);

            if ((index = GetIndexOnItem(item)) < 0)
            {
                return;
            }

            if (index == min - 1)
            {
                ScrollVerticalScrollBarRelative(-1);
                return;
            }

            if (index == max)
            {
                if (infos != null)
                {
                    if (index == infos.Length - 1)
                    {
                        ScrollVerticalScrollBarRelative(1);
                        return;
                    }
                }

                ScrollVerticalScrollBarRelative(1);
                return;
            }

            if (index < min - 1 || index > max)
            {
                ScrollVerticalScrollBarAbsolute(index);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int GetVisibleItemCount(ITreeItemInfo[] infos, int beginIndex)
        {
            int height = ClientRectangleWithoutScrollBar.Height;
            int count = 0;

            for (int index = beginIndex; index < infos.Length; index++)
            {
                count++;

                height -= infos[index].Item.Length;
                if (height <= 0)
                {
                    break;
                }
            }
            return count;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int GetFullyVisibleItemCount(ITreeItemInfo[] infos, int beginIndex)
        {
            int height = ClientRectangleWithoutScrollBar.Height;
            int count = 0;

            for (int index = beginIndex; index < infos.Length; index++)
            {
                if (height - infos[index].Item.Length < 0)
                {
                    break;
                }

                count++;

                height -= infos[index].Item.Length;
                if (height <= 0)
                {
                    break;
                }
            }
            return count;
        }

        /// <summary>
        ///
        /// </summary>
        protected bool IsSelected(ITreeItem item)
        {
            return GetItemSelecteds().IsSelected(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ToggleExpandItem(ITreeItem item)
        {
            item.Expanded = !item.Expanded;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void UnselectedItem()
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.Clear();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SelectedItem(ITreeItem item, bool b)
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.Selected(item, b);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ToggleSelectedItem(ITreeItem item)
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.ToggleSelected(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SelectedCardinalTo(ITreeItem item)
        {
            // 未実装、現在は指定アイテムを選択するだけになります。
            GetItemSelecteds().Selected(item, true);
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの選択
        /// 論理矩形の範囲内が対象になる
        /// </summary>
        protected ITreeItem[] SelectedItemByLogicalRectangle(Rectangle rect)
        {
            ITreeItem[] items = GetIntersectItems(rect);

            foreach (ITreeItem item in items)
            {
                SelectedItem(item, true);
            }
            return items;
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの選択のトグル
        /// 論理矩形の範囲内が対象になる
        /// </summary>
        protected ITreeItem[] ToggleSelectedItemByLogicalRectangle(Rectangle rect)
        {
            ITreeItem[] items = GetIntersectItems(rect);

            foreach (ITreeItem item in items)
            {
                ToggleSelectedItem(item);
            }
            return items;
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの選択のトグル
        /// </summary>
        protected ITreeItem ToggleSelectedItemByLogicalPoint(Point point)
        {
            Rectangle bounds = new Rectangle(point.X, point.Y, 1, 1);
            ITreeItem[] items = ToggleSelectedItemByLogicalRectangle(bounds);
            return items.Length > 0 ? items[0] : null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ITreeItem CaretItem
        {
            get { return _CaretItem; }
        }

        /// <summary>
        ///
        /// </summary>
        protected bool Caret(ITreeItem item)
        {
            _CaretItem = item;
            OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem));
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected bool CaretWithoutFollowSelectedItem(ITreeItem item)
        {
            _CaretItem = item;
            OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem, true, false));
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaret(int index)
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (index < 0 || index >= infos.Length)
            {
                return false;
            }

            _CaretItem = infos[index].Item;

            AdjustVerticalScrollBarByItem(_CaretItem);
            OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem));
            return true;
        }

        /// <summary>
        /// キャレットが移動した時に呼ばれます。
        /// </summary>
        protected virtual void OnCaretMoved(TreeCaretMovedEventArgs e)
        {
            //移動が失敗したのか？
            if (e.Moved == false)
            {
                return;
            }

            ///選択がカレットに追従するのか？
            if (e.FollowSelectedItemToCaret != false &&
                OwnerCtrl.FollowSelectedItemToCaret != false)
            {
                UnselectedItem();
                SelectedItem(e.Item, true);
                OnSelectChanged();
                OwnerCtrl.DemandInvalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CollisionToRubberBand(Rectangle bounds)
        {
            if (_EnabledRubberBand == false)
            {
                return false;
            }

            Rectangle rubberBandBounds = GetNormalizedRubberBand();
            return IntersectBounds(rubberBandBounds, bounds, FullRowSelect);
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected TreeCollisionResult CollisionToItem(TreeItemDrawDescriptor desc, ITreeItemInfo info, int lx, int ly)
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            ITreeSubItem subItem = null;
            ITreeItem item = info.Item;
            Rectangle bounds;
            int indentLength = OwnerCtrl.IndentLength;
            int height = item.Length;
            int bx = indentLength * item.Depth;
            int by = info.Y;

            //
            if (item.Expander != false)
            {
                bounds = OwnerCtrl.Surveyer.GetExpanderBounds(indentLength, height);
                bounds.X += bx - indentLength;
                bounds.Y += by;

                if (bounds.Contains(lx, ly) != false)
                {
                    return new TreeCollisionResult(info, "Expander");
                }
            }

            //
            if ((subItem = OwnerCtrl.SubItems[item.SubItemName]) != null)
            {
                foreach (ITreePart part in subItem.GetParts(desc))
                {
                    bounds = part.Bounds;
                    bounds.X += bx;
                    bounds.Y += by;

                    if (bounds.Contains(lx, ly) != false)
                    {
                        bool selected = selectedItems.IsSelected(item);
                        return new TreeCollisionResult(info, part.Name, selected);
                    }
                }
            }

            return new TreeCollisionResult(info, "Row");
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected TreeCollisionResult Collision(int cx, int cy)
        {
            Graphics gc = null;
            TreeCollisionResult result = null;
            TreeItemDrawDescriptor desc = null;
            ITreeItemInfo[] infos = GetVisibleItemInfos();
            ITreeItem item = null;
            int lx = ToLogicalX(cx);
            int ly = ToLogicalY(cy);

            try
            {
                gc = CreateGraphics();
                desc = new TreeItemDrawDescriptor(gc, Font, OwnerCtrl.Images);

                foreach (ITreeItemInfo info in infos)
                {
                    item = info.Item;

                    if (ly >= info.Y && ly < info.Y + item.Length)
                    {
                        item.GetConstValues(desc.Parameters);
                        if ((result = CollisionToItem(desc, info, lx, ly)) != null)
                        {
                            return result;
                        }
                    }
                }
            }

            finally { gc.Dispose(); }
            return new TreeCollisionResult();
        }

        ///--------------------------------
        /// <summary>
        /// ドラック処理用の衝突判定
        /// 座標はクライアント座標を指定する
        /// </summary>
        protected TreeCollisionResult CollisionForDrag(int cx, int cy)
        {
            ITreeItemInfo[] infos = GetVisibleItemInfos();
            ITreeItem item = null;
            TreeItemInPosition inPosition = TreeItemInPosition.None;
            int lx = ToLogicalX(cx);
            int ly = ToLogicalY(cy);
            int cenY = 0;
            int uppY = 0;
            int lowY = 0;
            int y = 0;
            int height = 0;

            foreach (ITreeItemInfo info in infos)
            {

                item = info.Item;
                y = info.Y;
                height = item.Length;

                if (ly >= y && ly < y + height)
                {
                    cenY = y + (height / 2);
                    uppY = y + (height / 3);
                    lowY = y + (height / 3) * 2;

                    if (item.Container != false)
                    {
                        inPosition = TreeItemInPosition.Center;
                        if (ly < uppY) { inPosition = TreeItemInPosition.Upper; }
                        if (ly >= lowY) { inPosition = TreeItemInPosition.Lower; }
                        return new TreeCollisionResult(info, "Row", inPosition);
                    }
                    else
                    {

                        inPosition = TreeItemInPosition.Upper;
                        if (ly >= cenY) { inPosition = TreeItemInPosition.Lower; }
                        return new TreeCollisionResult(info, "Row", inPosition);
                    }
                }
            }

            return new TreeCollisionResult();
        }

        ///--------------------------------
        /// <summary>
        /// クライアント領域に表示されているアイテムの取得
        /// </summary>
        protected ITreeItemInfo[] GetVisibleItemInfos()
        {
            List<ITreeItemInfo> list = new List<ITreeItemInfo>();
            ITreeItemInfo[] infos = ItemInfos;
            int beginIndex = 0;
            int endIndex = 0;

            if (infos != null)
            {
                beginIndex = VerticalScrollBar.Value;
                endIndex = beginIndex + GetVisibleItemCount(infos, beginIndex) - 1;

                for (int index = beginIndex; index <= endIndex; index++)
                {
                    list.Add(infos[index]);
                }
            }
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool IntersectBounds(Rectangle bounds0, Rectangle bounds1, bool ignoreWidth)
        {
            if (ignoreWidth == false)
            {
                if (bounds0.Right < bounds1.Left ||
                    bounds0.Left > bounds1.Right)
                {
                    return false;
                }
            }

            if (bounds0.Bottom < bounds1.Top ||
                bounds0.Top > bounds1.Bottom)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 2点で指定される矩形範囲に掛かるアイテムの取得
        /// </summary>
        protected ITreeItem[] GetIntersectItems(Rectangle selectRect)
        {
            List<ITreeItem> list = new List<ITreeItem>();
            ITreeItemInfo[] infos = ItemInfos;
            ITreeItem item = null;
            ITreeSubItem subItem = null;
            int indentLength = OwnerCtrl.IndentLength;
            int x = 0;
            int y = 0;
            int height = 0;
            Graphics gc = null;
            Rectangle rect = new Rectangle();

            if (infos == null)
            {
                return null;
            }

            //
            try
            {
                gc = CreateGraphics();

                foreach (ITreeItemInfo info in infos)
                {
                    item = info.Item;

                    x = item.Depth * indentLength;
                    y = info.Y;
                    height = item.Length;

                    //座標が選択範囲の始点座標より小さいのか？
                    //その場合には衝突判定をパスする
                    if (y + height < selectRect.Y)
                    {
                        continue;
                    }

                    //
                    if (FullRowSelect != false)
                    {
                        rect.X = 0;
                        rect.Y = y;
                        rect.Width = 0;
                        rect.Height = height;
                    }
                    else
                    {

                        if ((subItem = OwnerCtrl.SubItems[item.SubItemName]) == null)
                        {
                            continue;
                        }

                        rect = subItem.BoundsForSelected;
                        rect.X += x;
                        rect.Y += y;
                    }

                    if (IntersectBounds(rect, selectRect, FullRowSelect) != false)
                    {
                        list.Add(item);
                    }

                    //座標が選択範囲の終点座標より大きいのか？
                    //その場合には衝突判定をパスする
                    if (y >= selectRect.Y + selectRect.Height)
                    {
                        break;
                    }
                }

                //
                return list.ToArray();
            }

            catch { }
            finally { gc.Dispose(); }
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// ドラック中の自動スクロールを開始する境界の取得
        /// 引数はクライアント
        /// </summary>
        protected Rectangle DragScrollBorderRectangle
        {
            get
            {
                int wm = 0;
                int hm = 0;

                return new Rectangle(wm, hm, Width - wm * 2, Height - hm * 2);
            }
        }

        ///--------------------------------
        /// <summary>
        /// 横スクロールバーの相対スクロール
        /// </summary>
        protected bool ScrollHorizontalScrollBarRelative(int value)
        {
            int index = HorizontalScrollBar.Value;
            int count = ClientRectangleWithoutScrollBar.Width;
            int min = HorizontalScrollBar.Minimum;
            int max = HorizontalScrollBar.Maximum - count;

            if (ItemInfos == null)
            {
                return false;
            }

            if (max < 0)
            {
                return false;
            }

            if (value < 0)
            {
                if (HorizontalScrollBar.Value + value < min)
                {
                    value = min - HorizontalScrollBar.Value;
                }
            }

            if (value > 0)
            {
                if (HorizontalScrollBar.Value + value >= max)
                {
                    value = max - HorizontalScrollBar.Value + 1;
                }
            }

            if (value == 0)
            {
                return false;
            }

            HorizontalScrollBar.Value += value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールバーの相対スクロール
        /// </summary>
        protected bool ScrollVerticalScrollBarRelative(int value)
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return false;
            }

            int index = VerticalScrollBar.Value;
            int count = GetFullyVisibleItemCount(infos, index);
            int min = VerticalScrollBar.Minimum;
            int max = VerticalScrollBar.Maximum - count + 1;

            if (max < 0)
            {
                return false;
            }

            if (value < 0)
            {
                if (VerticalScrollBar.Value + value < min)
                {
                    value = min - VerticalScrollBar.Value;
                }
            }

            if (value > 0)
            {
                if (VerticalScrollBar.Value + value >= max)
                {
                    value = (max + 0) - VerticalScrollBar.Value;
                }
            }

            if (value == 0)
            {
                return false;
            }

            if (VerticalScrollBar.Value + value > VerticalScrollBar.Maximum)
            {
                VerticalScrollBar.Value = VerticalScrollBar.Maximum;
                return true;
            }

            VerticalScrollBar.Value += value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールバーの相対スクロール(ページスクロール)
        /// </summary>
        protected bool ScrollPageVerticalScrollBarRelative(int value)
        {
            ITreeItemInfo[] infos = ItemInfos;
            int relativeValue = 0;
            int index = VerticalScrollBar.Value;

            if (infos == null || _CaretItem == null || value == 0)
            {
                return false;
            }

            relativeValue = GetFullyVisibleItemCount(infos, index);

            if (value < 0)
            {
                relativeValue = -relativeValue;
            }

            ///
            ITreeItemInfo info = null;
            int offset = 0;
            bool result = false;

            if ((info = GetItemInfo(_CaretItem)) == null)
            {
                return false;
            }

            offset = GetIndexOnItem(info.Item) - index;

            result = ScrollVerticalScrollBarRelative(relativeValue);
            index = VerticalScrollBar.Value + offset;

            if (index < 0) { index = 0; }
            if (index > infos.Length - 1) { index = infos.Length - 1; }

            MoveCaret(index);
            return result;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールバーの絶対スクロール
        /// </summary>
        protected bool ScrollVerticalScrollBarAbsolute(int index)
        {
            ITreeItemInfo[] infos = ItemInfos;
            int min = 0;
            int max = 0;

            if (infos == null)
            {
                return false;
            }

            min = VerticalScrollBar.Minimum;
            max = VerticalScrollBar.Maximum -
                GetFullyVisibleItemCount(infos, VerticalScrollBar.Value) + 1;

            if (index < min) { index = min; }
            if (index >= max) { index = max; }

            VerticalScrollBar.Value = index;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ドラック時のスクロール
        /// </summary>
        protected void DragScroll(int cx, int cy)
        {
            Rectangle rect = DragScrollBorderRectangle;
            bool redraw = false;

            if (cx < rect.Left)
            {
                ScrollHorizontalScrollBarRelative(-HorizontalScrollBarSmallChange);
                redraw = true;
            }

            if (cx > rect.Right)
            {
                ScrollHorizontalScrollBarRelative(HorizontalScrollBarSmallChange);
                redraw = true;
            }

            if (cy < rect.Top)
            {
                ScrollVerticalScrollBarRelative(-VerticalScrollBarSmallChange);
                redraw = true;
            }

            if (cy > rect.Bottom)
            {
                ScrollVerticalScrollBarRelative(VerticalScrollBarSmallChange);
                redraw = true;
            }

            if (redraw != false)
            {
                Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        /// 正規化されたラバーバンドの取得
        /// 論理座標を返す
        /// </summary>
        protected Rectangle GetNormalizedRubberBand()
        {
            return NormalizedRectangle(_RubberBandBeginPoint, _RubberBandEndPoint);
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドの設定
        /// </summary>
        protected void RubberBand(Point beginPoint, Point endPoint, bool toggle)
        {
            _EnabledRubberBand = true;
            _RubberBandToggle = toggle;
            _RubberBandBeginPoint = beginPoint;
            _RubberBandEndPoint = endPoint;
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドの設定
        /// </summary>
        protected void RubberBand(Point point, bool toggle)
        {
            RubberBand(point, point, toggle);
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドの抹消
        /// </summary>
        protected void RubberBand()
        {
            _EnabledRubberBand = false;
        }

        ///--------------------------------
        /// <summary>
        /// ドラック用マーカーの設定
        /// </summary>
        protected void DragMarker(ITreeItem item, TreeItemInPosition inPosition)
        {
            _EnabledDragMarker = true;
            _DragItem = item;
            _DragItemInPosition = inPosition;
        }

        ///--------------------------------
        /// <summary>
        /// ドラック用マーカーの消去
        /// </summary>
        protected void DragMarker()
        {
            _EnabledDragMarker = false;
            _DragItem = null;
            _DragItemInPosition = TreeItemInPosition.None;
        }

        ///--------------------------------
        /// <summary>
        /// 指定ID番号の表示情報の配列のインデックスを取得
        /// </summary>
        protected int GetIndexOnItem(ITreeItem item)
        {
            ITreeItemInfo[] infos = ItemInfos;
            int index = 0;

            if (infos == null || item == null)
            {
                return -1;
            }

            foreach (ITreeItemInfo info in infos)
            {
                if (info.Item == item)
                {
                    return index;
                }
                index++;
            }
            return index;
        }

        ///--------------------------------
        /// <summary>
        /// カレットのあるセルを表示するクライアント矩形の取得
        /// </summary>
        protected virtual Rectangle GetCaretItemClientBounds()
        {
            if (_CaretItem == null)
            {
                return Rectangle.Empty;
            }

            Rectangle bounds = GetPartBounds(_CaretItem, "Caption");
            return ToClientRectangle(bounds);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットのある表示情報の配列のインデックスを取得
        /// </summary>
        protected int IndexOnCaret
        {
            get { return GetIndexOnItem(_CaretItem); }
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを親のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretParentItem()
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (infos == null ||
                _CaretItem == null)
            {
                return false;
            }

            ITreeItem parentItem = _CaretItem.Parent;
            int index = GetIndexOnItem(parentItem);

            if (index < 0)
            {
                OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem, false));
                return false;
            }

            MoveCaret(infos[index].Item);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを最初の子供のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretFirstChildItem()
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return false;
            }

            if (_CaretItem.Children.Count <= 0)
            {
                return false;
            }

            int index = GetIndexOnItem(_CaretItem.Children[0]);

            if (index < 0 || index >= infos.Length)
            {
                OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem, false));
                return false;
            }

            MoveCaret(infos[index].Item);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一つ前のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretPreviousItem()
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return false;
            }

            int index = IndexOnCaret;
            int limit = 0;

            if (index <= limit)
            {
                OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem, false));
                return false;
            }

            MoveCaret(infos[index - 1].Item);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一つ後のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretNextItem()
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return false;
            }

            int index = IndexOnCaret;
            int limit = infos.Length - 1;

            if (index < 0 || index >= limit)
            {
                OnCaretMoved(new TreeCaretMovedEventArgs(_CaretItem, false));
                return false;
            }

            MoveCaret(infos[index + 1].Item);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaretPreviousPageItem()
        {
            if (ScrollPageVerticalScrollBarRelative(-1) == false)
            {
                return MoveCaretMostPreviousItem();
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaretNextPageItem()
        {
            if (ScrollPageVerticalScrollBarRelative(1) == false)
            {
                return MoveCaretMostNextItem();
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一番前のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretMostPreviousItem()
        {
            return MoveCaret(0);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一番後のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretMostNextItem()
        {
            ITreeItemInfo[] infos = ItemInfos;
            if (infos == null) { return false; }
            return MoveCaret(infos.Length - 1);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaretMostPreviousItemInClient()
        {
            ITreeItemInfo info = null;
            int index = VerticalScrollBar.Value;

            if ((info = GetItemInfo(index)) == null)
            {
                return false;
            }

            if (info.Item == _CaretItem)
            {
                return ScrollPageVerticalScrollBarRelative(-1);
            }

            return Caret(info.Item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaretMostNextItemInClient()
        {
            ITreeItemInfo[] infos = ItemInfos;
            ITreeItemInfo info = null;
            int index = VerticalScrollBar.Value;
            int count = 0;

            if (infos == null)
            {
                return false;
            }

            count = GetFullyVisibleItemCount(infos, index) - 1;
            if ((index += count) >= infos.Length)
            {
                index = infos.Length - 1;
            }

            if ((info = GetItemInfo(index)) == null)
            {
                return false;
            }

            if (info.Item == _CaretItem)
            {
                return ScrollPageVerticalScrollBarRelative(1);
            }

            return Caret(info.Item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool HasAncestor(ITreeItem ancestorItem, ITreeItem item)
        {
            while (item != null)
            {
                if (item == ancestorItem)
                {
                    return true;
                }
                item = item.Parent;
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CanMoveSelectedItem(ITreeItem targetItem)
        {
            ITreeItem[] items = null;

            if (ItemsSource == null)
            {
                return false;
            }

            items = GetSelectedItems();
            if (items.Length <= 0)
            {
                return false;
            }

            foreach (ITreeItem item in items)
            {
                if (HasAncestor(item, targetItem) != false)
                {
                    return false;
                }
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CanMoveSelectedItemToChild(ITreeItem targetItem)
        {
            if (CanMoveSelectedItem(targetItem) == false)
            {
                return false;
            }

            return ((ITreeCtrlContainer)OwnerCtrl).CanMoveSelectedItemToChild(targetItem, GetSelectedItems());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CanMoveSelectedItemToPreviousSibling(ITreeItem targetItem)
        {
            if (CanMoveSelectedItem(targetItem) == false)
            {
                return false;
            }

            return ((ITreeCtrlContainer)OwnerCtrl).CanMoveSelectedItemToPreviousSibling(targetItem, GetSelectedItems());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CanMoveSelectedItemToNextSibling(ITreeItem targetItem)
        {
            if (CanMoveSelectedItem(targetItem) == false)
            {
                return false;
            }

            return ((ITreeCtrlContainer)OwnerCtrl).CanMoveSelectedItemToNextSibling(targetItem, GetSelectedItems());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CanExpanded(ITreeItem targetItem)
        {
            return ((ITreeCtrlContainer)OwnerCtrl).CanExpanded(targetItem);
        }

        ///--------------------------------
        /// <summary>
        /// 選択されているアイテムの指定アイテムの子供として移動
        /// </summary>
        protected bool MoveSelectedItemToChild(ITreeItem targetItem)
        {
            //TreeItemSelectedDictionary  selectedItems = GetItemSelecteds();
            //ITreeItem[] items = null;
            //ITreeItem   parentItem = null;

            if (CanMoveSelectedItemToChild(targetItem) == false)
            {
                return false;
            }

            ItemsSource.BeginTransaction();

            //items = selectedItems.Keys .ToArray();
            ITreeItem[] items = GetSelectedItems();
            foreach (ITreeItem item in items)
            {
                ITreeItem parentItem = item.Parent;
                parentItem.Children.Remove(item);
                targetItem.Children.Add(item);
            }

            ((ITreeCtrlContainer)OwnerCtrl).MovedItems(items);
            ItemsSource.EndTransaction();
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 選択されているアイテムの指定アイテムの兄として移動
        /// </summary>
        protected bool MoveSelectedItemToPreviousSibling(ITreeItem targetItem)
        {
            //TreeItemSelectedDictionary  selectedItems = GetItemSelecteds();
            //ITreeItem[] items = null;
            //ITreeItem   targetParentItem = null;
            //ITreeItem   parentItem = null;
            //int         targetItemIndex = -1;

            if (CanMoveSelectedItemToPreviousSibling(targetItem) == false)
            {
                return false;
            }

            //
            if (targetItem.Parent == null)
            {
                return false;
            }
            ITreeItem targetParentItem = targetItem.Parent;
            int targetItemIndex = targetParentItem.Children.IndexOf(targetItem);

            //
            ItemsSource.BeginTransaction();

            //items = selectedItems.Keys .ToArray();
            ITreeItem[] items = GetSelectedItems();
            foreach (ITreeItem item in items)
            {
                ITreeItem parentItem = item.Parent;
                parentItem.Children.Remove(item);

                targetItemIndex = targetParentItem.Children.IndexOf(targetItem);
                targetParentItem.Children.Insert(targetItemIndex, item);
            }

            ((ITreeCtrlContainer)OwnerCtrl).MovedItems(items);
            ItemsSource.EndTransaction();
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 選択されているアイテムの指定アイテムの弟として移動
        /// </summary>
        protected bool MoveSelectedItemToNextSibling(ITreeItem targetItem)
        {
            //TreeItemSelectedDictionary  selectedItems = GetItemSelecteds();
            //ITreeItem[] items = null;
            //ITreeItem   targetParentItem = null;
            //ITreeItem   parentItem = null;
            //int         targetItemIndex = -1;

            if (CanMoveSelectedItemToNextSibling(targetItem) == false)
            {
                return false;
            }

            //
            if (targetItem.Parent == null)
            {
                return false;
            }
            ITreeItem targetParentItem = targetItem.Parent;
            int targetItemIndex = targetParentItem.Children.IndexOf(targetItem);

            //
            ItemsSource.BeginTransaction();

            //items = selectedItems.Keys .ToArray();
            ITreeItem[] items = GetSelectedItems();
            foreach (ITreeItem item in items)
            {
                ITreeItem parentItem = item.Parent;
                parentItem.Children.Remove(item);

                targetItemIndex = targetParentItem.Children.IndexOf(targetItem) + 1;
                targetParentItem.Children.Insert(targetItemIndex, item);
            }

            ((ITreeCtrlContainer)OwnerCtrl).MovedItems(items);
            ItemsSource.EndTransaction();
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DrawCaretItem()
        {
            DrawItem(_CaretItem);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DrawItem(ITreeItem item)
        {
            if (item == null)
            {
                return;
            }

            Graphics gc = null;
            TreeDrawDescriptor drawDesc = null;
            TreeItemDrawDescriptor itemDrawDesc = null;
            ITreeItemInfo info = null;
            int width = ClientRectangleWithoutScrollBar.Width;

            try
            {
                gc = CreateGraphics();

                drawDesc = new TreeDrawDescriptor(gc);
                drawDesc.ClientWidth = width;

                itemDrawDesc = new TreeItemDrawDescriptor(gc, Font, OwnerCtrl.Images);
                itemDrawDesc.Focused = Focused;
                itemDrawDesc.Counter = AnimationCounter;

                if ((info = GetItemInfo(item)) == null)
                {
                    return;
                }

                DrawItem(gc, info, drawDesc, itemDrawDesc);
            }

            finally { gc.Dispose(); }
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの描画
        /// </summary>
        protected void DrawItem(Graphics gc, ITreeItemInfo info, TreeDrawDescriptor drawDesc, TreeItemDrawDescriptor itemDrawDesc)
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            ITreePartDrawer partDrawer = null;
            int indentLength = OwnerCtrl.IndentLength;
            int width = ClientRectangleWithoutScrollBar.Width;
            int height = 0;
            ITreeItem item = null;
            ITreeSubItem subItem = null;
            TreeItemInPosition inPosition = TreeItemInPosition.None;
            Rectangle rect = new Rectangle();
            int bx = 0;
            int by = 0;

            item = info.Item;
            bx = indentLength * item.Depth;
            by = info.Y;
            height = item.Length;

            //
            if ((subItem = OwnerCtrl.SubItems[item.SubItemName]) == null)
            {
                return;
            }

            //
            rect.X = bx;
            rect.Y = by;
            rect.Width = width;
            rect.Height = height;
            rect = ToClientRectangle(rect);

            if (rect.X < 0)
            {
                //rect.Rightが常に ClientRectangle.Rightになるようにする
                //(クライアント領域、一杯に描画する為)
                rect.Width += -rect.X;
            }

            item.GetConstValues(itemDrawDesc.Parameters);
            itemDrawDesc.Bounds = rect;
            itemDrawDesc.Caret = item == _CaretItem ? true : false;
            itemDrawDesc.Enabled = item.Enabled;
            itemDrawDesc.Selected = selectedItems.IsSelected(item);
            itemDrawDesc.TemporarySelected = false;
            itemDrawDesc.TreeItem = item;

            //ラバーバンドとの衝突判定
            if (_EnabledRubberBand != false)
            {
                if (FullRowSelect != false)
                {
                    rect.X = bx;
                    rect.Y = by;
                    rect.Width = 0;
                    rect.Height = height;
                }
                else
                {

                    rect = subItem.BoundsForSelected;
                    rect.X += bx;
                    rect.Y += by;
                }

                itemDrawDesc.TemporarySelected = CollisionToRubberBand(rect);
                if (_RubberBandToggle != false &&
                    itemDrawDesc.TemporarySelected != false)
                {
                    itemDrawDesc.TemporarySelected = !itemDrawDesc.Selected;
                    itemDrawDesc.Selected = false;
                }
            }

            //アイテムの描画
            OwnerCtrl.ItemDrawer.Draw(itemDrawDesc);

            //サブアイテムの描画
            foreach (ITreePart part in subItem.GetParts(itemDrawDesc))
            {
                if ((partDrawer = OwnerCtrl.PartDrawers[part.Name]) != null)
                {
                    rect = part.Bounds;
                    rect.X += bx;
                    rect.Y += by;
                    rect = ToClientRectangle(rect);

                    itemDrawDesc.Bounds = rect;
                    partDrawer.Draw(itemDrawDesc);
                }
            }

            //ドラックマーカーの描画
            rect.X = 0;
            rect.Y = info.Y;
            rect.Width = indentLength * item.Depth;
            rect.Height = height;
            rect = ToClientRectangle(rect);

            drawDesc.Rectangle = rect;
            drawDesc.IndentLength = indentLength;
            //drawer.HorizontalLineLength = 0;      //drawerDescではないのか？
            //drawer.VerticalLineLength = 0;

            inPosition = TreeItemInPosition.None;
            if (_EnabledDragMarker != false)
            {
                if (_DragItem == item)
                {
                    inPosition = _DragItemInPosition;
                }
            }

            //関係線、エキスパンダの描画
            drawDesc.DragItemInPosition = inPosition;
            drawDesc.DivergePoint = OwnerCtrl.Surveyer.GetDivergePoint(indentLength, height);
            drawDesc.ExpanderBounds = OwnerCtrl.Surveyer.GetExpanderBounds(indentLength, height);
            OwnerCtrl.Drawer.Draw(drawDesc, item);
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected override void Draw(Graphics gc)
        {
#if DEBUG
            //Debug.WriteLine( "-------- Draw " + _DrawCounter.ToString());
            _DrawCounter++;
#endif

            TreeDrawDescriptor drawDesc = null;
            TreeItemDrawDescriptor itemDrawDesc = null;
            ITreeItemInfo[] infos = GetVisibleItemInfos();
            int width = ClientRectangleWithoutScrollBar.Width;

            //
            drawDesc = new TreeDrawDescriptor(gc);
            drawDesc.ClientWidth = width;

            itemDrawDesc = new TreeItemDrawDescriptor(gc, Font, OwnerCtrl.Images);
            itemDrawDesc.Focused = Focused;
            itemDrawDesc.Counter = AnimationCounter;

            //
            foreach (ITreeItemInfo info in infos)
            {
                DrawItem(gc, info, drawDesc, itemDrawDesc);
            }
        }

        ///--------------------------------
        /// <summary>
        /// ドラックの開始
        /// </summary>
        protected void TrackDragDrop(DragDropEffects effects)
        {
            DataObject data = null;
            data = new DataObject();

            DoDragDrop(data, effects);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ITreeItemInfo GetItemInfo(ITreeItem item)
        {
            ITreeItemInfo[] itemInfos = ItemInfos;

            if (itemInfos == null)
            {
                return null;
            }

            try
            {
                var infos = itemInfos
                    .Where(i => i.Item == item);
                return infos.FirstOrDefault();
            }
            catch { return null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ITreeItemInfo GetItemInfo(int index)
        {
            ITreeItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return null;
            }

            if (index < 0 || index >= infos.Length)
            {
                return null;
            }

            return infos[index];
        }

        ///--------------------------------
        /// <summary>
        /// 指定ID番号、インデックスのアイテムの編集矩形の取得
        /// 論理座標で返ってくる
        /// </summary>
        protected Rectangle GetPartBounds(ITreeItem item, string partName)
        {
            Graphics gc = null;
            TreeItemDrawDescriptor desc = null;
            ITreeSubItem subItem = null;
            ITreeItemInfo info = null;
            Rectangle bounds;
            int bx = 0;
            int by = 0;

            if ((info = GetItemInfo(item)) == null)
            {
                return Rectangle.Empty;
            }

            bx = IndentLength * item.Depth;
            by = info.Y;

            try
            {
                gc = CreateGraphics();
                desc = new TreeItemDrawDescriptor(gc, Font, OwnerCtrl.Images);
                item.GetConstValues(desc.Parameters);

                if ((subItem = OwnerCtrl.SubItems[item.SubItemName]) != null)
                {
                    foreach (ITreePart part in subItem.GetParts(desc))
                    {
                        if (part.Name == partName)
                        {
                            bounds = part.Bounds;
                            bounds.X += bx;
                            bounds.Y += by;
                            return bounds;
                        }
                    }
                }
            }

            finally { gc.Dispose(); }
            return Rectangle.Empty;
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        public bool BeginEdit(ITreeItem item)
        {
            return BeginEdit(item, "Caption", null, false, false, 0);
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        protected bool BeginEdit(ITreeItem item, string name, string text, bool beginByEditBeginKey, bool immediateEdit, Keys immediateEditKey)
        {
            if (_ActiveEditor != null)
            {
                return false;
            }

            if (ReadOnly != false ||
                item == null)
            {
                return false;
            }

            //
            InplaceEditor editor = null;
            Rectangle bounds = GetPartBounds(item, name);

            bounds = ToClientRectangle(bounds);

            if ((editor = OwnerCtrl.InplaceEditors[name]) == null)
            {
                return false;
            }

            //editor.InitialText = text;
            //editor.InitialValue = new TextParameterValue( text != null ? text : String.Empty);

            //
            IParameterValue value = null;

            if ((value = item.GetValue(name)) != null)
            {
                editor.InitialValue = value;
            }

            editor.ParameterManipulator = value;

            if (value == null)
            {
                return false;
            }

            //
            if (immediateEdit != false)
            {
                if (editor.CanImmediateEdit == false)
                {
                    return false;
                }
            }

            //
            editor.OwnerCtrl = this;
            editor.Owner = this;
            editor.BeginByEditBeginKey = beginByEditBeginKey;

            Debug.WriteLine("BeginEdit");
            if (editor.BeginEdit(bounds, text, ReadOnly) == false)
            {
                return false;
            }

            _ActiveEditor = editor;
            _EditingItem = item;

            //
            if (editor.Permanent == false)
            {
                //EndEdit( editor.Cancel);
                EndEdit();
            }
            else
            {

                OnEditBegan();
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 選択が変更された時に呼ばれる
        /// </summary>
        protected void OnSelectChanged()
        {
            OwnerCtrl.OnSelectChanged();
        }

        ///--------------------------------
        /// <summary>
        /// 編集が開始された時に呼ばれる
        /// </summary>
        protected void OnEditBegan()
        {
            OwnerCtrl.OnEditBegan();
        }

        ///--------------------------------
        /// <summary>
        /// 編集が終了した時に呼ばれる
        /// </summary>
        protected void OnEditEnded()
        {
            OwnerCtrl.OnEditEnded();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnDoubleClicked(string name, ITreeItem item)
        {
            TreeDoubleClickedAction action;

            action = OwnerCtrl.OnDoubleClicked(item, name);
            switch (action)
            {
                case TreeDoubleClickedAction.None:
                    break;

                case TreeDoubleClickedAction.ToggleExpand:
                    if (item.Expander != false)
                    {
                        ToggleExpandItem(item);
                        OwnerCtrl.UpdateInfos();
                    }
                    break;

                case TreeDoubleClickedAction.Edit:
                    BeginEdit(_CaretItem);
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnTrackContextMenu(MouseEventArgs e)
        {
            if (_ActiveEditor != null)
            {
                return;
            }

            ContextMenuStrip contextMenu = null;

            if ((contextMenu = OwnerCtrl.AttachedContextMenuStrip) != null)
            {
                contextMenu.Show(this, new Point(e.X, e.Y));
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンの押下処理
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (IsEditing == true)
            {
                EndEdit();
            }
            else
            {
                base.OnMouseDown(e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnLeave(EventArgs e)
        {
            base.OnLeave(e);

            Invalidate();
        }

        ///@@@
        ///------------------------------------------------------------------------
        /// <summary>
        /// 基本のオペレータ
        /// </summary>
        public class TreeCtrlCoreOperator : OperatableControlOperator
        {

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public TreeCtrlCoreOperator(TreeCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected TreeCtrlCore OwnerCtrl
            {
                get { return (TreeCtrlCore)base.OperatableControl; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedControlKey
            {
                get { return (Control.ModifierKeys & Keys.Control) != 0; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedShiftKey
            {
                get { return (Control.ModifierKeys & Keys.Shift) != 0; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected Point GetValidLogialPoint(int x, int y)
            {
                Size size = OwnerCtrl.ClientSize;

                if (x < 0) { x = 0; }
                if (y < 0) { y = 0; }
                if (x > size.Width) { x = size.Width - 1; }
                if (y > size.Height) { y = size.Height - 1; }

                return OwnerCtrl.ToLogicalPoint(new Point(x, y));
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// 待機状態のオペレータ
        /// </summary>
        private class ExchangeOperator : TreeCtrlCoreOperator
        {
            private Point _FirstMouseDownPoint = new Point(-1, -1);
            private bool _InterimDrag = false;  //ドラッグ処理の可能性
            private bool _InterimSelect = false;  //選択処理の可能性

            /// キャプション編集の可能性がある場合にtrueになる
            /// Down、Upの両方がtrueにならないと編集モードにはならない
            private bool _InterimEditCaptionDown = false;
            private bool _InterimEditCaptionUp = false;
            private ITreeItem _InterimEditItem = null;

            ///
            private bool interimOneSelectItemEnabled = false;
            private Point interimOneSelectItemPoint;
            private ITreeItem interimOneSelectItem = null;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public ExchangeOperator(TreeCtrlCore ctrl) : base(ctrl)
            {
            }

            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _InterimDrag = false;
                _InterimSelect = false;
                _InterimEditCaptionDown = false;
                _InterimEditCaptionUp = false;
                _InterimEditItem = null;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        return MouseLeftButtonDownOperation(e);

                    case MouseButtons.Middle:
                        break;

                    case MouseButtons.Right:
                        MouseRightButtonDownOperation(e);
                        break;
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        return MouseLeftButtonUpOperation(e);

                    case MouseButtons.Middle:
                        break;

                    case MouseButtons.Right:
                        TrackContextMenu(e);
                        break;
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDoubleClick(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        MouseLeftButtonDoubleClickedOperation(e);
                        _InterimEditCaptionDown = false;
                        _InterimEditCaptionUp = false;
                        _InterimEditItem = null;
                        break;
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                Size dragSize = SystemInformation.DragSize;

                if (_InterimDrag != false || _InterimSelect != false)
                {
                    if (Math.Abs(e.X - _FirstMouseDownPoint.X) > dragSize.Width ||
                        Math.Abs(e.Y - _FirstMouseDownPoint.Y) > dragSize.Height)
                    {

                        if (_InterimDrag != false)
                        {
                            OwnerCtrl.ChangeOperatorToMove(_FirstMouseDownPoint);
                        }

                        if (_InterimSelect != false &&
                            OwnerCtrl.MultiSelect != false)
                        {
                            OwnerCtrl.ChangeOperatorToSelect(_FirstMouseDownPoint);
                        }
                    }
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseWheel(MouseEventArgs e)
            {
                int relativeMove = 0;

                relativeMove = e.Delta * SystemInformation.MouseWheelScrollLines / 120;

                //
                if (PressedShiftKey != false)
                {
                    if (OwnerCtrl.ScrollHorizontalScrollBarRelative
                        (-relativeMove * 16) != false)
                    {
                        OwnerCtrl.Invalidate();
                    }
                    return true;
                }

                //
                if (OwnerCtrl.ScrollVerticalScrollBarRelative
                    (-relativeMove) != false)
                {
                    OwnerCtrl.UpdateScrollBar();
                    OwnerCtrl.Invalidate();
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyDown(KeyEventArgs e)
            {
                Rectangle bounds;
                bool redraw = false;

                switch (e.KeyData)
                {
                    case Keys.Up:
                        redraw = OwnerCtrl.MoveCaretPreviousItem();
                        break;

                    case Keys.Down:
                        redraw = OwnerCtrl.MoveCaretNextItem();
                        break;

                    //ノードが開いていたら閉じる
                    //閉じていたらカレットを親のアイテムに移動
                    case Keys.Left:
                        if (OwnerCtrl.CaretItem == null)
                        {
                            break;
                        }

                        if (OwnerCtrl.CaretItem.Expanded == false)
                        {
                            redraw = OwnerCtrl.MoveCaretParentItem();
                        }
                        else
                        {

                            if (OwnerCtrl.CanExpanded(OwnerCtrl.CaretItem) != false)
                            {
                                OwnerCtrl.CaretItem.Expanded = !OwnerCtrl.CaretItem.Expanded;
                                OwnerCtrl.UpdateInfos();
                            }
                        }
                        break;

                    //ノードが閉じていたら開く
                    //開いていたらカレットを最初の子供のアイテムに移動
                    case Keys.Right:
                        if (OwnerCtrl.CaretItem == null)
                        {
                            break;
                        }

                        if (OwnerCtrl.CaretItem.Expanded != false)
                        {
                            redraw = OwnerCtrl.MoveCaretFirstChildItem();
                        }
                        else
                        {

                            if (OwnerCtrl.CanExpanded(OwnerCtrl.CaretItem) != false)
                            {
                                OwnerCtrl.CaretItem.Expanded = !OwnerCtrl.CaretItem.Expanded;
                                OwnerCtrl.UpdateInfos();
                            }
                        }
                        break;

                    //カレットを親のアイテムに移動
                    case Keys.Back:
                        redraw = OwnerCtrl.MoveCaretParentItem();
                        break;

                    case Keys.PageUp:
                        redraw = OwnerCtrl.MoveCaretPreviousPageItem();
                        break;

                    case Keys.PageDown:
                        redraw = OwnerCtrl.MoveCaretNextPageItem();
                        break;

                    case Keys.Home:
                        redraw = OwnerCtrl.MoveCaretMostPreviousItem();
                        break;

                    case Keys.End:
                        redraw = OwnerCtrl.MoveCaretMostNextItem();
                        break;

                    case Keys.Apps:
                        bounds = OwnerCtrl.GetCaretItemClientBounds();
                        TrackContextMenu
                            (new MouseEventArgs(MouseButtons.None, 0, bounds.X, bounds.Y, 0));
                        break;

                    case Keys.Return:
                        bounds = OwnerCtrl.GetCaretItemClientBounds();
                        MouseLeftButtonDoubleClickedOperation
                            (new MouseEventArgs(MouseButtons.None, 0, bounds.X, bounds.Y, 0));
                        break;
                }

                //
                if (redraw != false)
                {
                    OwnerCtrl.Invalidate();
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnAnimation(EventArgs e)
            {
                //OwnerCtrl.DrawCaretItem();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnTimer(EventArgs e)
            {
                if (_InterimEditCaptionDown != false &&
                    _InterimEditCaptionUp != false)
                {

                    OwnerCtrl.StopTimer();
                    OwnerCtrl.BeginEdit(_InterimEditItem);

                    _InterimEditCaptionDown = false;
                    _InterimEditCaptionUp = false;
                    _InterimEditItem = null;
                }
            }

            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragDrop(DragEventArgs e)
            {
                if (e.Data.GetDataPresent(DataFormats.FileDrop) == true)
                {
                    string[] filePaths = (string[])e.Data.GetData(DataFormats.FileDrop);
                    Array.Sort(filePaths);

                    TreeCollisionResult result = OwnerCtrl.CollisionForDrag(e.X, e.Y);
                    OwnerCtrl.DropFiles(filePaths, result.Item);
                }
            }

            /// <summary>
            /// マウスの左ボタンを押した時の処理です。
            /// </summary>
            protected bool MouseLeftButtonDownOperation(MouseEventArgs e)
            {
                SelectionItemProcessByPoint(e.X, e.Y);
                return true;
            }

            /// <summary>
            /// マウスの左ボタンを離した時の処理です。
            /// </summary>
            private bool MouseLeftButtonUpOperation(MouseEventArgs e)
            {
                //
                if (this.interimOneSelectItemEnabled != false)
                {
                    Size dragSize = SystemInformation.DragSize;

                    if (Math.Abs(this.interimOneSelectItemPoint.X - e.X) < dragSize.Width &&
                        Math.Abs(this.interimOneSelectItemPoint.Y - e.Y) < dragSize.Height)
                    {
                        ITreeItem item = this.interimOneSelectItem;
                        OwnerCtrl.UnselectedItem();
                        OwnerCtrl.SelectedItem(item, true);
                        OwnerCtrl.CaretWithoutFollowSelectedItem(item);

                        OwnerCtrl.OnSelectChanged();
                        OwnerCtrl.DemandInvalidate();
                    }
                }
                this.interimOneSelectItemEnabled = false;

                //
                if (_InterimEditCaptionDown != false)
                {
                    OwnerCtrl.StartTimer(SystemInformation.DoubleClickTime + 1);
                    _InterimEditCaptionUp = true;
                }

                _InterimDrag = false;
                _InterimSelect = false;
                return true;
            }

            /// <summary>
            /// マウスの右ボタンを押した時の処理
            /// </summary>
            private bool MouseRightButtonDownOperation(MouseEventArgs e)
            {
                SelectionItemProcessByPoint(e.X, e.Y);
                return true;
            }

            /// <summary>
            ///
            /// </summary>
            private void MouseLeftButtonDoubleClickedOperation(MouseEventArgs e)
            {
                TreeCollisionResult result = OwnerCtrl.Collision(e.X, e.Y);
                string name = result.Name;

                if (OwnerCtrl.FullRowDoubleClickedCaption != false)
                {
                    name = "Caption";
                }

                //
                switch (name)
                {
                    case "Caption":
                    case "Icon":
                        OwnerCtrl.UnselectedItem();
                        OwnerCtrl.SelectedItem(result.Item, true);

                        OwnerCtrl.OnSelectChanged();
                        OwnerCtrl.DemandInvalidate();
                        OwnerCtrl.OnDoubleClicked(name, result.Item);
                        break;
                }
            }

            /// <summary>
            ///
            /// </summary>
            private void SelectionItemProcessByPoint(int x, int y)
            {
                TreeCollisionResult result = OwnerCtrl.Collision(x, y);
                string name = result.Name;

                // 行選択が可能な場合には、"Caption"と同じ処理を行います。
                if (name == "Row" &&
                    OwnerCtrl.FullRowSelect != false)
                {
                    name = "Caption";
                }

                //
                switch (name)
                {
                    case "Caption":
                    case "Icon":
                        // シフトキーが押されている場合には、基準点からの範囲選択になります。
                        // (現在、未実装)
                        if (PressedShiftKey != false)
                        {
                            if (PressedControlKey == false ||
                                OwnerCtrl.MultiSelect == false)
                            {
                                OwnerCtrl.UnselectedItem();
                            }

                            OwnerCtrl.SelectedCardinalTo(result.Item);
                            OwnerCtrl.CaretWithoutFollowSelectedItem(result.Item);
                            break;
                        }

                        //コントロールキーが押されているのか？
                        if (PressedControlKey != false)
                        {
                            // 複数選択が有効なのか？
                            if (OwnerCtrl.MultiSelect != false)
                            {
                                OwnerCtrl.ToggleSelectedItem(result.Item);
                            }
                            else
                            {
                                bool selected = OwnerCtrl.IsSelected(result.Item);
                                OwnerCtrl.UnselectedItem();
                                OwnerCtrl.SelectedItem(result.Item, !selected);
                            }
                            OwnerCtrl.CaretWithoutFollowSelectedItem(result.Item);
                        }
                        else
                        {
                            // 非選択アイテムの場合には選択にします。
                            if (result.Selected == false)
                            {
                                OwnerCtrl.UnselectedItem();
                                OwnerCtrl.SelectedItem(result.Item, true);
                                OwnerCtrl.CaretWithoutFollowSelectedItem(result.Item);
                            }
                            else
                            {
                                //
                                this.interimOneSelectItemEnabled = true;
                                this.interimOneSelectItemPoint = new Point(x, y);
                                this.interimOneSelectItem = result.Item;
                            }

                            //
                            if (result.Selected != false)
                            {
                                _InterimEditCaptionDown = true;
                                _InterimEditCaptionUp = false;
                                _InterimEditItem = result.Item;
                            }
                        }

                        //ドラッグが可能なのか？
                        if (OwnerCtrl.AllowDragItem != false &&
                            OwnerCtrl.ReadOnly == false)
                        {
                            _InterimDrag = true;
                        }

                        _FirstMouseDownPoint.X = x;
                        _FirstMouseDownPoint.Y = y;
                        OwnerCtrl.Invalidate();
                        break;

                    case "Expander":
                        result.Info.Item.Expanded = !result.Info.Item.Expanded;
                        OwnerCtrl.UpdateInfos();
                        break;

                    case "None":
                        //シフトキーが押されている場合には何もしません。
                        if (PressedShiftKey != false)
                        {
                            break;
                        }

                        _InterimSelect = true;
                        _FirstMouseDownPoint.X = x;
                        _FirstMouseDownPoint.Y = y;

                        // アイテム表示領域外のクリックで選択解除を行なうのか？
                        if (OwnerCtrl.ClickedOutsideUnselectItem != false)
                        {
                            // コントロールキーが押されている場合には選択解除は行いません。
                            if (PressedControlKey == false)
                            {
                                OwnerCtrl.UnselectedItem();
                            }
                        }
                        break;
                }

                //
                OwnerCtrl.OnSelectChanged();
                OwnerCtrl.DemandInvalidate();
            }

            /// <summary>
            ///
            /// </summary>
            private void TrackContextMenu(MouseEventArgs e)
            {
                _InterimDrag = false;
                _InterimEditCaptionDown = false;
                _InterimEditCaptionUp = false;
                _InterimEditItem = null;

                OwnerCtrl.OnTrackContextMenu(e);
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// 選択処理状態のオペレータ
        /// </summary>
        private class SelectOperator : TreeCtrlCoreOperator
        {

            private Point _BeginPoint = new Point(-1, -1);
            private Point _EndPoint = new Point(-1, -1);
            private bool _LastControlKey = false;

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public SelectOperator(TreeCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _BeginPoint = OwnerCtrl.ToLogicalPoint(beginPoint);
                OwnerCtrl.ToggleSelectedItemByLogicalPoint(beginPoint);
                OwnerCtrl.BeginEventLoop();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                _EndPoint = GetValidLogialPoint(e.X, e.Y);

                OwnerCtrl.EndEventLoop();
                OwnerCtrl.RubberBand();

                //
                ITreeItem[] items = null;
                Rectangle rect;
                bool selectChanged = false;

                if (PressedControlKey == false && PressedShiftKey == false)
                {
                    OwnerCtrl.UnselectedItem();
                    selectChanged = true;
                }

                rect = OwnerCtrl.NormalizedRectangle(_BeginPoint, _EndPoint);

                if (PressedControlKey != false)
                {
                    items = OwnerCtrl.ToggleSelectedItemByLogicalRectangle(rect);
                }
                else
                {
                    items = OwnerCtrl.SelectedItemByLogicalRectangle(rect);
                }

                if (items.Length > 0)
                {
                    selectChanged = true;
                }

                //
                if (selectChanged != false)
                {
                    OwnerCtrl.OnSelectChanged();
                    OwnerCtrl.DemandInvalidate();
                }
                else
                {

                    OwnerCtrl.Invalidate();
                }

                OwnerCtrl.ChangeOperatorToExchange(_EndPoint);
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                _EndPoint = GetValidLogialPoint(e.X, e.Y);
                OwnerCtrl.RubberBand(_BeginPoint, _EndPoint, _LastControlKey);

                OwnerCtrl.Invalidate();
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnEventLoop(MouseEventArgs e)
            {
                OwnerCtrl.DragScroll(e.X, e.Y);

                //コントロールキーの状態が変化したら再描画を行う
                if (_LastControlKey != PressedControlKey)
                {
                    OwnerCtrl.Invalidate();
                }
                _LastControlKey = PressedControlKey;

                //
                _EndPoint = GetValidLogialPoint(e.X, e.Y);
                OwnerCtrl.RubberBand(_BeginPoint, _EndPoint, _LastControlKey);
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// 移動処理状態のオペレータ
        /// </summary>
        private class MoveOperator : TreeCtrlCoreOperator
        {

            private bool _CanMove = false;

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MoveOperator(TreeCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                //DragDropEffects effects = DragDropEffects.None;

                //開始時の座標で選択する
                //ドラックの開始しただけでは、選択にならない為
                //選択にしないと、ドラックを開始したアイテムが移動の対象にならない
                beginPoint = OwnerCtrl.ToLogicalPoint(beginPoint);

                //クリックの開始位置に選択されているアイテムが無く、コントロールキーが
                //押されていない場合に、全選択を解除する
                TreeCollisionResult result = null;

                result = OwnerCtrl.Collision(beginPoint.X, beginPoint.Y);

                switch (result.Name)
                {
                    case "Caption":
                    case "Icon":
                        if (result.Selected == false && PressedControlKey == false)
                        {
                            OwnerCtrl.UnselectedItem();
                        }
                        break;

                    default:
                        return;
                }

                OwnerCtrl.SelectedItemByLogicalRectangle
                    (new Rectangle(beginPoint, new Size(0, 0)));

                //
                _CanMove = true;        //とりあえず

                OwnerCtrl.BeginEventLoop();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                OwnerCtrl.EndEventLoop();
                OwnerCtrl.DragMarker();

                //
                TreeCollisionResult result = null;

                result = OwnerCtrl.CollisionForDrag(e.X, e.Y);

                if (result.Name != "None")
                {
                    MoveItem(result.Info, result.InPosition);
                }

                //
                OwnerCtrl.Invalidate();
                OwnerCtrl.ChangeOperatorToExchange(GetValidLogialPoint(e.X, e.Y));
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                if (_CanMove == false)
                {
                    OwnerCtrl.DragMarker();
                    OwnerCtrl.Invalidate();
                    return true;
                }

                TreeCollisionResult result = null;

                OwnerCtrl.DragScroll(e.X, e.Y);
                result = OwnerCtrl.CollisionForDrag(e.X, e.Y);

                if (result.Name == "Row")
                {
                    OwnerCtrl.DragMarker(result.Item, result.InPosition);
                }
                else
                {
                    OwnerCtrl.DragMarker();
                }

                OwnerCtrl.Invalidate();
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnEventLoop(MouseEventArgs e)
            {
                OwnerCtrl.DragScroll(e.X, e.Y);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool MoveItem(ITreeItemInfo info, TreeItemInPosition inPosition)
            {
                switch (inPosition)
                {
                    case TreeItemInPosition.Center:
                        return OwnerCtrl.MoveSelectedItemToChild(info.Item);

                    case TreeItemInPosition.Upper:
                        return OwnerCtrl.MoveSelectedItemToPreviousSibling(info.Item);

                    case TreeItemInPosition.Lower:
                        return OwnerCtrl.MoveSelectedItemToNextSibling(info.Item);
                }
                return false;
            }
        }
    }
}
