﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Windows.CommandBars;

    /// <summary>
    /// ToolStrip とコマンドバーモデルの同期をサポートします。
    /// </summary>
    public class ToolStripAdapter
    {
        private ToolStrip _toolStrip = null;
        private CommandBar _commandBar = null;

        private Dictionary<Type, CreateItemHandler> _itemFactoryMethods = new Dictionary<Type, CreateItemHandler>();

        private ICommandTarget _commandTarget = null;
        private CommandStatus _commandStatusMask = CommandStatus.Unsupported;

        private bool _isBuildUILocked = false;
        private bool _isBuildUIDirty = false;

        /// <summary>
        /// ToolStripItem を作成するメソッド。
        /// </summary>
        /// <param name="item">コマンドバーアイテム。</param>
        /// <param name="status">コマンド状態。</param>
        /// <param name="target">コマンドターゲット。</param>
        /// <returns>作成された ToolStripItem。</returns>
        private delegate ToolStripItem CreateItemHandler(CommandBarItem item, CommandStatus status, ICommandTarget target);

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="toolStrip">対象 ToolStrip。</param>
        public ToolStripAdapter(ToolStrip toolStrip)
        {
            if (null == toolStrip) { throw new ArgumentNullException("toolStrip"); }
            _toolStrip = toolStrip;

            _itemFactoryMethods.Add(typeof(CommandBarMenuItem), CreateMenuItem);
            _itemFactoryMethods.Add(typeof(CommandBarPopupItem), CreatePopupMenuItem);
            _itemFactoryMethods.Add(typeof(CommandBarButton), CreateButton);
            _itemFactoryMethods.Add(typeof(CommandBarLabel), CreateLabel);
            _itemFactoryMethods.Add(typeof(CommandBarDropDownButton), CreateDropDownButton);
            _itemFactoryMethods.Add(typeof(CommandBarComboBox), CreateComboBox);

            if (toolStrip is MenuStrip)
            {

                MenuStrip menuStrip = toolStrip as MenuStrip;
                menuStrip.MenuActivate += OnMenuActivate;
                menuStrip.MenuDeactivate += OnMenuDeactivate;

            }
        }

        /// <summary>
        /// 対象 ToolStrip を取得します。
        /// </summary>
        public ToolStrip ToolStrip
        {
            get { return _toolStrip; }
        }

        /// <summary>
        /// ToolStrip のモデルを取得または設定します。
        /// </summary>
        public CommandBar CommandBar
        {
            get { return _commandBar; }
            set
            {
                if (_commandBar == value) { return; }

                _commandBar = value;
                BuildUI();
            }
        }

        /// <summary>
        /// ToolStrip のコマンドターゲットを取得または設定します。
        /// </summary>
        public ICommandTarget CommandTarget
        {
            get { return _commandTarget; }
            set
            {
                if (_commandTarget == value) { return; }

                _commandTarget = value;
                BuildUI();
            }
        }

        /// <summary>
        /// マスクするコマンドの状態を取得または設定します。
        /// </summary>
        public CommandStatus CommandStatusMask
        {
            get { return _commandStatusMask; }
            set { _commandStatusMask = value; }
        }

        /// <summary>
        /// コマンドが実行されると発生します。
        /// </summary>
        public event CommandEventHandler CommandExecuted;

        /// <summary>
        /// 既存の ToolStrip からモデルを作成し、ToolStripAdapter を構築します。
        /// </summary>
        /// <param name="commandService">コマンドサービス。</param>
        /// <param name="toolStrip">ソースとなる ToolStrip。</param>
        /// <returns>作成された ToolStripAdapter。</returns>
        public static ToolStripAdapter FromToolStrip(CommandService commandService, ToolStrip toolStrip)
        {
            ToolStripAdapter adapter = new ToolStripAdapter(toolStrip);
            adapter.CommandBar = new CommandBarFactory(commandService).CreateCommandBar(toolStrip);

            return adapter;
        }

        /// <summary>
        /// モデルから ToolStrip を再構築します。
        /// </summary>
        public void BuildUI()
        {
            if (_isBuildUILocked)
            {
                _isBuildUIDirty = true;
                return;
            }

            _toolStrip.SuspendLayout();

            _toolStrip.Items.Clear();

            if (null != _commandBar && null != _commandTarget)
            {
                BuildUI(_toolStrip.Items, _commandBar.Groups, _commandTarget);
            }

            _toolStrip.ResumeLayout(true);

            _isBuildUIDirty = false;
        }

        /// <summary>
        /// ToolStrip 再構築要求の受付を停止します。
        /// </summary>
        public void SuspendBuildUI()
        {
            _isBuildUILocked = true;
        }

        /// <summary>
        /// ToolStrip 再構築要求の受付を再開します。
        /// </summary>
        public void ResumeBuildUI()
        {
            if (!_isBuildUILocked) { return; }

            _isBuildUILocked = false;

            if (_isBuildUIDirty)
            {
                BuildUI();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public CommandBarItem[] Find(Command targetCommand)
        {
            foreach (CommandBarItemGroup group in CommandBar.Groups)
            {
                CommandBarItem[] items = group.FindItems(targetCommand.Uri)
                    .Cast<CommandBarItem>()
                    .ToArray();
                if (items.Length > 0)
                {
                    return items;
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual ToolStripDropDownItem CreateDropDownItem(CommandBarItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            return new ToolStripMenuItem(item.Text);
        }

        protected virtual ToolStripItem CreateItem(CommandBarItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            return new ToolStripMenuItem(item.Text);
        }

        protected void OnExecuteCommand(object sender, EventArgs e)
        {
            ToolStripItem targetItem = sender as ToolStripItem;
            if (null == targetItem) { throw new Exception("item must be ToolStripItem."); }

            CommandBarExecutableItem executableItemModel = (targetItem.Tag as CommandBarExecutableItem);
            if (null == executableItemModel) { throw new Exception("item must refer CommandBarItem."); }

            ExecuteCommand(executableItemModel.Command);
        }

        protected void OnCommandExecuted(CommandEventArgs e)
        {
            if (null != CommandExecuted)
            {
                CommandExecuted(this, e);
            }
        }

        public bool ExecuteCommand(Command command, IEnumerable<KeyValuePair<string, string>> parameters)
        {
            return this.ExecuteCommand(new Command(command, parameters));
        }

        private bool ExecuteCommand(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }
            if (null == _commandTarget) { throw new Exception("command target not selected."); }

            ICommandTarget executeTarget = _commandTarget.FindTarget(command);
            if (null == executeTarget) { throw new Exception("command target not found."); }

            bool result = executeTarget.Execute(command);
            OnCommandExecuted(new CommandEventArgs(command, result));

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        private void BuildUI(ToolStripItemCollection items, IEnumerable<CommandBarItemGroup> groups, ICommandTarget target)
        {
            if (null == groups) { throw new ArgumentNullException("groups"); }
            if (null == target) { throw new ArgumentNullException("target"); }

            bool firstGroup = true;
            bool separatorAdded = false;

            foreach (CommandBarItemGroup group in groups)
            {

                foreach (CommandBarItem item in group.Items)
                {

                    if (!item.Visible) { continue; }
                    if (!_itemFactoryMethods.ContainsKey(item.GetType())) { continue; }

                    Command command = GetCommand(item);
                    CommandStatus status = CommandStatus.SupportedAndEnabledAndVisible;

                    if (null != command)
                    {

                        ICommandTarget executeTarget = target.FindTarget(command);

                        if (null != executeTarget)
                        {
                            status = GetCommandStatus(executeTarget, command);
                            if (!status.IsVisible()) { continue; }
                        }
                        else
                        {
                            status = CommandStatus.Supported;
                        }

                    }

                    ToolStripItem toolItem = _itemFactoryMethods[item.GetType()](item, status, target);
                    if (null == toolItem) { continue; }

                    if (!item.Enable)
                    {
                        toolItem.Enabled = false;
                    }

                    // グループの先頭アイテムの前にセパレータを追加する
                    // ただしメニューの先頭アイテムは除く
                    if (!separatorAdded && !firstGroup)
                    {
                        items.Add(CreateSeparator());
                        separatorAdded = true;
                    }

                    items.Add(toolItem);

                }

                if (0 < items.Count)
                {
                    firstGroup = false;
                }

                separatorAdded = false;

            }
        }

        private Command GetCommand(CommandBarItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            if (!(item is CommandBarExecutableItem)) { return null; }
            return (item as CommandBarExecutableItem).Command;
        }

        private CommandStatus GetCommandStatus(ICommandTarget commandTaret, Command command)
        {
            if (null == commandTaret) { throw new ArgumentNullException("commandTaret"); }
            if (null == command) { throw new ArgumentNullException("command"); }

            return commandTaret.QueryStatus(command) & ~_commandStatusMask;
        }

        private ToolStripMenuItem CreateMenuItem(CommandBarItem item, CommandStatus status, ICommandTarget target)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == target) { throw new ArgumentNullException("target"); }
            if (!(item is CommandBarMenuItem)) { throw new ArgumentException("invalid item model type."); }

            CommandBarMenuItem menuItem = item as CommandBarMenuItem;
            ToolStripMenuItem toolItem = item.Tag as ToolStripMenuItem;

            if (null == toolItem)
            {

                toolItem = new ToolStripMenuItem(item.Text, menuItem.Image);

                item.Tag = toolItem;
                toolItem.Click += OnExecuteCommand;

            }
            else
            {
                toolItem.Text = item.Text;
                toolItem.Image = menuItem.Image;
            }

            toolItem.ShortcutKeyDisplayString = (item as CommandBarMenuItem).ShortcutKeyText;
            toolItem.Tag = item;
            toolItem.Enabled = status.IsEnabled();
            toolItem.Visible = status.IsVisible();
            toolItem.Checked = status.IsChecked();

            return toolItem;
        }

        private ToolStripMenuItem CreatePopupMenuItem(CommandBarItem item, CommandStatus status, ICommandTarget target)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == target) { throw new ArgumentNullException("target"); }
            if (!(item is CommandBarPopupItem)) { throw new ArgumentException("invalid item model type."); }

            ToolStripMenuItem toolItem = new ToolStripMenuItem(item.Text);
            BuildUI(toolItem.DropDownItems, (item as CommandBarPopupItem).Groups, target);

            // 子供が存在しない場合は、DropDownButton 自体も追加しない
            if (0 == toolItem.DropDownItems.Count) { return null; }

            // 子供が全て無効の場合は、無効にする
            toolItem.Enabled = !IsAllItemsDisabled(toolItem.DropDownItems);

            return toolItem;
        }

        private ToolStripButton CreateButton(CommandBarItem item, CommandStatus status, ICommandTarget target)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == target) { throw new ArgumentNullException("target"); }
            if (!(item is CommandBarButton)) { throw new ArgumentException("invalid item model type."); }

            CommandBarButton button = item as CommandBarButton;
            ToolStripButton toolItem = item.Tag as ToolStripButton;

            if (null == toolItem)
            {

                toolItem = new ToolStripButton(button.Text, button.Image);

                item.Tag = toolItem;
                toolItem.Click += OnExecuteCommand;

            }
            else
            {
                toolItem.Text = item.Text;
                toolItem.Image = button.Image;
                toolItem.ToolTipText = button.ToolTipText;
                if (button.ShortcutKeyText != string.Empty)
                {
                    toolItem.ToolTipText += "(" + button.ShortcutKeyText + ")";
                }
            }

            toolItem.Tag = item;
            toolItem.Enabled = status.IsEnabled();
            toolItem.Visible = status.IsVisible();
            toolItem.Checked = status.IsChecked();
            toolItem.Width = item.Width;
            toolItem.AutoSize = (0 >= item.Width);

            switch (button.Style)
            {
                case CommandBarButtonStyle.TextOnly:
                    toolItem.DisplayStyle = ToolStripItemDisplayStyle.Text;
                    break;

                case CommandBarButtonStyle.ImageOnly:
                    toolItem.DisplayStyle = ToolStripItemDisplayStyle.Image;
                    break;

                case CommandBarButtonStyle.TextAfterImage:
                    toolItem.DisplayStyle = ToolStripItemDisplayStyle.ImageAndText;
                    toolItem.TextImageRelation = TextImageRelation.ImageBeforeText;
                    break;

                case CommandBarButtonStyle.TextBeforeImage:
                    toolItem.DisplayStyle = ToolStripItemDisplayStyle.ImageAndText;
                    toolItem.TextImageRelation = TextImageRelation.TextBeforeImage;
                    break;
            }

            return toolItem;
        }

        private ToolStripLabel CreateLabel(CommandBarItem item, CommandStatus status, ICommandTarget target)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (!(item is CommandBarLabel)) { throw new ArgumentException("invalid item model type."); }

            ToolStripLabel toolItem = item.Tag as ToolStripLabel;

            if (null == toolItem)
            {
                toolItem = new ToolStripLabel(item.Text);
                item.Tag = toolItem;
            }
            else
            {
                toolItem.Text = item.Text;
            }

            toolItem.Width = item.Width;
            toolItem.AutoSize = (0 >= item.Width);

            return toolItem;
        }

        private ToolStripDropDownButton CreateDropDownButton(CommandBarItem item, CommandStatus status, ICommandTarget target)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == target) { throw new ArgumentNullException("target"); }
            if (!(item is CommandBarDropDownButton)) { throw new ArgumentException("invalid item model type."); }

            ToolStripDropDownButton toolItem = item.Tag as ToolStripDropDownButton;

            if (null == toolItem)
            {
                toolItem = new ToolStripDropDownButton(item.Text);
                item.Tag = toolItem;
            }
            else
            {
                toolItem.Text = item.Text;
            }

            toolItem.Width = item.Width;
            toolItem.AutoSize = (0 >= item.Width);

            BuildUI(toolItem.DropDownItems, (item as CommandBarPopupItem).Groups, target);

            // 子供が存在しない場合は、DropDownButton 自体も追加しない
            if (0 == toolItem.DropDownItems.Count) { return null; }

            // 子供が全て無効の場合は、無効にする
            toolItem.Enabled = !IsAllItemsDisabled(toolItem.DropDownItems);

            return toolItem;
        }

        private ToolStripComboBox CreateComboBox(CommandBarItem item, CommandStatus status, ICommandTarget target)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (!(item is CommandBarComboBox)) { throw new ArgumentException("invalid item model type."); }

            ToolStripComboBox toolItem = item.Tag as ToolStripComboBox;

            if (null == toolItem)
            {
                toolItem = new ToolStripComboBox();
                item.Tag = toolItem;
            }

            toolItem.Width = item.Width;
            toolItem.AutoSize = (0 >= item.Width);

            switch (((CommandBarComboBox)item).DropDownStyle)
            {
                case CommandBarComboBoxStyle.DropDownList:
                    toolItem.DropDownStyle = ComboBoxStyle.DropDownList;
                    break;

                case CommandBarComboBoxStyle.DropDown:
                default:
                    toolItem.DropDownStyle = ComboBoxStyle.DropDown;
                    break;
            }

            return toolItem;
        }

        private ToolStripSeparator CreateSeparator()
        {
            return new ToolStripSeparator();
        }

        /// <summary>
        /// 全てのアイテムが無効かどうか調べます。
        /// </summary>
        /// <param name="items">対象アイテム。</param>
        /// <returns>全て無効な場合は true、それ以外の場合は false。</returns>
        private bool IsAllItemsDisabled(ToolStripItemCollection items)
        {
            if (null == items) { throw new ArgumentNullException("items"); }

            foreach (ToolStripItem toolItemChild in items)
            {
                if (toolItemChild.Enabled) { return false; }
            }

            return true;
        }

        private void OnMenuActivate(object sender, EventArgs e)
        {
            SuspendBuildUI();
        }

        private void OnMenuDeactivate(object sender, EventArgs e)
        {
            _toolStrip.BeginInvoke(new MethodInvoker(OnResumeBuildUI));
        }

        private void OnResumeBuildUI()
        {
            ResumeBuildUI();
        }
    }
}
