﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Parameters;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundFoundation.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public class ParameterListAdapter : IListItemsSource
    {
        private ParameterListItemCollection items = null;
        private IList<ComponentReference> ownerParameterList = null;

        private bool suspendUpdatedEvent = false;
        private bool updatedEventWasIgnored = false;

        private bool suspendCollectionChanged = false;
        private bool suspendParameterCollectionChanged = false;

        public event EventHandler Updated;
        public event OperationExecutedEventHandler OperationExecuted;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ParameterListAdapter()
        {
            InterlockParameter = true;

            this.items = new ParameterListItemCollection();
            this.items.CollectionChanged += OnCollectionChanged;
        }

        /// <summary>
        /// パラメータコレクション
        /// </summary>
        public IList<ComponentReference> OwnerItemList
        {
            get { return this.ownerParameterList; }
        }

        ///--------------------------------
        /// <summary>
        /// パラメータコレクションを登録
        /// </summary>
        public void AddItems(IList<ComponentReference> list)
        {
            INotifyCollectionChanged ownerList = this.ownerParameterList as INotifyCollectionChanged;
            if (ownerList != null)
            {
                ownerList.CollectionChanged -= OnParameterCollectionChanged;
            }

            InterlockParameter = false;
            Items.Clear();

            if (list != null)
            {
                foreach (IParameterProvider provider in list)
                {
                    Items.Add(CreateListItem(provider));
                }
            }

            InterlockParameter = true;

            this.ownerParameterList = list;

            ownerList = this.ownerParameterList as INotifyCollectionChanged;
            if (ownerList != null)
            {
                ownerList.CollectionChanged += OnParameterCollectionChanged;
            }
        }

        ///--------------------------------
        /// <summary>
        /// Itemsの取得
        /// </summary>
        public ParameterListItemCollection Items
        {
            get { return this.items; }
        }

        ///--------------------------------
        /// <summary>
        /// Itemsの取得
        /// </summary>
        IListItemCollection IListItemsSource.Items
        {
            get { return Items; }
        }

        ///--------------------------------
        /// <summary>
        /// ロックの設定、取得
        /// </summary>
        public bool InterlockParameter { get; set; }

        ///--------------------------------
        /// <summary>
        /// Transaction の開始
        /// </summary>
        public void BeginTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.BeginTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        /// Transaction の終了
        /// </summary>
        public void EndTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.EndTransaction();
            }

            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        /// Transaction のキャンセル
        /// </summary>
        public void CancelTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        /// オペレーションヒストリー
        /// </summary>
        public OperationHistory OperationHistory { get; set; }

        ///--------------------------------
        /// <summary>
        /// ソート
        /// </summary>
        public void Sort(string name, Type type, SortOrder sortOrder)
        {
            ParameterListItem[] items = Items.ToArray();

            if (items.Length <= 0)
            {
                return;
            }

            //
            switch (sortOrder)
            {
                case SortOrder.Ascending:
                    //Array.Sort(items, CreateComparer(name, type, true));
                    items = Sort(items, CreateComparer(name, type, true));
                    break;

                case SortOrder.Descending:
                    //Array.Sort(items, CreateComparer(name, type, false));
                    items = Sort(items, CreateComparer(name, type, false));
                    break;

                case SortOrder.None:
                    items = SortByOriginalOrder(items);
                    break;
            }

            InterlockParameter = false;
            Items.Clear();
            foreach (IListItem item in items)
            {
                Items.Add(item);
            }
            InterlockParameter = true;
        }

        ///--------------------------------
        /// <summary>
        /// 値の設定
        /// </summary>
        public virtual void SetValue(IParameterProvider provider, string name, object value)
        {
            //Operation: ここで Operationにする

            SetParameterOperation operation = null;

            if (provider.Parameters.ContainsKey(name) == true)
            {
                operation = new SetParameterOperation(provider.Parameters, name, value);
                operation.Execute();
                OnOperationExecuted(operation);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SuspendUpdatedEvent()
        {
            this.suspendUpdatedEvent = true;
            this.updatedEventWasIgnored = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ResumeUpdatedEvent()
        {
            this.suspendUpdatedEvent = false;

            if (this.updatedEventWasIgnored != false)
            {
                OnUpdated();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ParameterListItem CreateListItem(IParameterProvider provider)
        {
            return new ParameterListItem(provider);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ListComparer CreateComparer(string name, Type type, bool ascending)
        {
            return new ListComparer(name, type, ascending);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ParameterListItem GetNextSibling(ParameterListItem item)
        {
            int index = this.items.IndexOf(item) + 1;

            if (index >= this.items.Count)
            {
                return null;
            }

            return this.items[index] as ParameterListItem;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnOperationExecuted(Operation operation)
        {
            if (OperationHistory != null)
            {
                OperationHistory.AddOperation(operation);
            }

            OnOperationExecuted(new OperationExecutedEventArgs(operation));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnOperationExecuted(OperationExecutedEventArgs e)
        {
            if (OperationExecuted != null)
            {
                OperationExecuted(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnUpdated()
        {
            OnUpdated(new EventArgs());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpdated(EventArgs e)
        {
            if (this.suspendUpdatedEvent != false)
            {
                this.updatedEventWasIgnored = true;
                return;
            }

            if (Updated != null)
            {
                Updated(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AddParameter(ParameterListItem item)
        {
            if (InterlockParameter == false)
            {
                return;
            }

            // Operation ここで Operation にする
            ParameterListItem nextSiblingItem = null;
            IList<ComponentReference> parent = null;
            ComponentReference nextSibling = null;

            nextSiblingItem = GetNextSibling(item);
            parent = this.ownerParameterList;

            nextSibling =
                (ComponentReference)(nextSiblingItem != null ? nextSiblingItem.Target : null);

            InsertParameterOperation operation = null;
            operation = new InsertParameterOperation(
                                                     parent,
                                                     nextSibling,
                                                     new[] { (ComponentReference)(item.Target) }
                                                     );
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RemoveParameter(ParameterListItem item)
        {
            if (InterlockParameter == false)
            {
                return;
            }

            //Operation: ここで Operationにする
            RemoveParameterOperation operation = null;
            operation = new RemoveParameterOperation(
                                                     this.ownerParameterList,
                                                     new[] { (ComponentReference)(item.Target) }
                                                     );
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.suspendParameterCollectionChanged = true;

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (ParameterListItem newItem in e.NewItems)
                    {
                        newItem.Adapter = this;

                        if (this.suspendCollectionChanged == false)
                        {
                            AddParameter(newItem);
                        }
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (ParameterListItem oldItem in e.OldItems)
                    {
                        if (this.suspendCollectionChanged == false)
                        {
                            RemoveParameter(oldItem);
                        }
                        oldItem.Adapter = null;
                    }
                    break;
            }

            this.suspendParameterCollectionChanged = false;

            //
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ParameterListItem[] FindParameterListItems(IParameterProvider provider)
        {
            return this.items.Cast<ParameterListItem>()
                .Where(i => i.Target == provider).ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AddListItem(ComponentReference provider)
        {
            if (InterlockParameter == false)
            {
                return;
            }

            IParameterProvider nextProvider = null;
            int index = -1;

            index = this.ownerParameterList.IndexOf(provider) + 1;
            if (index < this.ownerParameterList.Count)
            {
                nextProvider = this.ownerParameterList[index];

                ParameterListItem[] items = null;

                items = FindParameterListItems(nextProvider);
                Debug.Assert(items.Length == 1, "There are plural target parameter");
                index = this.items.IndexOf(items[0]);
                this.items.Insert(index, CreateListItem(provider));
            }
            else
            {
                this.items.Add(CreateListItem(provider));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RemoveListItem(IParameterProvider provider)
        {
            if (InterlockParameter == false)
            {
                return;
            }

            ParameterListItem[] items = FindParameterListItems(provider);
            foreach (ParameterListItem item in items)
            {
                this.items.Remove(item);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnParameterCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (this.suspendParameterCollectionChanged != false)
            {
                return;
            }

            this.suspendCollectionChanged = true;

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (ComponentReference newItem in e.NewItems)
                    {
                        AddListItem(newItem);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (ComponentReference oldItem in e.OldItems)
                    {
                        RemoveListItem(oldItem);
                    }
                    break;
            }

            this.suspendCollectionChanged = false;

            //
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ParameterListItem GetItemByParameter(ParameterListItem[] items,
                                                     IParameterProvider provider)
        {
            foreach (ParameterListItem item in items)
            {
                if (item.Target == provider)
                {
                    return item;
                }
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ParameterListItem[] SortByOriginalOrder(ParameterListItem[] items)
        {
            List<ParameterListItem> list = new List<ParameterListItem>();
            ParameterListItem item = null;

            if (this.ownerParameterList != null)
            {
                foreach (IParameterProvider provider in this.ownerParameterList)
                {
                    item = GetItemByParameter(items, provider);
                    list.Add(item);
                }
            }

            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ParameterListItem[] Sort(ParameterListItem[] items, IComparer<IListItem> comparer)
        {
            if (items.Length <= 0)
            {
                return new ParameterListItem[0];
            }

            List<ParameterListItem> list = new List<ParameterListItem>();

            list.AddRange(items);
            Sort(comparer, list);
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void Sort(IComparer<IListItem> comparer, List<ParameterListItem> list)
        {
            List<ParameterListItem> listL = null;
            List<ParameterListItem> listR = null;
            ParameterListItem item = null;
            int center = 0;
            int index = 0;
            int indexL = 0;
            int indexR = 0;

            switch (list.Count)
            {
                case 0:
                    Debug.Assert(false, "list count is 0");
                    break;

                case 1:
                    break;

                case 2:
                    if (comparer.Compare(list[0], list[1]) > 0)
                    {
                        item = list[0];
                        list[0] = list[1];
                        list[1] = item;
                    }
                    break;

                default:
                    listL = new List<ParameterListItem>();
                    listR = new List<ParameterListItem>();

                    center = list.Count / 2;
                    while (index < center)
                    {
                        listL.Add(list[index]);
                        index++;
                    }

                    while (index < list.Count)
                    {
                        listR.Add(list[index]);
                        index++;
                    }

                    Sort(comparer, listL);
                    Sort(comparer, listR);

                    list.Clear();
                    while (indexL < listL.Count && indexR < listR.Count)
                    {
                        if (comparer.Compare(listL[indexL], listR[indexR]) <= 0)
                        {
                            list.Add(listL[indexL]);
                            indexL++;
                        }
                        else
                        {
                            list.Add(listR[indexR]);
                            indexR++;
                        }
                    }

                    while (indexL < listL.Count)
                    {
                        list.Add(listL[indexL]);
                        indexL++;
                    }

                    while (indexR < listR.Count)
                    {
                        list.Add(listR[indexR]);
                        indexR++;
                    }
                    break;
            }
        }
    }
}
