﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using System.Windows.Forms.VisualStyles;
    using NintendoWare.SoundFoundation.Core;

    public class HeaderCtrl : OperatableControl
    {
        private System.ComponentModel.IContainer components = null;
        private MouseNormalOperator _MouseNormalOperator = null;
        private MouseMoveOperator _MouseMoveOperator = null;
        private MouseLengthMoveOperator _MouseLengthMoveOperator = null;
        private List<ToolStripMenuItem> _ToolStripMenuItems = new List<ToolStripMenuItem>();
        private ToolStripMenuItem[] _InsertContextMenuItems = new ToolStripMenuItem[0];
        private Point _CurrentPoint;
        private int _Position;
        private int _LengthMoveOffset = 8;
        private Dictionary<string, int> _Dictionary;
        private ContextMenuStrip contextMenuStrip;
        private ToolStripMenuItem toolStripMenuItem1;
        private ToolStripSeparator toolStripSeparator;
        private IHeaderSource _Source;

        public event EventHandler OrderChanged;
        public event EventHandler SortOrderChanged;
        public event EventHandler PositionChanged;
        public event EventHandler ItemLengthChanged;
        public event EventHandler ItemVisibleChanged;
        public event CancelEventHandler ContextMenuOpening;

        public class VSplitDoubleClickEventArgs : EventArgs
        {
            public string Name { get; set; }
            public VSplitDoubleClickEventArgs(string name) { Name = name; }
        }
        public delegate void VSplitDoubleClickEventHandler(object sender,
                                                            VSplitDoubleClickEventArgs e);
        public event VSplitDoubleClickEventHandler VSplitDoubleClicked;


        public HeaderCtrl()
        {
            InitializeComponent();
            _MouseNormalOperator = new MouseNormalOperator(this);
            _MouseMoveOperator = new MouseMoveOperator(this);
            _MouseLengthMoveOperator = new MouseLengthMoveOperator(this);
            _Dictionary = new Dictionary<string, int>();
            CurrentOperator = _MouseNormalOperator;
            HorizontalScrollBar.Visible = false;
            VerticalScrollBar.Visible = false;
            SortEnabled = true;
        }
        public HeaderCtrl(IHeaderSource source) : this()
        {
            Source = source;
        }

        public IHeaderSource Source
        {
            get
            {
                return _Source;
            }
            set
            {
                if (_Source != null)
                {
                    _Source.Updated -= UpdateData;
                }
                _Source = value;
                UpdateData();
                if (_Source != null)
                {
                    _Source.Updated += UpdateData;
                }
            }
        }

        public bool SortEnabled
        {
            get; set;
        }

        public int Position
        {
            get
            {
                return _Position;
            }
            set
            {
                _Position = value;

                if (_Position < 0)
                {
                    _Position = 0;
                }
                else
                {
                    int max =
                        (TotalFreeItemsLength -
                          (Width - TotalLeftFixedItemsLength - TotalRightFixedItemsLength));
                    if (0 <= max && max < value)
                    {
                        _Position = max;
                    }
                }
                Invalidate();
                OnPositionChanged(EventArgs.Empty);
            }
        }

        public int TotalFreeItemsLength
        {
            get
            {
                int length = 0;

                List<IHeaderItem> items = GetFreeItems();
                foreach (IHeaderItem item in items)
                {
                    length += item.Length;
                }

                return length;
            }
        }

        public int TotalLeftFixedItemsLength
        {
            get
            {
                List<IHeaderItem> items = GetLeftFixedItems();
                return TotalItemsLength(items);
            }
        }

        public int TotalRightFixedItemsLength
        {
            get
            {
                List<IHeaderItem> items = GetRightFixedItems();
                return TotalItemsLength(items);
            }
        }

        public string[] ItemsName
        {
            get
            {
                if (Source == null) { return new string[0]; }
                return Source.Items.Select(i => i.Name).ToArray();
            }
        }

        public string[] FreeItemsName
        {
            get
            {
                return GetFreeItems().Select(i => i.Name).ToArray();
            }
        }

        public string[] DisplayedFreeItemsName
        {
            get
            {
                List<string> names = new List<string>();
                List<IHeaderItem> freeList = GetFreeItems();

                foreach (IHeaderItem item in freeList)
                {
                    int position = GetPositionByItem(item);
                    if (position + item.Length < TotalLeftFixedItemsLength)
                    {
                        continue;
                    }
                    else if (Width - TotalRightFixedItemsLength < position)
                    {
                        continue;
                    }
                    else
                    {
                        names.Add(item.Name);
                    }
                }

                return names.ToArray();
            }
        }

        public string[] DisplayedLeftFixedItemsName
        {
            get
            {
                List<string> names = new List<string>();
                List<IHeaderItem> leftFixedList = GetLeftFixedItems();

                foreach (IHeaderItem item in leftFixedList)
                {
                    int position = GetPositionByItem(item);
                    if (Width - TotalRightFixedItemsLength < position)
                    {
                        continue;
                    }
                    else
                    {
                        names.Add(item.Name);
                    }
                }

                return names.ToArray();
            }
        }

        public string[] DisplayedRightFixedItemsName
        {
            get
            {
                List<string> names = new List<string>();
                List<IHeaderItem> rightFixedList = GetRightFixedItems();

                foreach (IHeaderItem item in rightFixedList)
                {
                    int position = GetPositionByItem(item);
                    if (position + item.Length < 0)
                    {
                        continue;
                    }
                    else
                    {
                        names.Add(item.Name);
                    }
                }

                return names.ToArray();
            }
        }

        public ToolStripMenuItem[] InsertContextMenuItems
        {
            get
            {
                return _InsertContextMenuItems;
            }
            set
            {
                _InsertContextMenuItems = value;
                if (_InsertContextMenuItems == null)
                {
                    _InsertContextMenuItems = new ToolStripMenuItem[0];
                }
            }
        }

        //

        public bool IsVisibleItemByIndex(int index)
        {
            if (Source == null) { return false; }
            else { return Source.Items[index].Visible; }
        }

        public int GetPositionByName(string name)
        {
            IHeaderItem item = GetItemByName(name);
            Debug.Assert((item != null), "GetLengthByName '" + name + "' doesn't exist.");
            int position = GetPositionByItem(item);

            return position;
        }

        public int GetLengthByName(string name)
        {
            IHeaderItem item = GetItemByName(name);
            Debug.Assert((item != null), "GetLengthByName '" + name + "' doesn't exist.");

            return item.Length;
        }

        public int GetIndexByName(string name)
        {
            int index = 0;

            if (_Dictionary.TryGetValue(name, out index) != true)
            {
                index = -1;
            }

            return index;
        }

        public string GetNameByIndex(int index)
        {
            IHeaderItem item = Source.Items[index];
            return item.Name;
        }

        public int GetDisplayedPosition(string name)
        {
            IHeaderItem item = GetItemByName(name);
            int position = GetPositionByItem(item);

            if (item.Type == HeaderItemType.Free)
            {
                if (position < TotalLeftFixedItemsLength &&
                    TotalLeftFixedItemsLength < position + item.Length)
                {
                    position = TotalLeftFixedItemsLength;
                }
            }
            else if (item.Type == HeaderItemType.RightFixed)
            {
                if (position < 0 && 0 < position + item.Length)
                {
                    position = 0;
                }
            }

            return position;
        }

        public int GetDisplayedLength(string name)
        {
            int length = 0;
            int beginHideLength = 0;
            int endHideLength = 0;
            IHeaderItem item = GetItemByName(name);
            int begin = GetPositionByName(name);
            int end = begin + item.Length;

            switch (item.Type)
            {
                case HeaderItemType.Free:
                    if (begin < TotalLeftFixedItemsLength)
                    {
                        beginHideLength = TotalLeftFixedItemsLength - begin;
                    }
                    if (Width - TotalRightFixedItemsLength <= end)
                    {
                        endHideLength = end - (Width - TotalRightFixedItemsLength);
                    }
                    break;

                case HeaderItemType.LeftFixed:
                    if (Width - TotalRightFixedItemsLength <= end)
                    {
                        endHideLength = end - (Width - TotalRightFixedItemsLength);
                    }
                    break;

                case HeaderItemType.RightFixed:
                    if (begin < 0)
                    {
                        beginHideLength = -begin;
                    }
                    break;
            }

            length = item.Length - beginHideLength - endHideLength;
            if (length < 0)
            {
                length = 0;
            }

            return length;
        }

        public void FitDisplay(string name)
        {
            IHeaderItem item = GetItemByName(name);
            if (item.Type == HeaderItemType.Free)
            {
                int position = GetPositionByFreeItem(item);
                if (position < TotalLeftFixedItemsLength)
                {
                    FitDisplayLeft(item);
                }
                else if (Width - TotalRightFixedItemsLength < position + item.Length)
                {
                    int hideLength = ((position + item.Length) -
                                       (Width - TotalRightFixedItemsLength));

                    if (position - hideLength < TotalLeftFixedItemsLength)
                    {
                        FitDisplayLeft(item);
                    }
                    else
                    {
                        FitDisplayRight(item);
                    }
                }
            }
        }

        private class HeaderCtrlHideWindowItem
        {
            private string text;

            public HeaderCtrlHideWindowItem(IHeaderItem item, string text)
            {
                this.HeaderItem = item;
                this.text = text;
            }

            public IHeaderItem HeaderItem
            {
                get;
            }

            public override string ToString()
            {
                return this.text;
            }
        }

        private void SetHeaderItemsParameter(HeaderCtrlHideWindow hchw)
        {
            List<IHeaderItem> newItems = new List<IHeaderItem>();
            hchw.Items
                .Cast<HeaderCtrlHideWindowItem>()
                .ForEach((item, i) =>
                {
                    item.HeaderItem.Visible = hchw.GetItemChecked(i);
                    newItems.Add(item.HeaderItem);
                });

            List<IHeaderItem> all = new List<IHeaderItem>();

            all.AddRange(GetAllLeftFixedItems());  // 左側固定カラム
            all.AddRange(newItems);                // 普通のカラム
            all.AddRange(GetAllRightFixedItems()); // 右側固定カラム

            Source.Items.Clear();
            Source.Items.AddRange(all);
            UpdateData();

            OnOrderChanged(EventArgs.Empty);
            OnPositionChanged(EventArgs.Empty);
            OnItemLengthChanged(EventArgs.Empty);
            OnItemVisibleChanged(EventArgs.Empty);
        }

        /// <summary>
        ///
        /// </summary>
        public void SetSortState(string name, SortOrder sortOrder)
        {
            IHeaderItem headerItem = GetItemByName(name);
            Source.SortItem = headerItem;
            Source.SortOrder = sortOrder;
            Invalidate();
        }

        /// <summary>
        /// ダブルクリックすると発生します。
        /// </summary>
        protected void OnVSplitDoubleClicked(VSplitDoubleClickEventArgs e)
        {
            if (VSplitDoubleClicked != null)
                VSplitDoubleClicked(this, e);
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            Invalidate();
        }

        protected override void OnPaint(PaintEventArgs pe)
        {
            base.OnPaint(pe);
            if (Source != null)
            {
                PaintHeader(pe.Graphics);
            }
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing && (components != null))
            {
                components.Dispose();
            }
            base.Dispose(disposing);
        }

        //

        private void FitDisplayLeft(IHeaderItem item)
        {
            if (item.Type == HeaderItemType.Free)
            {
                int position = GetPositionByFreeItem(item);
                Position = position + Position - TotalLeftFixedItemsLength;
            }
        }

        private void FitDisplayRight(IHeaderItem item)
        {
            if (item.Type == HeaderItemType.Free)
            {
                int position = GetPositionByFreeItem(item);
                position += Position - TotalLeftFixedItemsLength;
                position += item.Length;
                Position = -((Width - TotalRightFixedItemsLength - TotalLeftFixedItemsLength)
                               - position);
            }
        }

        private void UpdateData()
        {
            UpdateData(this, EventArgs.Empty);
        }
        private void UpdateData(object sender, EventArgs e)
        {
            if (Source != null)
            {
                _Dictionary.Clear();
                int index = 0;
                foreach (IHeaderItem item in Source.Items)
                {
                    _Dictionary.Add(item.Name, index);
                    ++index;
                }
                Invalidate();
            }
        }

        private void UpdateContextMenuStrip()
        {
            _ToolStripMenuItems.Clear();
            contextMenuStrip.Items.Clear();
            contextMenuStrip.Items.Add(toolStripMenuItem1);
            if (0 < InsertContextMenuItems.Length)
            {
                contextMenuStrip.Items.AddRange(InsertContextMenuItems);
            }
            contextMenuStrip.Items.Add(toolStripSeparator);

            List<IHeaderItem> freeItems = GetFreeItems();
            List<IHeaderItem> allFreeItems = GetAllFreeItems();
            foreach (IHeaderItem item in allFreeItems)
            {
                string text = String.Copy(item.Text).Replace("\r\n", " ");
                ToolStripMenuItem menuItem = new ToolStripMenuItem(text);
                menuItem.Checked = (item.Visible == true);
                menuItem.Click += new System.EventHandler(OnMenuItemClick);
                menuItem.Enabled = (1 != freeItems.Count || item.Visible == false);
                contextMenuStrip.Items.Add(menuItem);
                _ToolStripMenuItems.Add(menuItem);
            }
        }

        private void OnMenuItemClick(object sender, EventArgs e)
        {
            List<IHeaderItem> allFreeItems = GetAllFreeItems();
            if (sender is ToolStripMenuItem)
            {
                ToolStripMenuItem menuItem = (sender as ToolStripMenuItem);
                if (_ToolStripMenuItems.Contains(menuItem) == true)
                {
                    IHeaderItem item = allFreeItems[_ToolStripMenuItems.IndexOf(menuItem)];
                    item.Visible = !(item.Visible);
                    OnItemVisibleChanged(EventArgs.Empty);
                    Invalidate();
                }
            }
        }

        private void OnHideWindowToolStripMenuItemClick(object sender, EventArgs e)
        {
            List<IHeaderItem> items = GetAllFreeItems();
            HeaderCtrlHideWindow hchw = new HeaderCtrlHideWindow();

            foreach (var item in items)
            {
                var text = string.Copy(item.Text).Replace("\r\n", " ");
                hchw.Items.Add(new HeaderCtrlHideWindowItem(item, text), item.Visible);
            }

            if (hchw.ShowDialog() == DialogResult.OK)
            {
                this.SetHeaderItemsParameter(hchw);
            }
        }


        private void OnOrderChanged(EventArgs e)
        {
            if (OrderChanged != null)
            {
                OrderChanged(this, e);
            }
        }

        private void OnSortOrderChanged(EventArgs e)
        {
            if (SortEnabled == true)
            {
                if (SortOrderChanged != null)
                {
                    SortOrderChanged(this, e);
                }
            }
        }

        private void OnPositionChanged(EventArgs e)
        {
            if (PositionChanged != null)
            {
                PositionChanged(this, e);
            }
        }

        private void OnItemLengthChanged(EventArgs e)
        {
            if (ItemLengthChanged != null)
            {
                ItemLengthChanged(this, e);
            }
        }

        private void OnItemVisibleChanged(EventArgs e)
        {
            if (ItemVisibleChanged != null)
            {
                ItemVisibleChanged(this, e);
            }
        }

        private void OnContextMenuOpening(object sender, CancelEventArgs e)
        {
            this.toolStripMenuItem1.Enabled = this.GetAllFreeItems().Count() > 0;

            if (ContextMenuOpening != null)
            {
                ContextMenuOpening(this, e);
            }
        }

        //

        private IHeaderItem GetItemByName(string name)
        {
            IHeaderItem targetItem = null;

            if (Source != null)
            {
                IHeaderItemCollection items = Source.Items;
                foreach (IHeaderItem item in items)
                {
                    if (item.Name == name)
                    {
                        targetItem = item;
                        break;
                    }
                }
            }

            return targetItem;
        }

        private int TotalItemsLength(List<IHeaderItem> items)
        {
            int length = 0;

            foreach (IHeaderItem item in items)
            {
                length += item.Length;
            }

            return length;
        }

        private void ChangeOperatorToMove(Point beginPoint)
        {
            CurrentOperator = _MouseMoveOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        private void ChangeOperatorToLengthMove(Point beginPoint)
        {
            CurrentOperator = _MouseLengthMoveOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        private void ChangeOperatorToNormal()
        {
            CurrentOperator = _MouseNormalOperator;
            _MouseNormalOperator.Push = null;
            Invalidate();
        }

        private void MoveItem(IHeaderItem dragItem, IHeaderItem dropItem)
        {
            if (CanMoveItem(dragItem, dropItem) == true)
            {
                List<IHeaderItem> itemList = new List<IHeaderItem>();
                IHeaderItemCollection items = Source.Items;

                if (dropItem == null)
                {
                    IHeaderItem current = null;
                    foreach (IHeaderItem item in items)
                    {
                        if (item.Type == HeaderItemType.LeftFixed)
                        {
                            current = item;
                        }
                        else
                        {
                            break;
                        }
                    }
                    dropItem = current;
                }

                foreach (IHeaderItem item in items)
                {
                    if (item != dragItem)
                    {
                        itemList.Add(item);
                    }
                    if (item == dropItem)
                    {
                        itemList.Add(dragItem);
                    }
                }

                items.Clear();
                items.AddRange(itemList.ToArray());
            }
        }

        private bool CanMoveItem(IHeaderItem drag, IHeaderItem drop)
        {
            bool ret = true;

            if (drag == drop)
            {
                ret = false;
            }
            else
            {
                int dragIndex = Source.Items.IndexOf(drag);

                if (drop == null)
                {
                    if (dragIndex == 0)
                    {
                        ret = false;
                    }
                }
                else
                {
                    int dropIndex = Source.Items.IndexOf(drop);
                    if (dragIndex - 1 == dropIndex)
                    {
                        ret = false;
                    }
                }
            }

            return ret;
        }

        private bool CanDrag(Point pos)
        {
            bool ret = false;

            List<HeaderItemRectangle> freeItems = GetHeaderItemFree();
            foreach (HeaderItemRectangle hi in freeItems)
            {
                if (hi.Rect.Contains(pos))
                {
                    ret = true;
                    break;
                }
            }

            return ret;
        }

        private bool CanDrop(Point pos)
        {
            bool ret = false;
            IHeaderItem item = null;
            HeaderItemRectangle current = null;

            item = GetHeaderItemByPoint(pos);
            if (item != null && item.Type == HeaderItemType.LeftFixed)
            {
                ret = true;
            }
            else if (item != null && item.Type == HeaderItemType.RightFixed)
            {
                ret = true;
            }
            else
            {
                List<HeaderItemRectangle> freeItems = GetHeaderItemFree();

                foreach (HeaderItemRectangle hi in freeItems)
                {
                    if (hi.Rect.Contains(pos))
                    {
                        item = hi.Item;
                        current = hi;
                        break;
                    }
                }

                if (item != null)
                {
                    if (pos.X <= current.Rect.X + (current.Rect.Width / 2))
                    {
                        ret = true;
                    }
                    else
                    {
                        Point p = new Point(current.Rect.X + current.Rect.Width, current.Rect.Y);
                        IHeaderItem im = GetHeaderItemByPoint(p);
                        if (im == null)
                        {
                            ret = true;
                        }
                        else if (im.Type == HeaderItemType.Free)
                        {
                            ret = true;
                        }
                    }
                }
            }

            return ret;
        }

        private void PaintHeader(Graphics gc)
        {
            List<IHeaderItem> items;
            int position;

            items = GetFreeItems();
            position = TotalLeftFixedItemsLength;
            foreach (IHeaderItem item in items)
            {
                PaintItem(gc, item, position - Position);
                position += item.Length;
            }

            items = GetLeftFixedItems();
            position = 0;
            foreach (IHeaderItem item in items)
            {
                PaintItem(gc, item, position);
                position += item.Length;
            }

            items = GetRightFixedItems();
            items.Reverse();
            position = Width;
            foreach (IHeaderItem item in items)
            {
                position -= item.Length;
                PaintItem(gc, item, position);
            }

            if (CurrentOperator == _MouseMoveOperator)
            {
                int beforeLength = 0;
                int length = 0;
                IHeaderItem dropItem;

                if (CanDrop(_CurrentPoint))
                {
                    dropItem = GetDropHeaderItemByPoint(_CurrentPoint);

                    if (dropItem != null)
                    {
                        length = dropItem.Length;
                        beforeLength = GetPositionByFreeItem(dropItem);
                    }
                    else
                    {
                        length = 0;
                        beforeLength = -Position + TotalLeftFixedItemsLength;
                    }

                    if (length + beforeLength < TotalLeftFixedItemsLength)
                    {
                        length = 0;
                        beforeLength = TotalLeftFixedItemsLength;
                    }
                    else if (Width - TotalRightFixedItemsLength <= beforeLength + length)
                    {
                        length = 0;
                        beforeLength = Width - TotalRightFixedItemsLength;
                    }

                    if (Contains(_CurrentPoint) == true)
                    {
                        int x = beforeLength + length;
                        gc.DrawLine(SystemPens.Highlight, x - 1, 0, x - 1, Height);
                        gc.DrawLine(SystemPens.Highlight, x, 0, x, Height);
                    }
                }
            }
        }

        private void PaintItem(Graphics g, IHeaderItem item, int position)
        {
            if (item.Length == 0) return;

            const int SortButtonSize1_2 = 7; // ソートボタンの幅の2分の1;
            const int SortButtonMargin = 4;

            ButtonState state = ButtonState.Normal;
            SortOrder sortOrder = Source.SortOrder;

            if ((CurrentOperator == _MouseNormalOperator && _MouseNormalOperator.Push == item) ||
               (CurrentOperator == _MouseMoveOperator && _MouseMoveOperator.DragItem == item))
            {
                state = ButtonState.Pushed;
            }

            Rectangle bounds = new Rectangle(position, 0, item.Length, Height);
            if (VisualStyleRenderer.IsSupported)
            {
                PaintVisualStyleItem(g, bounds, state);
            }
            else
            {
                ControlPaint.DrawButton(g, bounds, state);
            }

            //
            int SortButtonFullSize = SortButtonSize1_2 * 2;
            int x = position;
            int y = Height - SortButtonFullSize;
            int width = item.Length;
            int height = Height;

            StringFormat format = new StringFormat(StringFormatFlags.NoWrap);
            format.Alignment = StringAlignment.Near;
            format.Trimming = StringTrimming.EllipsisCharacter;
            format.LineAlignment = StringAlignment.Far;

            //
            bool sortEnabled = (SortEnabled == true &&
                                 item == Source.SortItem && sortOrder != SortOrder.None);
            int textWidth = (int)g.MeasureString(item.Text, Font, width, format).Width;
            int textDrawWidth = width;
            int sortButtonWidth = 0;

            if (sortEnabled == true)
            {
                sortButtonWidth = SortButtonFullSize + SortButtonMargin * 2;
                textDrawWidth -= sortButtonWidth;
            }

            // ボタンを描画するスペースが存在しないのか？
            if (width <= sortButtonWidth)
            {
                // 全ての描画スペースをテキストに割り当てます。
                textDrawWidth = width;
            }
            else if (textDrawWidth > width)
            {
                // ボタンの描画スペースを優先にして、テキスト描画スペースを
                // 描画スペースの大きさにします。
                textDrawWidth = width - sortButtonWidth;
            }

            switch (item.Alignment)
            {
                case HorizontalAlignment.Left:
                    {
                        int space = textDrawWidth - textWidth;
                        if (space < 0)
                        {
                            textWidth += space;
                        }
                    }
                    break;

                case HorizontalAlignment.Center:
                    break;

                case HorizontalAlignment.Right:
                    {
                        int space = textDrawWidth - textWidth;
                        x = x + space;
                        if (x < position)
                        {
                            x = position;
                            textWidth += space;
                        }
                    }
                    break;
            }

            //
            if (sortEnabled == true)
            {
                SortOrderMark(g, x + textWidth, y, sortOrder);
            }

            //
            g.DrawString(item.Text, Font, SystemBrushes.WindowText,
                          new RectangleF(x + 1, 1, textDrawWidth - 2, height - 2), format);
            return;
        }

        /// <summary>
        ///
        /// </summary>
        private void SortOrderMark(Graphics gc, int x, int y, SortOrder sortOrder)
        {
            Point[] points = null;
            switch (sortOrder)
            {
                case SortOrder.Ascending:
                    points = new Point[] {
                        new Point(  0 + x, 7 + y),
                        new Point( 12 + x, 7 + y),
                        new Point(  6 + x, 0 + y),
                        new Point(  0 + x, 7 + y) };
                    break;

                case SortOrder.Descending:
                    points = new Point[] {
                        new Point(  1 + x, 1 + y),
                        new Point( 12 + x, 1 + y),
                        new Point(  6 + x, 7 + y),
                        new Point(  1 + x, 1 + y) };
                    break;
            }
            gc.FillPolygon(new SolidBrush(SystemColors.ControlDark), points);
        }

        /// <summary>
        ///
        /// </summary>
        private void PaintVisualStyleItem(Graphics g, Rectangle r, ButtonState state)
        {
            VisualStyleRenderer renderer = null;
            if (state == ButtonState.Pushed)
            {
                renderer = new VisualStyleRenderer(VisualStyleElement.Header.Item.Pressed);
            }
            else
            {
                renderer = new VisualStyleRenderer(VisualStyleElement.Header.Item.Normal);
            }
            renderer.DrawBackground(g, r);
        }

        //

        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(HeaderCtrl));
            this.contextMenuStrip = new System.Windows.Forms.ContextMenuStrip(this.components);
            this.toolStripMenuItem1 = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator = new System.Windows.Forms.ToolStripSeparator();
            this.contextMenuStrip.SuspendLayout();
            this.SuspendLayout();
            //
            // contextMenuStrip
            //
            this.contextMenuStrip.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.toolStripMenuItem1,
            this.toolStripSeparator});
            this.contextMenuStrip.Name = "contextMenuStrip";
            this.contextMenuStrip.Opening += OnContextMenuOpening;
            resources.ApplyResources(this.contextMenuStrip, "contextMenuStrip");
            //
            // toolStripMenuItem1
            //
            this.toolStripMenuItem1.Name = "toolStripMenuItem1";
            resources.ApplyResources(this.toolStripMenuItem1, "toolStripMenuItem1");
            this.toolStripMenuItem1.Click += new System.EventHandler(this.OnHideWindowToolStripMenuItemClick);
            //
            // toolStripSeparator
            //
            this.toolStripSeparator.Name = "toolStripSeparator";
            resources.ApplyResources(this.toolStripSeparator, "toolStripSeparator");
            //
            // HeaderCtrl
            //
            this.Name = "HeaderCtrl";
            this.contextMenuStrip.ResumeLayout(false);
            this.ResumeLayout(false);

        }


        // Mouse


        private class MouseNormalOperator : OperatableControlOperator
        {
            private Point _BeginPoint;
            private IHeaderItem _ClickItem;

            public IHeaderItem Push { get; set; }

            public HeaderCtrl OwnerCtrl
            {
                get
                {
                    return (HeaderCtrl)OwnerControl;
                }
            }

            public MouseNormalOperator(OperatableControl headerCtrl) : base(headerCtrl)
            {
            }

            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                Point point = new Point(e.X, e.Y);
                _ClickItem = OwnerCtrl.GetHeaderItemByPoint(point);

                if (OwnerCtrl.Contains(point) && _ClickItem != null)
                {
                    if (e.Button == MouseButtons.Left)
                    {
                        if (IsLengthMove(e.X, e.Y))
                        {
                            OwnerCtrl.ChangeOperatorToLengthMove(point);
                            _BeginPoint = new Point(-1, -1);
                            Push = null;
                        }
                        else
                        {
                            this.Initialize(point);
                            Push = _ClickItem;
                        }
                    }
                }

                OwnerCtrl.Invalidate();
                return true;
            }

            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                if (e.Button == MouseButtons.Left)
                {
                    if (IsDrag(e.X, e.Y) == true)
                    {
                        if (OwnerCtrl.CanDrag(_BeginPoint))
                        {
                            OwnerCtrl.ChangeOperatorToMove(_BeginPoint);
                            _BeginPoint = new Point(-1, -1);
                            Push = null;
                        }
                    }
                }
                else if (IsLengthMove(e.X, e.Y) == true)
                {
                    OwnerCtrl.Cursor = Cursors.VSplit;
                    Push = null;
                }
                else
                {
                    OwnerCtrl.Cursor = Cursors.Default;
                    Push = null;
                }

                return true;
            }

            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                Point point = new Point(e.X, e.Y);

                switch (e.Button)
                {
                    case MouseButtons.Left:
                        IHeaderItem item = OwnerCtrl.GetHeaderItemByPoint(point);
                        if (item != null && item == _ClickItem)
                        {
                            OwnerCtrl.SetSortItem(item);
                            OwnerCtrl.Invalidate();
                            OwnerCtrl.OnSortOrderChanged(EventArgs.Empty);
                        }
                        break;

                    case MouseButtons.Right:
                        OwnerCtrl.UpdateContextMenuStrip();
                        OwnerCtrl.contextMenuStrip.Show(OwnerCtrl, new Point(e.X, e.Y));
                        break;
                }

                _BeginPoint = new Point(-1, -1);
                Push = null;
                OwnerCtrl.Invalidate();

                return true;
            }

            protected internal override void Initialize(Point beginPoint)
            {
                _BeginPoint = beginPoint;
            }

            private bool IsDrag(int x, int y)
            {
                bool ret = false;

                IHeaderItem item = OwnerCtrl.GetHeaderItemByPoint(new Point(x, y));
                if (item != null && item.Type == HeaderItemType.Free)
                {
                    int xdif = _BeginPoint.X - x;
                    int ydif = _BeginPoint.Y - y;
                    int width = SystemInformation.DragSize.Width;
                    int height = SystemInformation.DragSize.Height;

                    ret = (Math.Abs(xdif) > Math.Abs(width) ||
                            Math.Abs(ydif) > Math.Abs(height));
                }

                return ret;
            }

            private bool IsLengthMove(int x, int y)
            {
                IHeaderItem item = OwnerCtrl.GetLengthMoveItemByPoint(new Point(x, y));

                return item != null;
            }
        }


        private class MouseMoveOperator : OperatableControlOperator
        {
            private int _MovePotisionLength = 30;
            private Point _BeginPoint;
            private DateTime _BeginDateTime;
            private IHeaderItem _DragItem;

            public HeaderCtrl OwnerCtrl
            {
                get
                {
                    return (HeaderCtrl)OwnerControl;
                }
            }

            public MouseMoveOperator(OperatableControl headerCtrl) : base(headerCtrl)
            {
            }

            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                Point point = new Point(e.X, e.Y);

                OwnerCtrl._CurrentPoint = point;
                OwnerCtrl.Invalidate();

                return true;
            }

            public IHeaderItem DragItem
            {
                get { return _DragItem; }
            }

            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                Point point = new Point(e.X, e.Y);

                OwnerCtrl.EndEventLoop();

                if (OwnerCtrl.Contains(point) && OwnerCtrl.CanDrop(point))
                {
                    IHeaderItem dropItem = OwnerCtrl.GetDropHeaderItemByPoint(point);
                    OwnerCtrl.MoveItem(_DragItem, dropItem);
                    _DragItem = null;
                    OwnerCtrl.Invalidate();
                    OwnerCtrl.OnOrderChanged(EventArgs.Empty);
                }

                _DragItem = null;
                OwnerCtrl.ChangeOperatorToNormal();
                OwnerCtrl.Invalidate();

                return true;
            }

            protected internal override void Initialize(Point beginPoint)
            {
                _BeginPoint = beginPoint;
                _DragItem = OwnerCtrl.GetHeaderItemByPoint(_BeginPoint);
                _BeginDateTime = DateTime.Now;
                OwnerCtrl.BeginEventLoop();
            }

            protected internal override void OnEventLoop(MouseEventArgs e)
            {
                Point point = new Point(e.X, e.Y);
                DateTime dateTime = DateTime.Now;
                if (dateTime.Ticks - _BeginDateTime.Ticks >= 1600000)
                {
                    _BeginDateTime = dateTime;

                    IHeaderItem item = OwnerCtrl.GetHeaderItemByPoint(point);
                    if (item == null || item.Type != HeaderItemType.Free)
                    {
                        if (_BeginPoint.X > e.X)
                        {
                            OwnerCtrl.Position -= _MovePotisionLength;
                        }
                        else
                        {
                            OwnerCtrl.Position += _MovePotisionLength;
                        }
                    }
                }
            }
        }


        private class MouseLengthMoveOperator : OperatableControlOperator
        {
            private Point _BeginPoint;
            private IHeaderItem _TargetItem;
            private int _Length;

            public HeaderCtrl OwnerCtrl
            {
                get
                {
                    return (HeaderCtrl)OwnerControl;
                }
            }

            public MouseLengthMoveOperator(OperatableControl headerCtrl) : base(headerCtrl)
            {
            }

            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                int length;

                if (_TargetItem.Type == HeaderItemType.RightFixed)
                {
                    length = _TargetItem.Length - (e.X - _BeginPoint.X);
                }
                else
                {
                    length = _TargetItem.Length + (e.X - _BeginPoint.X);
                }
                _BeginPoint.X = e.X;
                _BeginPoint.Y = e.Y;
                if (_TargetItem.MaximumLength < length)
                {
                    length = _TargetItem.MaximumLength;
                }
                else if (length < _TargetItem.MinimumLength)
                {
                    length = _TargetItem.MinimumLength;
                }
                _TargetItem.Length = length;

                OwnerCtrl.Invalidate();

                return true;
            }

            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                OwnerCtrl.Cursor = Cursors.Default;
                OwnerCtrl.ChangeOperatorToNormal();
                OwnerCtrl.Invalidate();
                OwnerCtrl.OnItemLengthChanged(EventArgs.Empty);

                return true;
            }

            protected internal override void OnDoubleClick(MouseEventArgs e)
            {
                Point point = new Point(e.X, e.Y);
                IHeaderItem item = OwnerCtrl.GetLengthMoveItemByPoint(point);

                if (OwnerCtrl.Contains(point) && item != null)
                {
                    OwnerCtrl.OnVSplitDoubleClicked(new VSplitDoubleClickEventArgs(item.Name));
                }
            }

            protected internal override void Initialize(Point beginPoint)
            {
                _BeginPoint = beginPoint;
                _TargetItem = OwnerCtrl.GetLengthMoveItemByPoint(_BeginPoint);
                _Length = _TargetItem.Length;
                OwnerCtrl.Cursor = Cursors.VSplit;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool Contains(Point pos)
        {
            Rectangle r = new Rectangle(0, 0, Width, Height);

            return r.Contains(pos);
        }

        /// <summary>
        ///
        /// </summary>
        private class HeaderItemRectangle
        {
            public IHeaderItem Item
            {
                get; set;
            }
            public Rectangle Rect
            {
                get; set;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private IHeaderItem GetHeaderItemByPoint(Point pos)
        {
            IHeaderItem item = null;
            List<HeaderItemRectangle> fixItems = GetHeaderItemFix();

            foreach (HeaderItemRectangle hi in fixItems)
            {
                if (hi.Rect.Contains(pos))
                {
                    item = hi.Item;
                    break;
                }
            }

            if (item == null)
            {
                List<HeaderItemRectangle> freeItems = GetHeaderItemFree();

                foreach (HeaderItemRectangle hi in freeItems)
                {
                    if (hi.Rect.Contains(pos))
                    {
                        item = hi.Item;
                        break;
                    }
                }
            }

            return item;
        }

        /// <summary>
        ///
        /// </summary>
        private IHeaderItem GetDropHeaderItemByPoint(Point pos)
        {
            IHeaderItem item = null;
            HeaderItemRectangle current = null;

            item = GetHeaderItemByPoint(pos);
            if (item.Type == HeaderItemType.LeftFixed)
            {
                item = null;
            }
            else if (item.Type == HeaderItemType.RightFixed)
            {
                string[] names = DisplayedFreeItemsName;
                if (names.Length > 0)
                {
                    item = GetItemByName(names[names.Length - 1]);
                }
            }
            else
            {
                List<HeaderItemRectangle> freeItems = GetHeaderItemFree();

                foreach (HeaderItemRectangle hi in freeItems)
                {
                    if (hi.Rect.Contains(pos))
                    {
                        item = hi.Item;
                        current = hi;
                        break;
                    }
                }

                if (item != null)
                {
                    if (pos.X <= current.Rect.X + (current.Rect.Width / 2))
                    {
                        int index = freeItems.IndexOf(current);
                        if (index == 0)
                        {
                            item = null;
                        }
                        else
                        {
                            item = freeItems[index - 1].Item;
                        }
                    }
                    else
                    {
                        Point p = new Point(current.Rect.X + current.Rect.Width, current.Rect.Y);
                        IHeaderItem im = GetHeaderItemByPoint(p);
                        if (im != null && im.Type != HeaderItemType.Free)
                        {
                            item = null;
                        }
                    }
                }
            }

            return item;
        }

        private IHeaderItem GetLengthMoveItemByPoint(Point point)
        {
            Point pos;
            IHeaderItem item;

            // Right
            pos = new Point(point.X + _LengthMoveOffset, point.Y);
            item = GetHeaderItemByPoint(pos);
            if (item != null && item.Type == HeaderItemType.RightFixed)
            {
                int position = GetPositionByItem(item);
                if (position + _LengthMoveOffset < point.X)
                {
                    item = null;
                }
                else
                {
                    goto Return;
                }
            }

            // Left & Free
            pos = new Point(point.X, point.Y);
            item = GetHeaderItemByPoint(pos);
            if (item != null)
            {
                int position = GetPositionByItem(item);
                if (position + item.Length - _LengthMoveOffset <= point.X &&
                     point.X <= position + item.Length)
                {
                    if (item.Type == HeaderItemType.RightFixed)
                    {
                        item = null;
                    }
                }
                else if (position <= point.X &&
                          point.X <= position + _LengthMoveOffset)
                {
                    item = prevItem(item);
                    if (item != null && item.Type == HeaderItemType.RightFixed)
                    {
                        item = null;
                    }
                }
                else
                {
                    item = null;
                }
            }

            Return:
            return item;
        }

        private IHeaderItem prevItem(IHeaderItem next)
        {
            IHeaderItem prev = null;

            if (Source != null)
            {
                foreach (IHeaderItem item in Source.Items)
                {
                    if (item == next)
                    {
                        break;
                    }
                    if (item.Visible == true)
                    {
                        prev = item;
                    }
                }
            }

            return prev;
        }

        private List<IHeaderItem> GetAllFreeItems()
        {
            return GetItemsByType(HeaderItemType.Free, true);
        }

        private List<IHeaderItem> GetAllLeftFixedItems()
        {
            return GetItemsByType(HeaderItemType.LeftFixed, true);
        }

        private List<IHeaderItem> GetAllRightFixedItems()
        {
            return GetItemsByType(HeaderItemType.RightFixed, true);
        }


        private List<IHeaderItem> GetFreeItems()
        {
            return GetItemsByType(HeaderItemType.Free, false);
        }

        private List<IHeaderItem> GetLeftFixedItems()
        {
            return GetItemsByType(HeaderItemType.LeftFixed, false);
        }

        private List<IHeaderItem> GetRightFixedItems()
        {
            return GetItemsByType(HeaderItemType.RightFixed, false);
        }

        private List<IHeaderItem> GetItemsByType(HeaderItemType type, bool all)
        {
            List<IHeaderItem> itemList = new List<IHeaderItem>();

            if (Source != null)
            {
                IHeaderItemCollection items = Source.Items;

                foreach (IHeaderItem item in items)
                {
                    if (item.Type == type)
                    {
                        if (all == true)
                        {
                            itemList.Add(item);
                        }
                        else if (item.Visible == true && 0 < item.Length)
                        {
                            itemList.Add(item);
                        }
                    }
                }
            }

            return itemList;
        }

        private List<HeaderItemRectangle> GetHeaderItemFree()
        {
            List<HeaderItemRectangle> itemList = new List<HeaderItemRectangle>();

            List<IHeaderItem> items = GetFreeItems();

            int position = -Position + TotalLeftFixedItemsLength;
            foreach (IHeaderItem item in items)
            {
                HeaderItemRectangle hi = new HeaderItemRectangle();
                hi.Rect = new Rectangle(position, 0, item.Length, Height);
                hi.Item = item;
                itemList.Add(hi);
                position += item.Length;
            }

            return itemList;
        }

        private List<HeaderItemRectangle> GetHeaderItemFix()
        {
            List<HeaderItemRectangle> itemList = new List<HeaderItemRectangle>();
            List<IHeaderItem> items;
            int position;

            items = GetRightFixedItems();
            position = Width;
            int i;
            for (i = items.Count - 1; i >= 0; i--)
            {
                IHeaderItem item = items[i];
                HeaderItemRectangle hi = new HeaderItemRectangle();
                hi.Rect = new Rectangle(position - item.Length, 0, item.Length, Height);
                hi.Item = item;
                itemList.Add(hi);
                position -= item.Length;
            }

            items = GetLeftFixedItems();
            position = 0;
            foreach (IHeaderItem item in items)
            {
                HeaderItemRectangle hi = new HeaderItemRectangle();
                hi.Rect = new Rectangle(position, 0, item.Length, Height);
                hi.Item = item;
                itemList.Add(hi);
                position += item.Length;
            }

            return itemList;
        }

        private int GetPositionByFreeItem(IHeaderItem targetItem)
        {
            int position = 0;

            position = -Position + TotalLeftFixedItemsLength;
            List<IHeaderItem> items = GetFreeItems();
            foreach (IHeaderItem item in items)
            {
                if (item == targetItem)
                {
                    break;
                }
                position += item.Length;
            }

            return position;
        }

        private int GetPositionByItem(IHeaderItem targetItem)
        {
            int position = 0;

            position = -Position + TotalLeftFixedItemsLength;
            List<IHeaderItem> items = GetFreeItems();
            foreach (IHeaderItem item in items)
            {
                if (item == targetItem)
                {
                    goto Return;
                }
                position += item.Length;
            }

            position = 0;
            items = GetLeftFixedItems();
            foreach (IHeaderItem item in items)
            {
                if (item == targetItem)
                {
                    goto Return;
                }
                position += item.Length;
            }

            position = Width;
            items = GetRightFixedItems();
            items.Reverse();
            foreach (IHeaderItem item in items)
            {
                position -= item.Length;
                if (item == targetItem)
                {
                    goto Return;
                }
            }

            Return:
            return position;
        }

        //

        private void SetSortItem(IHeaderItem item)
        {
            if (SortEnabled == true)
            {

                Debug.Assert((item.SortOrders != null && item.SortOrders.Count != 0),
                             "SetSortItem '" + item.Name + "' SortOrders doesn't exist.");
                SortOrder[] sortOrder = item.SortOrders.ToArray();

                if (Source != null)
                {
                    if (item == Source.SortItem)
                    {
                        for (int i = 0; i < sortOrder.Length; i++)
                        {
                            if (Source.SortOrder == sortOrder[i])
                            {
                                if (i == sortOrder.Length - 1)
                                {
                                    Source.SortOrder = sortOrder[0];
                                }
                                else
                                {
                                    Source.SortOrder = sortOrder[i + 1];
                                }
                                break;
                            }
                        }
                    }
                    else
                    {
                        Source.SortItem = item;
                        Source.SortOrder = sortOrder[0];
                    }
                }
            }
        }
    }
}
