﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Data;
    using System.Text;
    using System.Windows.Forms;

    public partial class DockingFormContainer : NPanel
    {
        #region ** 固定値

        public enum DockAlignment
        {
            Top = 0,
            Bottom,
            Left,
            Right
        }

        #endregion

        public event EventHandler SelectedTabChanged;

        #region ** フィールド

        private DockingFormContainerTabControl _tabControl = null;

        private DockAlignment _alignment = DockAlignment.Top;

        private DockingPageCollection _pages;
        private DockingFormCollection _forms;

        private IDockingPage _menuTargetPage = null;

        #endregion

        public DockingFormContainer()
        {
            InitializeComponent();

            // タブコントロールを初期化する
            _tabControl = CreateTabControl();

            _tabControl.ContextMenuStrip = _dockingMenu;
            _tabControl.ItemMaxWidth = 200;
            _tabControl.Name = "_tabControl";
            _tabControl.Dock = DockStyle.Fill;
            _tabControl.ShowTabCloseButton = false;
            _tabControl.TabControlStyle = NTabControlStyle.NormalMinimizableBottomTab;
            _tabControl.CloseButtonClick += new EventHandler(OnTabCloseButtonClick);
            _tabControl.Swapped += new NTabControlSwapEventHandler(OnTabPageSwapped);
            _tabControl.Closing += new NTabControlCancelEventHandler(OnTabPageClosing);
            _tabControl.Closed += new NTabControlEventHandler(OnTabPageClosed);
            _tabControl.SelectedTabChanged += new EventHandler(OnSelectedTabChanged);
            _tabControl.Restored += new EventHandler(OnRestoreButtonClick);
            _tabControl.TabDoubleClick += new NTabControlEventHandler(OnTabDoubleClick);
            _tabControl.ContextMenuShown +=
                new NTabControlCancelEventHandler(OnTabContextMenuShown);

            Controls.Add(_tabControl);
            _tabControl.BringToFront();

            _pages = new DockingPageCollection(this);
            _forms = new DockingFormCollection();

            _pages.ItemInserted += OnPageInserted;
            _pages.ItemRemoved += OnPageRemoved;
            _pages.ItemClearing += OnPageClearing;

            _forms.ItemContextMenuShown += OnDockingFormMenuShown;
            _forms.ItemCaptionDoubleClick += OnDockingFormCaptionDoubleClick;
            _forms.ItemClosed += OnDockingFormClosed;
        }

        #region ** プロパティ

        public DockAlignment Alignment
        {
            get { return _alignment; }
            set
            {
                if (value == _alignment) { return; }

                SetAlignment(value);
            }
        }

        [Browsable(false)]
        public DockingPageCollection Pages
        {
            get { return _pages; }
        }

        [Browsable(false)]
        public bool IsMinimized
        {
            get { return _tabControl.Minimized; }
            set
            {
                if (value)
                {
                    Minimize();
                }
                else
                {
                    Restore();
                }
            }
        }

        [Browsable(false)]
        public int DockedCount
        {
            get { return _tabControl.TabPages.Count; }
        }

        [Browsable(false)]
        public IDockingPage SelectedDockedPage
        {
            get { return _tabControl.SelectedTab as IDockingPage; }
            set
            {
                if (null == value) { return; }
                if (!_tabControl.TabPages.Contains(value)) { return; }

                _tabControl.SelectedTab = value;
            }
        }

        #endregion

        #region ** イベント

        public event EventHandler Minimizing;
        public event EventHandler Minimized;
        public event EventHandler Restored;

        public event DockingPageEventHandler PageInserted;
        public event DockingPageEventHandler PageRemoved;

        public event DockingPageEventHandler PageDockStateChanged;
        public event DockingPageEventHandler PageDockVisibleChanged;

        #endregion

        #region ** イベントハンドラ

        #region ** 初期化イベント

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);

            ShowFloatingForms();
        }

        #endregion

        #region ** コントロールイベント

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            if (Visible && null != SelectedDockedPage)
            {
                _tabControl.EnsureVisible(SelectedDockedPage);
            }
        }

        #endregion

        #region ** ページ操作

        private void OnPageInserted(object sender, DockingPageEventArgs e)
        {
            if (null != e.Page.DockContainer)
            {
                throw new Exception("DockingPage has already DockingFormContainer.");
            }

            e.Page.DockStateChanged += OnPageDockStateChanged;
            e.Page.DockVisibleChanged += OnPageDockVisibleChanged;
            e.Page.DockContainer = this;

            UpdatePage(e.Page);

            if (null != PageInserted)
            {
                PageInserted(this, e);
            }
        }

        private void OnPageRemoved(object sender, DockingPageEventArgs e)
        {
            e.Page.DockStateChanged -= OnPageDockStateChanged;
            e.Page.DockVisibleChanged -= OnPageDockVisibleChanged;
            e.Page.DockContainer = null;

            if (e.Page.DockVisible)
            {

                switch (e.Page.DockState)
                {
                    case DockingPageState.Docked:
                        _tabControl.TabPages.Remove(e.Page);
                        break;

                    case DockingPageState.Floating:
                        DockingForm form = _forms[e.Page];
                        _forms.Remove(e.Page);
                        form.Close();
                        break;
                }

            }

            if (null != PageRemoved)
            {
                PageRemoved(this, e);
            }
        }

        private void OnPageClearing(object sender, EventArgs e)
        {
            foreach (IDockingPage page in _pages)
            {
                page.DockStateChanged -= OnPageDockStateChanged;
                page.DockVisibleChanged -= OnPageDockVisibleChanged;
            }

            _tabControl.TabPages.Clear();
            _forms.Clear();
        }

        #endregion

        #region ** タイトル操作イベント

        private void OnTitleMouseDown(object sender, MouseEventArgs e)
        {
            if (null == SelectedDockedPage) { return; }
            SelectedDockedPage.Select();
        }

        private void OnTitleDoubleClick(object sender, EventArgs e)
        {
            if (null == SelectedDockedPage)
            {
                Debug.Assert(false);
                return;
            }

            if (SelectedDockedPage.IsEnableDockingState(DockingPageState.Floating))
            {
                SelectedDockedPage.DockState = DockingPageState.Floating;
            }
        }

        private void OnTitleMouseClick(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {

                _menuTargetPage = SelectedDockedPage;

                _menuItemDocking.Checked = true;
                _menuItemFloating.Checked = false;

                _menuItemDocking.Enabled =
                    SelectedDockedPage.IsEnableDockingState(DockingPageState.Docked);
                _menuItemFloating.Enabled =
                    SelectedDockedPage.IsEnableDockingState(DockingPageState.Floating);
                _menuItemHide.Enabled = SelectedDockedPage.Closable;

                _dockingMenu.Show(sender as Control, e.Location);

            }
        }

        private void OnMinimizeButtonMouseClick(object sender, MouseEventArgs e)
        {
            if (MouseButtons.Left != e.Button) { return; }
            OnMinimizing();
            Minimize();
        }

        private void OnCloseButtonMouseClick(object sender, MouseEventArgs e)
        {
            if (MouseButtons.Left != e.Button) { return; }
            SelectedDockedPage.DockVisible = false;
        }

        #endregion

        #region ** ドッキングページイベント

        private void OnPageDockStateChanged(object sender, EventArgs e)
        {
            IDockingPage page = sender as IDockingPage;
            UpdatePage(page);

            if (null != PageDockStateChanged)
            {
                PageDockStateChanged(this, new DockingPageEventArgs(page, _pages.IndexOf(page)));
            }
        }

        private void OnPageDockVisibleChanged(object sender, EventArgs e)
        {
            IDockingPage page = sender as IDockingPage;
            UpdatePage(page);

            if (null != PageDockVisibleChanged)
            {
                PageDockVisibleChanged(this, new DockingPageEventArgs(page, _pages.IndexOf(page)));
            }
        }

        #endregion

        #region ** タブイベント

        private void OnSelectedTabChanged(object sender, EventArgs e)
        {
            if (NTabControl.InvalidIndex == _tabControl.SelectedIndex)
            {
                _title.Text = string.Empty;
                _title.CloseButtonVisible = false;
                return;
            }

            _title.Text = _tabControl.SelectedTab.Text;
            //_title.CloseButtonVisible = ( _tabControl.SelectedTab as IDockingPage ).Closable;

            if (this.SelectedTabChanged != null)
            {
                this.SelectedTabChanged(sender, e);
            }
        }

        private void OnTabPageSwapped(object sender, NTabControlSwapEventArgs e)
        {
            _pages.Swap(e.TabPage as IDockingPage, e.TabPage2 as IDockingPage);
        }

        private void OnTabPageClosing(object sender, NTabControlCancelEventArgs e)
        {
            IDockingPage page = e.TabPage as IDockingPage;

            if (!page.Closable)
            {
                e.Cancel = true;
            }
        }

        private void OnTabPageClosed(object sender, NTabControlEventArgs e)
        {
            IDockingPage page = e.TabPage as IDockingPage;

            if (DockingPageState.Docked == page.DockState)
            {
                page.DockVisible = false;
            }
        }

        private void OnTabDoubleClick(object sender, NTabControlEventArgs e)
        {
            if (IsMinimized)
            {
                Restore();
            }
            else
            {
                (e.TabPage as IDockingPage).DockState = DockingPageState.Floating;
            }
        }

        private void OnRestoreButtonClick(object sender, EventArgs e)
        {
            Restore();
        }

        private void OnTabCloseButtonClick(object sender, EventArgs e)
        {
            SelectedDockedPage.DockVisible = false;
        }

        #endregion

        #region ** フォームイベント

        private void OnDockingFormCaptionDoubleClick(object sender, DockingFormEventArgs e)
        {
            if (null == e.Form.Page) { return; }

            if (e.Form.Page.IsEnableDockingState(DockingPageState.Docked))
            {
                e.Form.Page.DockState = DockingPageState.Docked;
            }
        }

        private void OnDockingFormClosed(object sender, DockingFormEventArgs e)
        {
            if (null == e.Form.Page) { return; }
            e.Form.Page.DockVisible = false;
        }

        #endregion

        #region ** メニューイベント

        /// <summary>
        /// タブメニューが表示される前に発生します。
        /// </summary>
        private void OnTabContextMenuShown(object sender, NTabControlCancelEventArgs e)
        {
            _menuTargetPage = e.TabPage as IDockingPage;

            _menuItemDocking.Checked = true;
            _menuItemFloating.Checked = false;

            _menuItemDocking.Enabled =
                _menuTargetPage.IsEnableDockingState(DockingPageState.Docked);
            _menuItemFloating.Enabled =
                _menuTargetPage.IsEnableDockingState(DockingPageState.Floating);
            _menuItemHide.Enabled = _menuTargetPage.Closable;
        }

        private void OnDockingFormMenuShown(object sender, DockingFormEventArgs e)
        {
            _menuTargetPage = e.Form.Page;

            _menuItemDocking.Checked = false;
            _menuItemFloating.Checked = true;

            _menuItemDocking.Enabled =
                _menuTargetPage.IsEnableDockingState(DockingPageState.Docked);
            _menuItemFloating.Enabled =
                _menuTargetPage.IsEnableDockingState(DockingPageState.Floating);
            _menuItemHide.Enabled = _menuTargetPage.Closable;
        }

        private void OnDockingMenuItemFloating(object sender, EventArgs e)
        {
            if (null == _menuTargetPage) { throw new Exception(); }

            if (!_menuTargetPage.IsEnableDockingState(DockingPageState.Floating)) { return; }

            _menuTargetPage.DockState = DockingPageState.Floating;
            _menuTargetPage = null;
        }

        private void OnDockingMenuItemDocking(object sender, EventArgs e)
        {
            if (null == _menuTargetPage) { throw new Exception(); }

            if (!_menuTargetPage.IsEnableDockingState(DockingPageState.Docked)) { return; }

            _menuTargetPage.DockState = DockingPageState.Docked;
            _menuTargetPage = null;
        }

        private void OnDockingMenuItemHide(object sender, EventArgs e)
        {
            if (null == _menuTargetPage) { throw new Exception(); }

            _menuTargetPage.DockVisible = false;
            _menuTargetPage = null;
        }

        #endregion

        #endregion

        #region ** メソッド

        public void Minimize()
        {
            SuspendLayout();

            _title.Hide();
            MinimizeTabControl();
            _tabControl.Minimizable = true;

            switch (_alignment)
            {
                case DockAlignment.Top:
                case DockAlignment.Bottom:
                    MinimumSize = new Size(MinimumSize.Width, _tabControl.MinimumSize.Height);
                    break;

                case DockAlignment.Left:
                case DockAlignment.Right:
                    MinimumSize = new Size(_tabControl.MinimumSize.Width, MinimumSize.Height);
                    break;
            }

            ResumeLayout();

            if (null != Minimized)
            {
                Minimized(this, new EventArgs());
            }
        }

        public void Restore()
        {
            SuspendLayout();

            _title.Show();
            _tabControl.TabControlStyle = NTabControlStyle.NormalMinimizableBottomTab;
            _tabControl.Minimizable = false;

            MinimumSize = Size.Empty;

            ResumeLayout();

            if (null != Restored)
            {
                Restored(this, new EventArgs());
            }
        }

        public void SelectPage(IDockingPage page)
        {
            if (null == page) { throw new ArgumentNullException("page"); }
            if (!_pages.Contains(page)) { throw new ArgumentException("page"); }

            if (!page.DockVisible)
            {
                page.DockVisible = true;
            }

            switch (page.DockState)
            {
                case DockingPageState.Docked:
                    if (IsMinimized)
                    {
                        Restore();
                    }

                    _tabControl.SelectedTab = page;
                    _tabControl.SelectedTab.Select();
                    break;

                case DockingPageState.Floating:
                    DockingForm form = _forms[page];
                    form.BringToFront();
                    form.Select();
                    break;
            }
        }

        public void ShowFloatingForms()
        {
            foreach (DockingForm form in _forms)
            {
                form.Show();
            }
        }

        public void UpdateTitle()
        {
            _title.Text = _tabControl.SelectedTab == null ? string.Empty : _tabControl.SelectedTab.Text;

            foreach (var form in _forms)
            {
                form.Text = form.Page.Text;
                form.Invalidate();
            }
        }

        protected virtual DockingForm CreateDockingForm(Form owner, IDockingPage page)
        {
            return new DockingForm(owner, page);
        }

        protected virtual DockingFormContainerTabControl CreateTabControl()
        {
            return new DockingFormContainerTabControl();
        }

        private void OnMinimizing()
        {
            if (null != Minimizing)
            {
                Minimizing(this, new EventArgs());
            }
        }

        private void UpdatePage(IDockingPage page)
        {
            Debug.Assert(null != page);

            if (!page.DockVisible)
            {

                if (_tabControl.TabPages.Contains(page))
                {
                    _tabControl.TabPages.Remove(page);
                }

                if (_forms.Contains(page))
                {
                    _forms[page].Close();
                    _forms.Remove(page);
                }

                return;

            }

            _title.Select();

            switch (page.DockState)
            {
                case DockingPageState.Docked:
                    if (_tabControl.TabPages.Contains(page)) { return; }

                    if (_forms.Contains(page))
                    {

                        DockingForm form = _forms[page];
                        _forms.Remove(page);

                        form.Page = null;
                        form.Close();

                    }

                    _tabControl.TabPages.Insert(GetDockedIndex(page), page);
                    _tabControl.SelectedTab = page;
                    page.Select();

                    break;

                case DockingPageState.Floating:
                    if (_forms.Contains(page)) { return; }

                    if (_tabControl.TabPages.Contains(page))
                    {
                        _tabControl.TabPages.Remove(page);
                    }

                    Form ownerForm = FindForm();

                    DockingForm newForm = CreateDockingForm(ownerForm, page);
                    newForm.NcContextMenuStrip = _dockingMenu;
                    _forms.Add(newForm);

                    if (null != ownerForm && ownerForm.IsHandleCreated)
                    {
                        newForm.Show();
                    }

                    break;
            }
        }

        private int GetDockedIndex(IDockingPage targetPage)
        {
            if (null == targetPage) { throw new ArgumentNullException("targetPage"); }

            int index = 0;

            foreach (IDockingPage page in _pages)
            {
                if (page == targetPage) { break; }
                if (!page.DockVisible) { continue; }
                if (DockingPageState.Docked != page.DockState) { continue; }
                index++;
            }

            return index;
        }

        private void SetAlignment(DockAlignment newAlignment)
        {
            if (newAlignment == _alignment) { return; }

            _alignment = newAlignment;
            _title.Alignment = newAlignment;

            if (IsMinimized)
            {
                MinimizeTabControl();
            }
        }

        private void MinimizeTabControl()
        {
            switch (_alignment)
            {
                case DockAlignment.Top:
                    _tabControl.TabControlStyle = NTabControlStyle.MinimizedTopTab;
                    break;

                case DockAlignment.Bottom:
                    _tabControl.TabControlStyle = NTabControlStyle.MinimizedBottomTab;
                    break;

                case DockAlignment.Left:
                    _tabControl.TabControlStyle = NTabControlStyle.MinimizedLeftTab;
                    break;

                case DockAlignment.Right:
                    _tabControl.TabControlStyle = NTabControlStyle.MinimizedRightTab;
                    break;
            }

            _tabControl.PerformLayout();
        }

        #endregion

        #region ** コンテナ

        public class DockingPageCollection : Collection<IDockingPage>
        {
            #region ** フィールド

            private Control _owner = null;

            #endregion

            public DockingPageCollection(Control owner)
            {
                Debug.Assert(null != owner);
                _owner = owner;

                owner.ControlAdded += OnPageAdded;
            }

            #region ** イベント

            public event DockingPageEventHandler ItemInserted;
            public event DockingPageEventHandler ItemRemoved;
            public event EventHandler ItemClearing;
            public event DockingPageEventHandler ItemChanged;

            #endregion

            #region ** イベントハンドラ

            private void OnPageAdded(object sender, ControlEventArgs e)
            {
                IDockingPage page = e.Control as IDockingPage;
                if (null == page) { return; }

                _owner.Controls.Remove(e.Control);
                Add(page);
            }

            #endregion

            #region ** メソッド

            public void Swap(int index1, int index2)
            {
                if (index1 < 0) { throw new ArgumentOutOfRangeException("index1"); }
                if (index2 < 0) { throw new ArgumentOutOfRangeException("index2"); }
                if (Count <= index1) { throw new ArgumentOutOfRangeException("index1"); }
                if (Count <= index2) { throw new ArgumentOutOfRangeException("index2"); }

                IDockingPage page1 = Items[index1];
                IDockingPage page2 = Items[index2];

                SetItem(index1, page2);
                SetItem(index2, page1);
            }

            public void Swap(IDockingPage page1, IDockingPage page2)
            {
                if (null == page1) { throw new ArgumentNullException("page1"); }
                if (null == page1) { throw new ArgumentNullException("page2"); }

                int index1 = Items.IndexOf(page1);
                int index2 = Items.IndexOf(page2);

                SetItem(index1, page2);
                SetItem(index2, page1);
            }

            protected override void InsertItem(int index, IDockingPage item)
            {
                base.InsertItem(index, item);

                if (null != ItemInserted)
                {
                    ItemInserted(this, new DockingPageEventArgs(item, index));
                }
            }

            protected override void RemoveItem(int index)
            {
                IDockingPage item = Items[index];

                base.RemoveItem(index);

                if (null != ItemRemoved)
                {
                    ItemRemoved(this, new DockingPageEventArgs(item, index));
                }
            }

            protected override void ClearItems()
            {
                if (null != ItemClearing)
                {
                    ItemClearing(this, new EventArgs());
                }

                base.ClearItems();
            }

            protected override void SetItem(int index, IDockingPage item)
            {
                base.SetItem(index, item);

                if (null != ItemChanged)
                {
                    ItemChanged(this, new DockingPageEventArgs(item, index));
                }
            }

            #endregion
        }

        public class DockingFormCollection : KeyedCollection<IDockingPage, DockingForm>
        {
            #region ** イベント

            public event DockingFormEventHandler ItemInserted;
            public event DockingFormEventHandler ItemRemoved;
            public event EventHandler ItemCleared;
            public event DockingFormEventHandler ItemChanged;

            public event DockingFormEventHandler ItemContextMenuShown;
            public event DockingFormEventHandler ItemCaptionDoubleClick;
            public event DockingFormEventHandler ItemClosed;

            #endregion

            #region ** イベントハンドラ

            private void OnItemContextMenuShown(object sender, EventArgs e)
            {
                DockingForm target = sender as DockingForm;
                Debug.Assert(Contains(target));

                if (null != ItemContextMenuShown)
                {
                    ItemContextMenuShown(
                                         this,
                                         new DockingFormEventArgs(target, Items.IndexOf(target)));
                }
            }

            private void OnItemCaptionDoubleClick(object sender, EventArgs e)
            {
                DockingForm target = sender as DockingForm;
                Debug.Assert(Contains(target));

                if (null != ItemCaptionDoubleClick)
                {
                    ItemCaptionDoubleClick(
                                           this,
                                           new DockingFormEventArgs(target, Items.IndexOf(target))
                                           );
                }
            }

            private void OnDockingFormClosed(object sender, FormClosedEventArgs e)
            {
                if (e.CloseReason != CloseReason.UserClosing) { return; }

                DockingForm target = sender as DockingForm;
                Debug.Assert(Contains(target));

                if (null != ItemClosed)
                {
                    ItemClosed(this, new DockingFormEventArgs(target, Items.IndexOf(target)));
                }
            }

            #endregion

            #region ** メソッド

            protected override IDockingPage GetKeyForItem(DockingForm item)
            {
                return item.Page;
            }

            protected override void InsertItem(int index, DockingForm item)
            {
                base.InsertItem(index, item);

                item.NcContextMenuShown += OnItemContextMenuShown;
                item.CaptionDoubleClick += OnItemCaptionDoubleClick;
                item.FormClosed += OnDockingFormClosed;

                if (null != ItemInserted)
                {
                    ItemInserted(this, new DockingFormEventArgs(item, index));
                }
            }

            protected override void RemoveItem(int index)
            {
                DockingForm item = Items[index];

                base.RemoveItem(index);

                item.NcContextMenuShown -= OnItemContextMenuShown;
                item.CaptionDoubleClick -= OnItemCaptionDoubleClick;
                item.FormClosed -= OnDockingFormClosed;

                if (null != ItemRemoved)
                {
                    ItemRemoved(this, new DockingFormEventArgs(item, index));
                }
            }

            protected override void ClearItems()
            {
                foreach (DockingForm item in Items)
                {
                    item.NcContextMenuShown -= OnItemContextMenuShown;
                    item.CaptionDoubleClick -= OnItemCaptionDoubleClick;
                    item.FormClosed -= OnDockingFormClosed;
                }

                base.ClearItems();

                if (null != ItemCleared)
                {
                    ItemCleared(this, new EventArgs());
                }
            }

            protected override void SetItem(int index, DockingForm item)
            {
                base.SetItem(index, item);

                if (null != ItemChanged)
                {
                    ItemChanged(this, new DockingFormEventArgs(item, index));
                }
            }

            #endregion
        }

        #endregion
    }

    #region ** イベントデリゲートとイベントパラメータ

    public delegate void DockingPageEventHandler(object sender, DockingPageEventArgs e);
    public delegate void DockingFormEventHandler(object sender, DockingFormEventArgs e);

    public class DockingPageEventArgs : EventArgs
    {
        private IDockingPage _page = null;
        private int _index = -1;

        public DockingPageEventArgs(IDockingPage page, int index)
        {
            Debug.Assert(null != page);
            _page = page;
            _index = index;
        }

        public IDockingPage Page
        {
            get { return _page; }
        }

        public int Index
        {
            get { return _index; }
        }
    }

    public class DockingFormEventArgs : EventArgs
    {
        private DockingForm _form = null;
        private int _index = -1;

        public DockingFormEventArgs(DockingForm form, int index)
        {
            Debug.Assert(null != form);
            _form = form;
            _index = index;
        }

        public DockingForm Form
        {
            get { return _form; }
        }

        public int Index
        {
            get { return _index; }
        }
    }

    #endregion
}
