﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OriginalKeyController : OperatableController
    {
        private int _OriginalKey = -1;
        private OriginalKeyInfo _OriginalKeyInfo = null;

        private int[] GraduationWidthArray = null;
        private int GraduationWidth = 8;

        private ExchangeMouseOperator ExchangeOperator = null;
        private MoveKeyMouseOperator MoveKeyOperator = null;

        private OriginalKeyDrawer _Drawer = new OriginalKeyDrawer();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public delegate void KeyChangedEventHandler(object sender, EventArgs e);
        public event KeyChangedEventHandler KeyChanged;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public OriginalKeyController()
        {
            InitializeComponent();

            ExchangeOperator = new ExchangeMouseOperator(this);
            MoveKeyOperator = new MoveKeyMouseOperator(this);
            CurrentOperator = ExchangeOperator;

            AllowDrop = true;

            HorizontalScrollBar.Visible = false;
            VerticalScrollBar.Visible = false;

            AddDrawer(DefaultDrawer);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int OriginalKey
        {
            get { return _OriginalKey; }
            set
            {
                _OriginalKey = value;
                RefreshInnerData();
                Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        /// デフォルトドロワーの取得
        /// </summary>
        public virtual OperatableControllerDrawer DefaultDrawer
        {
            get { return _Drawer; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool CanMoveKey
        {
            get { return _OriginalKey < 0 ? false : true; }
        }

        ///--------------------------------
        /// <summary>
        /// 目盛り幅の設定
        /// min, maxは未使用
        /// </summary>
        public void MapGraduationWidth(int[] array)
        {
            GraduationWidthArray = array;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int GetGraduationWidth(int position)
        {
            if (GraduationWidthArray != null)
            {
                if (GraduationWidthArray.Length <= 0)
                {
                    return 0;
                }

                if (position >= GraduationWidthArray.Length)
                {
                    return GraduationWidth;
                }
                else
                {
                    return GraduationWidthArray[position];
                }
            }
            else
            {

                return GraduationWidth;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        /// <value>
        ///
        /// </value>
        public int Min
        {
            get { return 0; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Max
        {
            get { return 128; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int LogicalPosition
        {
            get { return HorizontalScrollBar.Value; }
        }

        ///--------------------------------
        /// <summary>
        /// 論理座標からクライアント座標に変換する
        /// </summary>
        public int LogicalToClientPosition(int position)
        {
            if (GraduationWidthArray != null)
            {
                int x = 0;

                position -= Min;
                for (int loop = 0; loop < position; loop++)
                {
                    if (position > GraduationWidthArray.Length)
                    {
                        x += GraduationWidth;
                    }
                    else
                    {
                        x += GraduationWidthArray[loop];
                    }
                }

                return x;
            }
            else
            {

                return position * GraduationWidth;
            }
        }

        ///--------------------------------
        /// <summary>
        /// クライアント座標から論理座標に変換する
        /// </summary>
        public int ClientToLogicalPosition(int x)
        {
            if (GraduationWidthArray != null)
            {
                if (GraduationWidthArray.Length <= 0)
                {
                    return 0;
                }

                int position = LogicalPosition;
                int tx = GraduationWidthArray[0];

                while (x >= tx)
                {
                    position++;
                    if (position >= GraduationWidthArray.Length)
                    {
                        tx += GraduationWidth;
                    }
                    else
                    {
                        tx += GraduationWidthArray[position];
                    }
                }

                position += Min;
                if (position < Min) { position = Min; }
                if (position >= Max) { position = Max - 1; }

                return position;
            }
            else
            {

                return x / GraduationWidth + LogicalPosition;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 表示に必要な情報を Mediatorから取得する
        /// </summary>
        //static int _RefreshCount = 0;

        public void RefreshInnerData()
        {

            OriginalKeyInfo info = null;

            if (_OriginalKey >= 0)
            {
                info = new OriginalKeyInfo();
                info.KeyPosition = _OriginalKey;
            }

            _OriginalKeyInfo = info;

            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool MoveKey(int position)
        {
            if (_OriginalKey < 0) { return false; }
            _OriginalKey = position;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 待機状態のオペレータを設定
        /// </summary>
        protected internal void ChangeOperatorToExchange(Point beginPoint)
        {
            CurrentOperator = ExchangeOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// キー移動状態のオペレータを設定
        /// </summary>
        protected internal void ChangeOperatorToMoveKey(Point beginPoint)
        {
            CurrentOperator = MoveKeyOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// 表示できる目盛りの数の取得
        /// </summary>
        protected int GetVisibleGraduationCount(int position)
        {
            if (GraduationWidthArray != null)
            {
                int count = 0;
                int totalWidth = 0;
                int width = 0;

                position -= Min;

                while (totalWidth < ClientSize.Width)
                {
                    if (position >= GraduationWidthArray.Length)
                    {
                        width = GraduationWidth;
                    }
                    else
                    {
                        width = GraduationWidthArray[position];
                    }

                    position++;
                    count++;
                    totalWidth += width;
                }
                return count;

            }
            else
            {

                return ClientSize.Width / GraduationWidth;
            }
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        /// 表示情報の取得
        /// </summary>
        //protected override BasicUserControlDrawInformation GetDrawInformation()
        protected override Object GetDrawData()
        {
            return _OriginalKeyInfo;
        }

        ///--------------------------------
        /// <summary>
        /// 衝突判定
        /// 座標はクライアント座標を指定する
        /// </summary>
        protected virtual CollisionResult Collision(int mx, int my)
        {
            CollisionResult result = null;
            Rectangle rect = new Rectangle();
            int width = ClientSize.Width;
            int height = ClientSize.Height;

            //
            result = new CollisionResult();

            //if( DrawInfo != null ) {
            if (_OriginalKeyInfo != null)
            {
                rect.X = 0;
                rect.Y = 0;
                rect.Width = width;
                rect.Height = height;

                //ここでキーと当り判定を行う
                int key = -1;
                Rectangle keyRect;

                key = _OriginalKeyInfo.KeyPosition;
                keyRect = GetKeyRect(rect, key);


                int position = -1;

                position = ClientToLogicalPosition(mx);
                if (key == position)
                {
                    result.HitToKey(position);
                }
                else
                {

                    result.HitToNone(position);
                }
            }

            return result;
        }

        ///--------------------------------
        /// <summary>
        /// ドラックする点の情報の保存
        /// クライアント座標を指定する
        /// </summary>
        private long _DragID = -1;
        private int _DragLogicalPosition = -1;

        protected void PreserveDragRegion(long id, int mx, int my)
        {
            _DragID = id;
            _DragLogicalPosition = ClientToLogicalPosition(mx);
        }

        protected int GetDragDifferenceLogicalPosition(int position)
        {
            return position - _DragLogicalPosition;
        }

        ///--------------------------------
        /// <summary>
        /// ドラックマーカーを非表示にする
        /// </summary>
        protected void DragMarkerNone()
        {
            if (_OriginalKeyInfo != null)
            {
                _OriginalKeyInfo.DragMarkerInfo.None();
            }
        }

        ///--------------------------------
        /// <summary>
        /// キーのドラックマーカーを表示する
        /// </summary>
        protected void DragMarker(int position)
        {
            if (_OriginalKeyInfo != null)
            {
                _OriginalKeyInfo.DragMarkerInfo.Key(position);
            }
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventKeyChanged()
        {
            if (KeyChanged != null)
            {
                KeyChanged(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                    components.Dispose();
            }
            base.Dispose(disposing);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Rectangle GetKeyRect(Rectangle rect, int position)
        {
            int x = 0;
            int width = 0;

            x = LogicalToClientPosition(position);
            width = GetGraduationWidth(position);

            return new Rectangle(x, rect.Top, width, rect.Height - 1);
        }

        ///--------------------------------
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        ///--------------------------------
        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード］エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // HorizontalScrollBar
            //
            this.HorizontalScrollBar.Location = new System.Drawing.Point(0, 77);
            this.HorizontalScrollBar.Name = "HorizontalScrollBar";
            this.HorizontalScrollBar.Size = new System.Drawing.Size(134, 16);
            //
            // vScrollBar
            //
            this.VerticalScrollBar.Location = new System.Drawing.Point(134, 0);
            this.VerticalScrollBar.Name = "vScrollBar";
            this.VerticalScrollBar.Size = new System.Drawing.Size(16, 77);
            //
            // OriginalKeyController
            //
            this.Name = "OriginalKeyController";
            this.Size = new System.Drawing.Size(150, 93);
            this.ResumeLayout(false);

        }
        #endregion


        ///------------------------------------------------------------------------
        /// <summary>
        /// オペレータのベースクラス
        /// </summary>
        public class MouseOperator : OperatableControllerOperator
        {
            protected new OriginalKeyController OwnerController
            {
                get { return (OriginalKeyController)OperatableController; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MouseOperator(OperatableController controller) : base(controller)
            {
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// 待機状態のオペレータ
        /// </summary>
        public class ExchangeMouseOperator : MouseOperator
        {
            private Point _MouseFirstDownPoint = new Point(-1, -1);
            private bool _InterimDragKeyMover = false;
            private bool _InterimKey = false;

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public ExchangeMouseOperator(OperatableController controller) :
                base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        return MouseLeftButtonDownOperation(e);

                    case MouseButtons.Right:
                        return MouseRightButtonDownOperation(e);

                    case MouseButtons.Middle:
                        break;
                }
                return true;
            }


            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                if (_InterimKey != false)
                {
                    int key = OwnerController.ClientToLogicalPosition(e.X);
                    OwnerController.MoveKey(key);
                    OwnerController.ExecuteEventKeyChanged();

                    Cursor.Current = Cursors.Arrow;
                    OwnerController.DragMarkerNone();

                    OwnerController.RefreshInnerData();
                }

                _InterimDragKeyMover = false;
                _InterimKey = false;
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                Size dragSize = SystemInformation.DragSize;

                if (_InterimDragKeyMover != false)
                {

                    if (Math.Abs(e.X - _MouseFirstDownPoint.X) > dragSize.Width ||
                        Math.Abs(e.Y - _MouseFirstDownPoint.Y) > dragSize.Height)
                    {

                        if (_InterimDragKeyMover != false)
                        {
                            OwnerController.ChangeOperatorToMoveKey(_MouseFirstDownPoint);
                        }
                    }
                }
                else
                {

                    //カーソル形状の変更
                    CollisionResult result = OwnerController.Collision(e.X, e.Y);

                    switch (result.Type)
                    {
                        case CollisionResult.HitType.None:
                            break;

                        case CollisionResult.HitType.Key:
                            Cursor.Current = Cursors.Arrow;
                            Cursor.Current = Cursors.SizeWE;
                            break;
                    }
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _InterimDragKeyMover = false;
                _InterimKey = false;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool MouseLeftButtonDownOperation(MouseEventArgs e)
            {
                if (OwnerController.CanMoveKey == false)
                {
                    return true;
                }

                CollisionResult result = OwnerController.Collision(e.X, e.Y);
                switch (result.Type)
                {

                    ///
                    case CollisionResult.HitType.None:
                        int key = OwnerController.ClientToLogicalPosition(e.X);

                        Cursor.Current = Cursors.SizeWE;
                        OwnerController.DragMarker(key);
                        OwnerController.Invalidate();

                        _InterimKey = true;
                        _InterimDragKeyMover = true;
                        _MouseFirstDownPoint.X = e.X;
                        _MouseFirstDownPoint.Y = e.Y;
                        break;

                    ///
                    case CollisionResult.HitType.Key:
                        _InterimDragKeyMover = true;
                        _MouseFirstDownPoint.X = e.X;
                        _MouseFirstDownPoint.Y = e.Y;
                        break;
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool MouseRightButtonDownOperation(MouseEventArgs e)
            {
                return true;
            }

        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// キー移動処理状態のオペレータ
        /// </summary>
        public class MoveKeyMouseOperator : MouseOperator
        {

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MoveKeyMouseOperator(OperatableController controller) : base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                SetTemporaryKey(beginPoint.X);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyDown(KeyEventArgs e)
            {
                if (e.KeyCode == Keys.Escape)
                {
                    Cursor.Current = Cursors.Arrow;
                    OwnerController.DragMarkerNone();
                    OwnerController.Invalidate();
                    OwnerController.ChangeOperatorToExchange(new Point(-1, -1));
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                int key = OwnerController.ClientToLogicalPosition(e.X);

                OwnerController.MoveKey(key);
                OwnerController.ExecuteEventKeyChanged();

                OwnerController.DragMarkerNone();

                OwnerController.RefreshInnerData();
                OwnerController.ChangeOperatorToExchange(new Point(e.X, e.Y));
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                SetTemporaryKey(e.X);
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void SetTemporaryKey(int x)
            {
                int key = OwnerController.ClientToLogicalPosition(x);

                Cursor.Current = Cursors.SizeWE;

                OwnerController.DragMarker(key);
                OwnerController.Invalidate();
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// 衝突判定の結果を保持するクラス
        /// </summary>
        public class CollisionResult
        {
            public enum HitType
            {
                None,
                Key,
            }

            private HitType _Type = HitType.None;
            private int _Position = -1;

            ///--------------------------------
            /// <summary>
            /// 衝突無し
            /// </summary>
            public void HitToNone(int position)
            {
                _Type = HitType.None;
                _Position = position;
            }

            ///--------------------------------
            /// <summary>
            /// キーと衝突
            /// </summary>
            public void HitToKey(int position)
            {
                _Type = HitType.Key;
                _Position = position;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public HitType Type
            {
                get { return _Type; }
            }

            public int Position
            {
                get { return _Position; }
            }
        }
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OriginalKeyDrawer : OperatableControllerDrawer
    {
        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected internal override void Draw(Graphics gc, Rectangle clientRectangle, Object information, bool enabled)

        {
            //OriginalKeyControllerDrawInformation drawInfo = null;
            OriginalKeyInfo info = null;
            int key = -1;
            Rectangle rect = clientRectangle;
            Rectangle keyRect;

            if ((info = (OriginalKeyInfo)information) == null)
            {
                return;
            }

            key = info.KeyPosition;

            keyRect = GetKeyRect(rect, key);
            ControlPaint.DrawButton(gc, keyRect, ButtonState.Normal);
            //ControlPaint.DrawButton( gc, keyRect, ButtonState.Pushed);

            if (info.DragMarkerInfo.IsMarkerKey() != false)
            {
                keyRect = GetKeyRect(rect, info.DragMarkerInfo.Position);
                ControlPaint.DrawButton(gc, keyRect, ButtonState.Pushed);
            }

            //
            gc.DrawRectangle(Pens.Gray,
                              rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected new OriginalKeyController OwnerController
        {
            get
            {
                return (OriginalKeyController)base.OwnerController;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Rectangle GetKeyRect(Rectangle rect, int position)
        {
            int x = 0;
            int width = 0;

            x = OwnerController.LogicalToClientPosition(position);
            width = OwnerController.GetGraduationWidth(position);

            return new Rectangle(x, rect.Top, width, rect.Height - 1);
        }

    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OriginalKeyInfo
    {
        public int KeyPosition { get; set; }
        public OriginalKeyDragMarkerInfo DragMarkerInfo { get; set; }

        public OriginalKeyInfo()
        {
            KeyPosition = -1;
            DragMarkerInfo = new OriginalKeyDragMarkerInfo();
        }
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OriginalKeyDragMarkerInfo
    {
        public enum MarkerType
        {
            None,
            Key,
        }

        private MarkerType Type = MarkerType.None;
        private int _Position = 0;

        public void None()
        {
            Type = MarkerType.None;
            _Position = 0;
        }

        public void Key(int position)
        {
            Type = MarkerType.Key;
            _Position = position;
        }

        public bool IsMarkerKey()
        {
            return Type == MarkerType.Key ? true : false;
        }

        public int Position
        {
            get { return _Position; }
        }
    }
}
