﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using NintendoWare.SoundFoundation.Core.Win32;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    [ToolboxBitmap(typeof(TrackBar))]
    public class NTrackBar : TrackBar, IValueSynchronizable
    {
        // コントロールの状態
        private bool _valueCommited = true;		// true:値が確定済、false:値が変更中

        // パラメータ
        private bool _showToolTip = false;	// トラック中のツールチップ表示
        private int _decimalPlaces = 0;		// 小数点の桁数
        private decimal _maximum = 10;	// 値の上限
        private decimal _minimum = 0;		// 値の下限
        private decimal _increment = 1;		// 最小増減値(精度）

        // コンポーネント
        private ValueSynchronizable _valueSync;	// 値の同期用コンポーネント


        public NTrackBar()
        {
            // コンポーネントの生成
            _valueSync = new ValueSynchronizable(this);
            Debug.Assert(null != _valueSync);

            // プロパティの初期化
            AutoSize = false;

            base.Maximum = ToInternalValue(_maximum);
            base.Minimum = ToInternalValue(_minimum);
            base.Value = 0;
            base.SmallChange = 1;
            base.LargeChange = 5;
            base.TickFrequency = 5;

            // イベントハンドラの設定
            base.ValueChanged += OnValueChanged;
        }

        #region ** プロパティ

        public bool ShowToolTip
        {
            get { return _showToolTip; }
            set { _showToolTip = value; }
        }

        /// <summary>
        /// 小数部の桁数を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(0)]
        public virtual int DecimalPlaces
        {
            get { return _decimalPlaces; }
            set
            {
                if (0 > value) { throw new ArgumentOutOfRangeException(); }
                if (99 < value) { throw new ArgumentOutOfRangeException(); }
                if (value == _decimalPlaces) { return; }

                Rescale(value);
            }
        }

        /// <summary>
        /// 最小の増減値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(1)]
        public virtual decimal Increment
        {
            get { return _increment; }
            set
            {
                if (0 == Increment) { throw new ArgumentException(); }
                if (value == _increment) { return; }

                _increment = value;
            }
        }

        /// <summary>
        /// NTrackBar の表示に必要な高さを取得します。
        /// </summary>
        [Browsable(false)]
        public virtual int VisibleHeight
        {
            get
            {
                switch (TickStyle)
                {
                    case TickStyle.BottomRight:
                        return 28;

                    case TickStyle.TopLeft:
                        return 36;

                    case TickStyle.Both:
                        return 42;

                    default:
                        return 23;
                }
            }
        }

        /// <summary>
        /// 値の同期用コンポーネント
        /// </summary>
        [Browsable(false)]
        public ValueSynchronizable ValueSynchronized
        {
            get { return _valueSync; }
            set { _valueSync = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsDragging
        {
            get
            {
                return _valueCommited == false ? true : false;
            }
        }

        /// <summary>
        /// この NTrackBar が使用する範囲を取得します。
        /// </summary>
        private decimal Range
        {
            get { return Maximum - Minimum; }
        }

        #endregion

        #region ** プロパティのオーバーライド

        [DefaultValue(false)]
        public override bool AutoSize
        {
            get { return base.AutoSize; }
            set { base.AutoSize = value; }
        }

        #endregion

        #region ** プロパティの再定義

        /// <summary>
        /// トラック バーにおけるスクロール ボックスの現在位置を表す数値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(0)]
        public new virtual decimal Value
        {
            get { return ToExternalValue(base.Value); }
            set { base.Value = ToInternalValue(value); }
        }

        /// <summary>
        /// この NTrackBar が使用する範囲の上限値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(10)]
        public new virtual decimal Maximum
        {
            get { return _maximum; }
            set
            {
                if (value == _maximum) { return; }

                if (value < _minimum)
                {
                    UpdateMinimum(value);
                }

                UpdateMaximum(value);
                Rescale();
            }
        }

        /// <summary>
        /// この NTrackBar が使用する範囲の下限値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.All)]
        [DefaultValue(0)]
        public new virtual decimal Minimum
        {
            get { return _minimum; }
            set
            {
                if (value == _minimum) { return; }

                if (value > _maximum)
                {
                    UpdateMaximum(value);
                }

                UpdateMinimum(value);
                Rescale();
            }
        }

        /// <summary>
        /// スクロール ボックスがわずかに移動したときに Value プロパティに対して加算または減算される値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(1)]
        public new virtual decimal SmallChange
        {
            get { return ToExternalValue(base.SmallChange); }
            set
            {
                base.SmallChange = ToInternalValue(value);
            }
        }

        /// <summary>
        /// スクロール ボックスが大きく移動したときに Value プロパティに対して加算または減算される値を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(5)]
        public new virtual decimal LargeChange
        {
            get { return ToExternalValue(base.LargeChange); }
            set
            {
                base.LargeChange = ToInternalValue(value);
            }
        }

        /// <summary>
        /// コントロールに描画された目盛り間のデルタを指定する値を取得または設定します。
        /// </summary>
        [Category("Appearance")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(5)]
        public new virtual decimal TickFrequency
        {
            get { return ToExternalValue(base.TickFrequency); }
            set
            {
                base.TickFrequency = ToInternalValue(value);
            }
        }

        #endregion

        #region ** イベント

        public new event ValueChangedEventHandler ValueChanged
        {
            add { _valueSync.ValueChanged += value; }
            remove { _valueSync.ValueChanged -= value; }
        }

        public event ValueChangedEventHandler Synchronized
        {
            add { _valueSync.Synchronized += value; }
            remove { _valueSync.Synchronized -= value; }
        }

        #endregion

        #region ** イベントハンドラ

        private void OnValueChanged(object sender, EventArgs e)
        {
            // 値の精度を Increment 単位に合わせる
            // ただし、指定値 = 上下限値の場合は丸めない
            decimal remainder = Convert.ToDecimal(Value) % Increment;

            if (base.Minimum < base.Value && base.Value < base.Maximum && 0 != remainder)
            {

                decimal value = Math.Round(Value / Increment) * Increment;

                if (value < Minimum)
                {
                    base.Value = base.Minimum;
                }
                else if (value > Maximum)
                {
                    base.Value = base.Maximum;
                }
                else
                {
                    Value = value;
                }

            }
            else
            {
                _valueSync.DispatchValueChangedEvent();
            }
        }

        #endregion

        #region ** メソッド

        protected int RescaleValue(int value, int oldPlaces, int newPlaces)
        {
            return Convert.ToInt32(value * Math.Pow(10.0, newPlaces) / Math.Pow(10.0, oldPlaces));
        }

        protected int ToInternalValue(decimal value)
        {
            if (value * Increment == base.Maximum) { return base.Maximum; }
            if (value * Increment == base.Minimum) { return base.Minimum; }
            return Convert.ToInt32(Math.Round(value * Convert.ToDecimal(Math.Pow(10.0, _decimalPlaces)), 0));
        }

        protected decimal ToExternalValue(int value)
        {
            decimal extValue = value / Convert.ToDecimal(Math.Pow(10.0, _decimalPlaces));

            // 上下限値を Increment で割り切れない値に設定された場合、
            // InternalValue ではなく、設定された上下限値を返す
            if (extValue > _maximum) { return _maximum; }
            if (extValue < _minimum) { return _minimum; }

            return extValue;
        }

        protected void IncrementValue(decimal delta)
        {
            int newValue = base.Value + ToInternalValue(delta);

            if (newValue > base.Maximum)
            {
                newValue = base.Maximum;
            }
            else if (newValue < base.Minimum)
            {
                newValue = base.Minimum;
            }

            base.Value = newValue;
        }

        private void Rescale()
        {
            Rescale(_decimalPlaces);
        }

        private void Rescale(int newPlaces)
        {
            if (newPlaces == _decimalPlaces) { return; }

            // 再計算
            int workMaximum = RescaleValue(base.Maximum, _decimalPlaces, newPlaces);
            int workMinimum = RescaleValue(base.Minimum, _decimalPlaces, newPlaces);
            int workSmallChange = RescaleValue(base.SmallChange, _decimalPlaces, newPlaces);
            int workLargeChange = RescaleValue(base.LargeChange, _decimalPlaces, newPlaces);
            int workTickFrequency = RescaleValue(base.TickFrequency, _decimalPlaces, newPlaces);
            int workValue = RescaleValue(base.Value, _decimalPlaces, newPlaces);

            _decimalPlaces = newPlaces;

            base.Maximum = workMaximum;
            base.Minimum = workMinimum;
            base.SmallChange = Math.Max(1, workSmallChange);
            base.LargeChange = Math.Max(1, workLargeChange);
            base.TickFrequency = workTickFrequency;
            base.Value = workValue;
        }

        private void UpdateMaximum(decimal value)
        {
            if (value < Value)
            {
                Value = value;
            }

            _maximum = value;
            base.Maximum = ToInternalValue(value);
        }

        private void UpdateMinimum(decimal value)
        {
            if (value > Value)
            {
                Value = value;
            }

            _minimum = value;
            base.Minimum = ToInternalValue(value);
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            IncrementValue((e.Delta > 0) ? Increment : -Increment);
        }

        /// <summary>
        /// Window プロシージャのオーバーライド
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // スクロール
                case WM.WM_REFLECT | WM.WM_HSCROLL:
                case WM.WM_REFLECT | WM.WM_VSCROLL:
                    {
                        int code = Utility.LOWORD(m.WParam);

                        // スクロール終了
                        if (code == TB.TB_ENDTRACK)
                        {

                            if (_valueCommited == false)
                            {
                                _valueCommited = true;
                                _valueSync.DispatchValueChangedEvent();
                            }

                        }
                        // スクロール中
                        else
                        {
                            _valueCommited = false;
                        }

                    }
                    break;

                // マウスホイール
                case WM.WM_MOUSEWHEEL:
                    {
                        // デフォルトの処理を行うとスクロール量が調節できないので、処理を差し替える
                        OnMouseWheel(new MouseEventArgs(MouseButtons, 0, Utility.LOWORD(m.LParam),
                                                            Utility.HIWORD(m.LParam),
                                                            Utility.SignedHIWORD(m.WParam)));
                    }
                    return;
            }

            base.WndProc(ref m);
        }

        #endregion

        #region ** オペレータ

        public static implicit operator ValueSynchronizable(NTrackBar src)
        {
            return src.ValueSynchronized;
        }

        #endregion
    }
}
