﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;

namespace NintendoWare.SoundFoundation.Utilities
{
    /// <summary>
    /// ラウドネスを計算します。
    /// </summary>
    /// <remarks>
    /// <code>
    /// var loudnessCalculator = new LoudnessCalculator();
    /// loudnessCalculator.SetParameter(LoudnessCalculator.Parameter48kHz);
    /// while (hasSamples) {
    ///     int sampleCount = ...
    ///     short[][] samples = ...
    ///     loudnessCalculator.Accumulate(samples, 0, sampleCount);
    /// }
    /// loudnessCalculator.Finish();
    /// var gatedLoudnessValue = loudnessCalculator.GatedLoudnessValue;
    /// </code>
    /// </remarks>
    public class LoudnessCalculator
    {
        public const int ChannelCount = 6; // FrontLeft, FrontRight, RearLeft, RearRight, FrontCenter, LFE
        public const int StepTime = 100; // [msec]
        public const int AbsGateValue = -70; // LKFS
        public const int RelGateValue = -10; // LKFS

        private const int GatingBlockStepCount = 4;

        /// <summary>
        /// サンプルレート 48KHz の場合のパラメータです。
        /// </summary>
        /// <see cref="SetParameter(Parameter)"/>
        public static readonly Parameter Parameter48kHz =
            new Parameter(
                48000,
                // shelvingFilterCoef
                new IIRFilterCoeffiency(
                    -1.69065929318241f,
                    0.73248077421585f,
                    1.53512485958697f,
                    -2.69169618940638f,
                    1.19839281085285f),
                // highPassFilterCoeff
                new IIRFilterCoeffiency(
                    -1.99004745483398f,
                    0.99007225036621f,
                    1.0f,
                    -2.0f,
                    1.0f));

        /// <summary>
        /// サンプルレート 32KHz の場合のパラメータです。
        /// </summary>
        /// <see cref="SetParameter(Parameter)"/>
        public static readonly Parameter Parameter32kHz =
            new Parameter(
                32000,
                // shelvingFilterCoef
                new IIRFilterCoeffiency(
                    -1.53883606126270f,
                    0.62679865465060f,
                    1.51120520480243f,
                    -2.46461515884083f,
                    1.04137254742629f),
                // highPassFilterCoeff
                new IIRFilterCoeffiency(
                    -1.98508966895223f,
                    0.98514532063474f,
                    1.0f,
                    -2.0f,
                    1.0f));

        private static readonly float[] ChannelWeightArray =
            new float[ChannelCount] { 1.0f, 1.0f, 1.41f, 1.41f, 1.0f, 0.0f };

        public class IIRFilterCoeffiency
        {
            public float A1 { get; }
            public float A2 { get; }
            public float B0 { get; }
            public float B1 { get; }
            public float B2 { get; }

            public IIRFilterCoeffiency(float a1, float a2, float b0, float b1, float b2)
            {
                this.A1 = a1;
                this.A2 = a2;
                this.B0 = b0;
                this.B1 = b1;
                this.B2 = b2;
            }
        }

        public class Parameter
        {
            public int SampleRate { get; }
            public int StepSampleCount { get; }
            public IIRFilterCoeffiency ShelvingFilterCoef { get; }
            public IIRFilterCoeffiency HighPassFilterCoef { get; }

            public Parameter(
                int sampleRate,
                IIRFilterCoeffiency shelvingFilterCoef,
                IIRFilterCoeffiency highPassFilterCoef)
            {
                this.SampleRate = sampleRate;
                this.StepSampleCount = sampleRate * StepTime / 1000;
                this.ShelvingFilterCoef = shelvingFilterCoef;
                this.HighPassFilterCoef = highPassFilterCoef;
            }
        }

        /// <summary>
        /// 100 msec 区間の計算結果を保持します。
        /// </summary>
        private class StepResult
        {
            public float MeanSquare { get; }

            public StepResult(float meanSquare)
            {
                this.MeanSquare = meanSquare;
            }
        }

        /// <summary>
        /// 400 msec 区間の計算結果を保持します。
        /// </summary>
        private class GatingBlockResult
        {
            public float MeanSquare { get; }
            public float LoudnessValue { get; }

            public GatingBlockResult(float meanSquare)
            {
                this.MeanSquare = meanSquare;
                this.LoudnessValue = CalcLoudnessValue(meanSquare);
            }
        }

        /// <summary>
        /// K-Filter の IIR フィルタです(ステージごとに１つ)。
        /// </summary>
        private class IIRFilter
        {
            private readonly IIRFilterCoeffiency _coef;
            private float _x1;
            private float _x2;

            public IIRFilter(IIRFilterCoeffiency coef)
            {
                _coef = coef;
            }

            public float Do(float x)
            {
                float x0 = x - _coef.A1 * _x1 - _coef.A2 * _x2;

                float y = _coef.B0 * x0 + _coef.B1 * _x1 + _coef.B2 * _x2;

                _x2 = _x1;
                _x1 = x0;

                return y;
            }
        }

        /// <summary>
        /// K-Filter です。
        /// </summary>
        public class KWeightingFilter
        {
            private readonly IIRFilter _stage1;
            private readonly IIRFilter _stage2;

            public KWeightingFilter(Parameter parameter)
            {
                _stage1 = new IIRFilter(parameter.ShelvingFilterCoef);
                _stage2 = new IIRFilter(parameter.HighPassFilterCoef);
            }

            public float Do(float x)
            {
                return _stage2.Do(_stage1.Do(x));
            }
        }

        // チャンネル毎のラウドネス計算の蓄積結果を保持します。
        private class ChannelContext
        {
            private readonly KWeightingFilter _kWeightingFilter;
            private double _squareSum;

            public double SquareSum => _squareSum;

            public ChannelContext(Parameter parameter)
            {
                _kWeightingFilter = new KWeightingFilter(parameter);
            }

            public void Accumlate(float x)
            {
                float y = _kWeightingFilter.Do(x);
                _squareSum += y * y;
            }

            public void Reset()
            {
                _squareSum = 0;
            }
        }

        private Parameter _parameter;
        private int _stepSampleCount;
        private readonly ChannelContext[] _channelContexts = new ChannelContext[ChannelCount];
        private readonly Queue<StepResult> _stepResults = new Queue<StepResult>();
        private readonly List<GatingBlockResult> _gatingBlockResults = new List<GatingBlockResult>();

        /// <summary>
        /// 平均ラウドネス値の計算結果です(LKFS)。
        /// </summary>
        public float GatedLoudnessValue { get; private set; }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <see cref="Parameter48kHz"/>
        /// <see cref="Parameter32kHz"/>
        public void SetParameter(Parameter parameter)
        {
            _parameter = parameter;
            Populate(_channelContexts, _ => new ChannelContext(parameter));
            _stepResults.Clear();
            _gatingBlockResults.Clear();
            this.GatedLoudnessValue = float.NaN;
        }

        /// <summary>
        /// ラウドネス計算を蓄積します。
        /// <para>
        /// <paramref name="samples"/> にはチャンネル毎の波形データの配列を収めた配列を指定します。
        /// 波形データの配列は FrontLeft, FrontRight, RearLeft, RearRight, FrontCenter, LFE のチャンネル順で指定します。
        /// 要素数が <see cref="ChannelCount"/> に満たない場合、または要素が null の場合は、そのチャンネルは無音として扱います。
        /// </para>
        /// </summary>
        /// <param name="samples">波形データの配列の配列です。</param>
        /// <param name="offset">積算を開始する波形データの先頭位置です。</param>
        /// <param name="count">積算する波形データの数です。</param>
        public void Accumulate(short[][] samples, int offset, int count)
        {
            Debug.Assert(_parameter != null, "SetParameter() should be called before.");

            for (int i = 0; i < count; i++)
            {
                for (int c = 0; c < ChannelCount; c++)
                {
                    if (samples[c] != null)
                    {
                        _channelContexts[c].Accumlate(samples[c][offset + i] / 32768.0f);
                    }
                    else
                    {
                        _channelContexts[c].Accumlate(0);
                    }
                }

                _stepSampleCount++;

                if (_stepSampleCount == _parameter.StepSampleCount)
                {
                    // チャンネル毎の二乗平均を重み付きで合算します。
                    var meanSquare = _channelContexts
                        .Select(it => it.SquareSum / _parameter.StepSampleCount)
                        .Zip(ChannelWeightArray, (it, weight) => it * weight)
                        .Sum();

                    if (_stepResults.Count == GatingBlockStepCount)
                    {
                        _stepResults.Dequeue();
                    }

                    _stepResults.Enqueue(new StepResult((float)meanSquare));

                    if (_stepResults.Count == GatingBlockStepCount)
                    {
                        var gatingBlockMeanSquare = _stepResults
                            .Select(it => it.MeanSquare)
                            .Average();

                        _gatingBlockResults.Add(new GatingBlockResult(gatingBlockMeanSquare));
                    }

                    _channelContexts.ForEach(it => it.Reset());
                    _stepSampleCount = 0;
                }
            }
        }

        /// <summary>
        /// ラウドネス計算を完了させます。
        /// 計算結果は <see cref="GatedLoudnessValue"/> プロパティで取得できます。
        /// </summary>
        public void Finish()
        {
            Debug.Assert(_parameter != null, "SetParameter() should be called before.");

            this.GatedLoudnessValue = float.NegativeInfinity;

            if (_gatingBlockResults.IsEmpty() == true)
            {
                return;
            }

            // 絶対ゲーティング
            var meanSquareAbsGated = Average(
                _gatingBlockResults
                .Where(it => it.LoudnessValue > AbsGateValue)
                .Select(it => it.MeanSquare));

            if (meanSquareAbsGated.HasValue == false)
            {
                return;
            }

            // 相対ゲーティング
            var absGatedLoudnessValue = CalcLoudnessValue(meanSquareAbsGated.Value);
            var relGateValue = Math.Max(absGatedLoudnessValue + RelGateValue, AbsGateValue);
            var meanSquareRelGated = Average(
                _gatingBlockResults
                .Where(it => it.LoudnessValue > relGateValue)
                .Select(it => it.MeanSquare));

            if (meanSquareRelGated.HasValue == false)
            {
                return;
            }

            this.GatedLoudnessValue = CalcLoudnessValue(meanSquareRelGated.Value);
        }

        /// <summary>
        /// 波形サンプルの二乗平均から平均ラウドネス値を計算します。
        /// </summary>
        private static float CalcLoudnessValue(float meanSquare)
        {
            return (float)(-0.691 + 10 * Math.Log10(meanSquare));
        }

        /// <summary>
        /// 配列のすべての要素に値を設定します。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="array"></param>
        /// <param name="factory">要素ごとに呼び出され、戻り値が要素に格納されます。引数に要素の番号(0~)が渡されます。</param>
        /// <returns>引数 a を返します。</returns>
        private static T[] Populate<T>(T[] array, Func<int, T> factory)
        {
            for (int i = 0; i < array.Length; ++i)
            {
                array[i] = factory(i);
            }

            return array;
        }

        private static float? Average(IEnumerable<float> values)
        {
            double sum = 0;
            long count = 0;

            foreach (var v in values)
            {
                sum += v;
                count++;
            }

            if (count == 0)
            {
                return null;
            }

            return (float)(sum / count);
        }
    }
}
