﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using NintendoWare.SoundFoundation.Core;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// ComponentChildrenChangedEvent のデータを格納します。
    /// </summary>
    public class ComponentChildrenChangedEventArgs : RoutingEventArgs
    {
        private ComponentChildrenChangedAction _action = ComponentChildrenChangedAction.Reset;
        private Component[] _components;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="action">アクション。</param>
        public ComponentChildrenChangedEventArgs(ComponentChildrenChangedAction action)
            : base(new ComponentChildrenChangedEvent())
        {
            if (ComponentChildrenChangedAction.Reset != action)
            {
                throw new ArgumentException("invalid action type.");
            }
            _action = action;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="action">アクション。</param>
        /// <param name="components">対象コンポーネント。</param>
        public ComponentChildrenChangedEventArgs(ComponentChildrenChangedAction action, IEnumerable<Component> components)
            : base(new ComponentChildrenChangedEvent())
        {
            if (null == components) { throw new ArgumentNullException("components"); }

            switch (action)
            {
                case ComponentChildrenChangedAction.Add:
                case ComponentChildrenChangedAction.Remove:
                case ComponentChildrenChangedAction.Move:
                    break;

                default:
                    throw new ArgumentException("invalid action type.");
            }

            _action = action;
            _components = components.ToArray();
        }

        /// <summary>
        /// ComponentChildrenChangedEvent イベントを発生させたアクションを取得します。
        /// </summary>
        public ComponentChildrenChangedAction Action
        {
            get { return _action; }
        }

        /// <summary>
        /// 対象コンポーネントを取得します。
        /// </summary>
        public Component[] Components
        {
            get { return _components; }
        }
    }

    /// <summary>
    /// ComponentChildrenChangedEvent イベントを発生させるアクションを説明します。
    /// </summary>
    public enum ComponentChildrenChangedAction
    {
        /// <summary>
        /// 1 つ以上の項目がコレクションに追加されました。
        /// </summary>
        Add = NotifyCollectionChangedAction.Add,

        /// <summary>
        /// 1 つ以上の項目がコレクションから削除されました。
        /// </summary>
        Remove = NotifyCollectionChangedAction.Remove,

        /// <summary>
        /// コレクションの内容が大幅に変更されました。
        /// </summary>
        Reset = NotifyCollectionChangedAction.Reset,

        /// <summary>
        /// 1 つ以上の項目がコレクション内で移動されました。
        /// </summary>
        Move = NotifyCollectionChangedAction.Move,
    }

}
