﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using Core;
    using Core.Parameters;
    using Resources;
    using ToolDevelopmentKit;

    /// <summary>
    /// PanCurve 型のパラメータ値を格納します。
    /// </summary>
    public class PanCurveParameterValue : ParameterValue<PanCurve>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PanCurveParameterValue()
            : this(PanCurve.Sqrt)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータ値を指定します。</param>
        public PanCurveParameterValue(PanCurve value)
            : base(value)
        {
        }

        /// <summary>
        /// パラメータの値を取得するキャストオペレータです。
        /// </summary>
        /// <param name="value">対象となるパラメータ値を指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        public static implicit operator PanCurve(PanCurveParameterValue value)
        {
            Ensure.Argument.NotNull(value);
            return value.Value;
        }

        /// <summary>
        /// パラメータの値からテキストを取得します。
        /// </summary>
        /// <returns>テキストを返します。</returns>
        public override string ToString()
        {
            switch (this.Value)
            {
                case PanCurve.Sqrt:
                    return MessageResource.Label_PanCurve_Sqrt;

                case PanCurve.Sqrt_0Db:
                    return MessageResource.Label_PanCurve_Sqrt_0Db;

                case PanCurve.Sqrt_0Db_Clamp:
                    return MessageResource.Label_PanCurve_Sqrt_0Db_Clamp;

                case PanCurve.SinCos:
                    return MessageResource.Label_PanCurve_SinCos;

                case PanCurve.SinCos_0Db:
                    return MessageResource.Label_PanCurve_SinCos_0Db;

                case PanCurve.SinCos_0Db_Clamp:
                    return MessageResource.Label_PanCurve_SinCos_0Db_Clamp;

                case PanCurve.Linear:
                    return MessageResource.Label_PanCurve_Linear;

                case PanCurve.Linear_0Db:
                    return MessageResource.Label_PanCurve_Linear_0Db;

                case PanCurve.Linear_0Db_Clamp:
                    return MessageResource.Label_PanCurve_Linear_0Db_Clamp;
            }

            Debug.Assert(false, "An unexpected value");
            return String.Empty;
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        protected override PanCurve ParseInternal(string text)
        {
            Ensure.Argument.NotNull(text);

            if (text == MessageResource.Label_PanCurve_Sqrt)
            {
                return PanCurve.Sqrt;
            }
            else if (text == MessageResource.Label_PanCurve_Sqrt_0Db)
            {
                return PanCurve.Sqrt_0Db;
            }
            else if (text == MessageResource.Label_PanCurve_Sqrt_0Db_Clamp)
            {
                return PanCurve.Sqrt_0Db_Clamp;
            }
            else if (text == MessageResource.Label_PanCurve_SinCos)
            {
                return PanCurve.SinCos;
            }
            else if (text == MessageResource.Label_PanCurve_SinCos_0Db)
            {
                return PanCurve.SinCos_0Db;
            }
            else if (text == MessageResource.Label_PanCurve_SinCos_0Db_Clamp)
            {
                return PanCurve.SinCos_0Db_Clamp;
            }
            else if (text == MessageResource.Label_PanCurve_Linear)
            {
                return PanCurve.Linear;
            }
            else if (text == MessageResource.Label_PanCurve_Linear_0Db)
            {
                return PanCurve.Linear_0Db;
            }
            else if (text == MessageResource.Label_PanCurve_Linear_0Db_Clamp)
            {
                return PanCurve.Linear_0Db_Clamp;
            }

            throw new ArgumentException("invalid text.");
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected override ValidationResult ValidateInternal(PanCurve value)
        {
            return ValidationResult.NoError;
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected override IEnumerable<byte> GetBytes()
        {
            return BitConverter.GetBytes((int)this.Value);
        }
    }
}
