﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using Core;
    using Core.Parameters;
    using Resources;
    using ToolDevelopmentKit;

    /// <summary>
    /// LoopType 型のパラメータ値を格納します。
    /// </summary>
    public class LoopTypeParameterValue : ParameterValue<LoopType>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LoopTypeParameterValue()
            : this(LoopType.InFile)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータ値を指定します。</param>
        public LoopTypeParameterValue(LoopType value)
            : base(value)
        {
        }

        /// <summary>
        /// パラメータの値を取得するキャストオペレータです。
        /// </summary>
        /// <param name="value">対象となるパラメータ値を指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        public static implicit operator LoopType(LoopTypeParameterValue value)
        {
            Ensure.Argument.NotNull(value);
            return value.Value;
        }

        /// <summary>
        /// パラメータの値からテキストを取得します。
        /// </summary>
        /// <returns>テキストを返します。</returns>
        public override string ToString()
        {
            switch (this.Value)
            {
                case LoopType.InFile:
                    return MessageResource.Label_LoopType_InFile;

                case LoopType.Manual:
                    return MessageResource.Label_LoopType_Manual;

                case LoopType.None:
                    return MessageResource.Label_LoopType_None;
            }

            Debug.Assert(false, "An unexpected value");
            return String.Empty;
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        protected override LoopType ParseInternal(string text)
        {
            Ensure.Argument.NotNull(text);

            if (text == MessageResource.Label_LoopType_InFile)
            {
                return LoopType.InFile;
            }
            else if (text == MessageResource.Label_LoopType_Manual)
            {
                return LoopType.Manual;
            }
            else if (text == MessageResource.Label_LoopType_None)
            {
                return LoopType.None;
            }

            throw new ArgumentException("invalid text.");
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected override ValidationResult ValidateInternal(LoopType value)
        {
            return ValidationResult.NoError;
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected override IEnumerable<byte> GetBytes()
        {
            return BitConverter.GetBytes((int)this.Value);
        }
    }
}
