﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using System.Xml;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.ToolDevelopmentKit;

    public abstract class ComplexParameterValue<TValue> : ParameterValue<TValue>, IParameterProvider
        where TValue : class, INotifyPropertyChanged
    {
        private readonly StaticParameterProvider<TValue> parameterProvider;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        protected ComplexParameterValue(TValue value)
            : base(value)
        {
            this.parameterProvider = new StaticParameterProvider<TValue>(value);
        }

        /// <summary>
        /// パラメータの辞書を取得します。
        /// </summary>
        public IParameterDictionary Parameters
        {
            get
            {
                return this.parameterProvider.Parameters;
            }
        }

        /// <summary>
        /// XML シリアライズ時の要素名を取得します。
        /// </summary>
        protected abstract string XmlElementName
        {
            get;
        }

        /// <summary>
        /// パラメータの値からテキストを取得します。
        /// </summary>
        /// <returns>テキストを返します。</returns>
        public sealed override string ToString()
        {
            StringBuilder stringBuilder = new StringBuilder();

            XmlWriterSettings settings = new XmlWriterSettings()
            {
                ConformanceLevel = ConformanceLevel.Fragment,
                Indent = false,
                OmitXmlDeclaration = true,
            };

            Assertion.Operation.True(!string.IsNullOrEmpty(this.XmlElementName));

            using (XmlWriter writer = XmlWriter.Create(stringBuilder, settings))
            {
                writer.WriteStartElement(this.XmlElementName);
                writer.WriteRaw(this.parameterProvider.ToString());
                writer.WriteEndElement();
            }

            return stringBuilder.ToString();
        }

        /// <summary>
        /// パラメータの値が変更されると発生します。
        /// </summary>
        /// <param name="e">イベントパラメータを指定します。</param>
        protected sealed override void OnValueChanged(ParameterEventArgs e)
        {
            this.parameterProvider.Value = this.Value;

            base.OnValueChanged(e);
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        protected sealed override TValue ParseInternal(string text)
        {
            Assertion.Argument.NotNull(text);

            XmlDocument document = new XmlDocument();
            document.LoadXml(text);

            Assertion.Operation.True(!string.IsNullOrEmpty(this.XmlElementName));

            if (document.DocumentElement == null ||
                document.DocumentElement.Name != this.XmlElementName)
            {
                return this.Value;
            }

            this.parameterProvider.Parse(document.DocumentElement.InnerXml);
            return this.Value;
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected override ValidationResult ValidateInternal(TValue value)
        {
            return ValidationResult.NoError;
        }

        /// <summary>
        /// パラメータ値のハッシュコードを作成します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="key">パラメータキーを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>作成したハッシュ値を返します。</returns>
        protected sealed override HashCode CreateHashCode(
            HashAlgorithm algorithm,
            string key,
            Func<IParameterValue, bool> filter)
        {
            return this.Parameters.GetParameterDictionaryHashCode(algorithm, key, filter);
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected sealed override IEnumerable<byte> GetBytes()
        {
            // ここにはこないはず。
            throw new Exception();
        }

        /// <summary>
        ///
        /// </summary>
        protected void SetParameterAttributes(string parameterName, params object[] attributes)
        {
            Assertion.Argument.StringNotEmpty(parameterName);
            Assertion.Operation.True(this.Parameters.ContainsKey(parameterName));
            Assertion.Argument.NotNull(attributes);

            foreach (var attribute in attributes)
            {
                this.Parameters[parameterName].Attributes.Add(attribute);
            }
        }
    }
}
