﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Documents;

namespace NintendoWare.SoundFoundation.Projects
{
    public abstract class SoundDocument : Document
    {
        private bool isReadOnly = false;
        private bool _isComponentsDirty = false;

        ///
        public bool IsReadOnly
        {
            get
            {
                if (this.isReadOnly == false)
                {
                    return !Resource.CanWrite;
                }

                return this.isReadOnly;
            }
            set
            {
                this.isReadOnly = value;
            }
        }

        /// <summary>
        /// ドキュメントの最上位コンポーネントの一覧を取得します。
        /// </summary>
        public abstract Component[] TopComponents { get; }

        /// <summary>
        /// 他の要因でドキュメントが編集されているかどうかを調べます。
        /// </summary>
        protected override bool IsDirtyOtherFactor
        {
            get { return _isComponentsDirty; }
        }

        /// <summary>
        /// コンポーネントの編集状態を更新します。
        /// </summary>
        public void UpdateComponentsDirty()
        {
            bool old = IsDirty;

            _isComponentsDirty = QueryComponentsDirty();

            if (old != IsDirty)
            {
                OnDirtyChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// コンポーネントが編集されているかどうかを調べます。
        /// </summary>
        /// <returns>編集されている場合 true、編集されていない場合 false。</returns>
        protected bool QueryComponentsDirty()
        {
            foreach (Component topComponent in TopComponents)
            {

                QueryComponentDirtyEventArgs e = new QueryComponentDirtyEventArgs();

                (topComponent as IEventRoutable).Router.DispatchEvent(e);

                if (e.Result) { return true; }

            }

            return false;
        }

        /// <summary>
        /// Dirty 状態がリセットされると発生します。
        /// </summary>
        protected override void OnResetDirty()
        {
            ResetComponentsDirty();
        }

        /// <summary>
        /// コンポーネントの編集状態をリセットします。
        /// </summary>
        private void ResetComponentsDirty()
        {
            bool old = IsDirty;

            foreach (Component topComponent in TopComponents)
            {
                (topComponent as IEventRoutable).Router.DispatchEvent(
                            new RoutingEventArgs(new ResetComponentDirtyEvent()));
            }

            if (old != IsDirty)
            {
                OnDirtyChanged(EventArgs.Empty);
            }
        }
    }
}
