﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Resources;
using NintendoWare.ToolDevelopmentKit.Collections;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using YamlDotNet;
using YamlDotNet.Core;
using YamlDotNet.RepresentationModel;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// コンポーネントのパラメータの初期値を管理します。
    /// </summary>

    public delegate Component CreateComponentDelegate(Type type);

    public class DefaultParamManager
    {
        private readonly Dictionary<Type, Dictionary<string, object>> _componentDefaultParameterDictionary = new Dictionary<Type, Dictionary<string, object>>();
        private readonly CreateComponentDelegate _createComponentDelegate;


        // <summary>
        // コンストラクタ
        // </summary>
        public DefaultParamManager(CreateComponentDelegate create)
        {
            Debug.Assert(create != null);

            _createComponentDelegate = create;

            // SoundMaker.exe でのみ有効 true にします。それ以外のツールでは無効 false です。初期値は無効 false です。
            this.Enabled = false;
        }

        /// <summary>
        /// コンポーネントのパラメータの初期値を有効にします。
        /// </summary>
        public bool Enabled
        {
            get; set;
        }

        // <summary>
        // パラメータ初期値設定ファイルを読み込みます。
        // </summary>
        // <param name="filePath">パラメータ初期値設定ファイルのパス</param>
        public void LoadFile(string filePath)
        {
            if (this.Enabled == false)
            {
                return;
            }

            Debug.Assert(string.IsNullOrEmpty(filePath) == false);

            _componentDefaultParameterDictionary.Clear();

            if (string.IsNullOrEmpty(filePath) == true || File.Exists(filePath) == false)
            {
                return;
            }

            var logMessages = new StringBuilder();

            var yaml = new YamlStream();
            StreamReader streamReader = null;
            try
            {
                streamReader = new StreamReader(filePath, Encoding.UTF8);
                yaml.Load(streamReader);
            }
            catch (YamlException e)
            {
                // Warning 不正なフォーマットです。
                logMessages.AppendLine(MessageResource.Message_ReadErrorDefaultParameterFile);
                logMessages.AppendLine(string.Format(MessageResource.Message_ConfirmFile, filePath));
                logMessages.AppendLine(MessageResource.Message_FormatErrorDefaultParameterFile);
                logMessages.AppendLine(e.Message);
                _componentDefaultParameterDictionary.Clear();
                throw new Exception(logMessages.ToString());
            }
            finally
            {
                if (streamReader != null)
                {
                    streamReader.Close();
                }
            }

            if (yaml.Documents.Count <= 0)
            {
                // 記述がない（全てコメントアウト）の場合は何もしない。（エラーにしない）
                _componentDefaultParameterDictionary.Clear();
                return;
            }

            var mapping = yaml.Documents[0].RootNode as YamlMappingNode;

            if (mapping == null)
            {
                // Warning 不正なフォーマットです。
                logMessages.AppendLine(MessageResource.Message_ReadErrorDefaultParameterFile);
                logMessages.AppendLine(string.Format(MessageResource.Message_ConfirmFile, filePath));
                var node = yaml.Documents[0].RootNode as YamlNode;

                if (node != null)
                {
                    // トークンや行番号が取得できるなら表示する
                    logMessages.AppendLine(string.Format(MessageResource.Message_FormatErrorDefaultParameterFileWithLine, node.ToString(), node.Start.Line));
                }
                else
                {
                    logMessages.AppendLine(MessageResource.Message_FormatErrorDefaultParameterFile);
                }

                _componentDefaultParameterDictionary.Clear();
                throw new Exception(logMessages.ToString());
            }

            foreach (var child in mapping.Children)
            {
                Type componentType = this.GetComponentType(child.Key.ToString()); // StreamSound, WaveSound などのアイテム名
                if (componentType == null)
                {
                    // Warning 存在しない名前 *** が指定されました。アイテムの種類の名前を確認してください。
                    logMessages.AppendLine(string.Format(MessageResource.Message_IllegalComponentName, child.Key.ToString(), child.Key.Start.Line));
                    continue;
                }

                var soundItem = child.Value as YamlMappingNode;
                if (soundItem == null)
                {
                    // パラメータ設定が一つもない場合はエラーにしない。
                    continue;
                }

                if (_componentDefaultParameterDictionary.ContainsKey(componentType) == false)
                {
                    _componentDefaultParameterDictionary.Add(componentType, new Dictionary<string, object>());
                }

                var dictionary = _componentDefaultParameterDictionary[componentType];
                var component = _createComponentDelegate(componentType);
                Debug.Assert(componentType.IsInstanceOfType(component) == true);

                foreach (var item in soundItem.Children)
                {
                    var key = item.Key.ToString();
                    string parameterName = this.GetParameterName(key);
                    if (string.IsNullOrEmpty(parameterName) == true)
                    {
                        // Warning 存在しないパラメータ名 *** が指定されました。パラメータ名を確認してください。
                        logMessages.AppendLine(string.Format(MessageResource.Message_IllegalParameterName, key, item.Key.Start.Line));
                        continue;
                    }

                    object parameterValue;
                    var result = this.GetParameterValue(component, parameterName, item.Value.ToString(), out parameterValue);
                    if (result.IsValid == false)
                    {
                        // Warning パラメータ *** の値の書式を確認してください。
                        logMessages.AppendLine(string.Format(MessageResource.Message_IllegalParameterValue, result.ToString(), key, item.Key.Start.Line));
                        continue;
                    }

                    if (dictionary.ContainsKey(parameterName) == false)
                    {
                        dictionary.Add(parameterName, parameterValue); // パラメータと値
                    }
                    else
                    {
                        // Warning 重複したパラメータ名です。
                        logMessages.AppendLine(string.Format(MessageResource.Message_SameParameterValue, key, item.Key.Start.Line));
                    }
                }
            }

            if (logMessages.Length > 0)
            {
                var sb = new StringBuilder();

                sb.AppendLine(string.Format(MessageResource.Message_ReadErrorDefaultParameterFile));
                sb.AppendLine(string.Format(MessageResource.Message_ConfirmFile, filePath));

                logMessages.Insert(0, sb.ToString());

                _componentDefaultParameterDictionary.Clear();
                throw new Exception(logMessages.ToString());
            }
        }

        // <summary>
        // パラメータ初期値設定ファイルを出力します。
        // </summary>
        // <param name="filePath">パラメータ初期値設定ファイルのパス</param>
        public void OutputFile(string filePath)
        {
            if (this.Enabled == false)
            {
                return;
            }

            if (File.Exists(filePath) == true)
            {
                return;
            }

            try
            {
                if (Thread.CurrentThread.CurrentCulture.Name == "ja-JP")
                {
                    File.WriteAllBytes(filePath, FileResource.DefaultParam_ja_JP);
                }
                else
                {
                    File.WriteAllBytes(filePath, FileResource.DefaultParam_en_US);
                }
            }
            catch
            {
            }
        }

        /// <summary>
        /// コンポーネントのパラメータに初期値を設定します。
        /// </summary>
        /// <param name="component">パラメータを初期値にしたいコンポーネント</param>
        public void ApplyDefaultParameters(Component component)
        {
            if (this.Enabled == false || _componentDefaultParameterDictionary.Count <= 0)
            {
                return;
            }

            var defaultParameterDictionary = _componentDefaultParameterDictionary
                .Where(it => it.Key.IsInstanceOfType(component))
                .Select(it => it.Value)
                .FirstOrDefault();

            this.ApplyDefaultParameters(defaultParameterDictionary, component);
        }

        private void ApplyDefaultParameters(Dictionary<string, object> dictionary, Component component)
        {
            if (dictionary == null)
            {
                return;
            }

            foreach (var pair in dictionary)
            {
                switch (pair.Key)
                {
                    case "SoundSetBankReference":
                        this.ApplySoundSetBankReference(component, 0, pair.Value);
                        break;

                    case "SoundSetBankReference1":
                        this.ApplySoundSetBankReference(component, 1, pair.Value);
                        break;

                    case "SoundSetBankReference2":
                        this.ApplySoundSetBankReference(component, 2, pair.Value);
                        break;

                    case "SoundSetBankReference3":
                        this.ApplySoundSetBankReference(component, 3, pair.Value);
                        break;

                    default:
                        component.Parameters[pair.Key].Value = pair.Value;
                        break;
                }
            }
        }

        private void ApplySoundSetBankReference(Component component, int index, object value)
        {
            SequenceSoundBase sequenceSound = component as SequenceSoundBase;
            var list = sequenceSound.SoundSetBankReferences;

            if (list.Count < SequenceSoundBase.BankReferenceCount)
            {
                list.Clear();

                for (int i = 0; i < SequenceSoundBase.BankReferenceCount; i++)
                {
                    list.Add(new ComponentReference());
                }
            }

            list[index].TargetName = value.ToString();
        }

        private Type GetComponentType(string name)
        {
            name = name.ToLower(); // 大文字小文字を区別しない。

            if (_componentTypes.ContainsKey(name) == true)
            {
                return _componentTypes[name];
            }

            return null;
        }

        private string GetParameterName(string name)
        {
            name = name.ToLower(); // 大文字小文字を区別しない。

            if (_parameterNames.ContainsKey(name) == true)
            {
                return _parameterNames[name];
            }

            return string.Empty;
        }

        private ValidationResult GetParameterValue(Component component, string parameterName, string valueText, out object value)
        {
            try
            {
                switch (parameterName)
                {
                    case "ColorIndex": return this.GetColorIndexParameterValue(valueText, out value);
                    case "BiquadType": return this.GetBiquadTypeParameterValue(valueText, out value);
                    case "Sound3D/DecayCurve3D": return this.GetDecayCurve3DParameterValue(valueText, out value);
                    case "PanMode": return this.GetPanModeParameterValue(valueText, out value);
                    case "PanCurve": return this.GetPanCurveParameterValue(valueText, out value);
                    case "SequenceSoundFileType": return this.GetSequenceSoundFileTypeParameterValue(valueText, out value);
                    case "GroupOutputType": return this.GetGroupOutputTypeParameterValue(valueText, out value);
                    case "InstrumentEnvelopeMode": return this.GetInstrumentEnvelopeModeParameterValue(valueText, out value);
                    case "WaveArchiveLoadType": return this.GetWaveArchiveLoadTypeParameterValue(valueText, out value);

                    case "WaveEncoding": valueText = this.TranslateWaveEncodingValueText(valueText); break; // これは下の処理にまかせる
                    case "WaveArchiveReference": valueText = this.TranslateWaveArchiveReferenceValueText(valueText); break; // これは下の処理にまかせる
                    case "SoundSetBankReference":
                    case "SoundSetBankReference1":
                    case "SoundSetBankReference2":
                    case "SoundSetBankReference3":
                        value = valueText;
                        return new ValidationResult(true); // この四つは常に成功にする
                }

                if (component.Parameters.ContainsKey(parameterName) == true)
                {
                    var parameterValue = component.Parameters[parameterName];
                    value = parameterValue.ParseValue(valueText);
                    var result = parameterValue.ValidateValue(value);
                    if (result.IsValid == false)
                    {
                        value = null;
                    }

                    return result;
                }
            }
            catch
            {
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetColorIndexParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_colorIndices.ContainsKey(text) == true)
            {
                value = _colorIndices[text];
                return new ValidationResult(true);
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetBiquadTypeParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_biquadTypes.ContainsKey(text) == true)
            {
                try
                {
                    value = BiquadTypeEx.Parse(_biquadTypes[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetDecayCurve3DParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_decayCurve3Ds.ContainsKey(text) == true)
            {
                try
                {
                    value = DecayCurve3DEx.Parse(_decayCurve3Ds[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetPanModeParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_panModes.ContainsKey(text) == true)
            {
                try
                {
                    value = PanModeEx.Parse(_panModes[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetPanCurveParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_panCurves.ContainsKey(text) == true)
            {
                try
                {
                    value = PanCurveEx.Parse(_panCurves[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetSequenceSoundFileTypeParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_sequenceSoundFileTypes.ContainsKey(text) == true)
            {
                try
                {
                    value = SequenceSoundFileTypeEx.Parse(_sequenceSoundFileTypes[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetGroupOutputTypeParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_groupOutputTypes.ContainsKey(text) == true)
            {
                try
                {
                    value = GroupOutputTypeEx.Parse(_groupOutputTypes[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetInstrumentEnvelopeModeParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_instrumentEnvelopeModes.ContainsKey(text) == true)
            {
                try
                {
                    value = InstrumentEnvelopeModeEx.Parse(_instrumentEnvelopeModes[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }

        private ValidationResult GetWaveArchiveLoadTypeParameterValue(string valueText, out object value)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_waveArchiveLoadTypes.ContainsKey(text) == true)
            {
                try
                {
                    value = WaveArchiveLoadTypeEx.Parse(_waveArchiveLoadTypes[text]);
                    return new ValidationResult(true);
                }
                catch
                {
                }
            }

            value = null;
            return new ValidationResult(false);
        }


        private string TranslateWaveEncodingValueText(string valueText)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_waveEncodings.ContainsKey(text) == true)
            {
                return _waveEncodings[text];
            }

            return string.Empty;
        }

        private string TranslateWaveArchiveReferenceValueText(string valueText)
        {
            var text = valueText.ToLower(); // 大文字小文字を区別しない。

            if (_waveArchiveReferences.ContainsKey(text) == true)
            {
                return _waveArchiveReferences[text];
            }

            return valueText; // それ以外はそのまま返す。
        }


        // 変換用データ

        // コンポーネントタイプ
        private readonly Dictionary<string, Type> _componentTypes = new Dictionary<string, Type>()
        {
            { "streamsound",      typeof(StreamSoundBase) },
            { "streamsoundtrack", typeof(StreamSoundTrackBase) },
            { "wavesoundset",     typeof(WaveSoundSetBase) },
            { "wavesound",        typeof(WaveSoundBase) },
            { "sequencesoundset", typeof(SequenceSoundSetBase) },
            { "sequencesound",    typeof(SequenceSoundBase) },
            { "soundsetbank",     typeof(SoundSetBankBase) },
            { "wavearchive",      typeof(WaveArchiveBase) },
            { "group",            typeof(GroupBase) },
            { "groupitem",        typeof(GroupItemBase) },
            { "player",           typeof(PlayerBase) },
            { "instrument",       typeof(Instrument) },
        };

        // パラメータ名
        private readonly Dictionary<string, string> _parameterNames = new Dictionary<string, string>()
        {
            { "3dfilter",                    "Sound3D/Enable3DFilter" },
            { "3dpan",                       "Sound3D/Enable3DPan" },
            { "3dpriority",                  "Sound3D/Enable3DPriority" },
            { "3dsurroundpan",               "Sound3D/Enable3DSurroundPan" },
            { "3dvolume",                    "Sound3D/Enable3DVolume" },
            { "actorplayer",                 "ActorPlayer" },
            { "auxasend",                    "Sends/AuxASend" },
            { "auxbsend",                    "Sends/AuxBSend" },
            { "auxcsend",                    "Sends/AuxCSend" },
            { "biquad",                      "Biquad" },
            { "biquadtype",                  "BiquadType" },
            { "channelpriority",             "ChannelPriority" },
            { "color",                       "ColorIndex" },
            { "comment",                     "Comment" },
            { "comment1",                    "Comment1" },
            { "comment2",                    "Comment2" },
            { "comment3",                    "Comment3" },
            { "comment4",                    "Comment4" },
            { "comment5",                    "Comment5" },
            { "comment6",                    "Comment6" },
            { "comment7",                    "Comment7" },
            { "comment8",                    "Comment8" },
            { "comment9",                    "Comment9" },
            { "decaycurve3d",                "Sound3D/DecayCurve3D" },
            { "decayratio3d",                "Sound3D/DecayRatio3D" },
            { "dopplerfactor3d",             "Sound3D/DopplerFactor3D" },
            { "envattack",                   "Envelope/Attack" },
            { "envdecay",                    "Envelope/Decay" },
            { "envhold",                     "Envelope/Hold" },
            { "envrelease",                  "Envelope/Release" },
            { "envsustain",                  "Envelope/Sustain" },
            { "enveloperelease",             "EnvelopeRelease" },
            { "envelopemode",                "InstrumentEnvelopeMode" },
            { "groupoutputtype",             "GroupOutputType" },
            { "heapsize",                    "PlayerHeapSize" },
            { "individual",                  "WaveArchiveLoadType"},
            { "lpf",                         "LPF" },
            { "mainsend",                    "Sends/MainSend" },
            { "pan",                         "Pan" },
            { "pancurve",                    "PanCurve" },
            { "panmode",                     "PanMode" },
            { "pitch",                       "Pitch" },
            { "pitchcents",                  "PitchCents" },
            { "pitchsemitones",              "PitchSemitones" },
            { "player",                      "PlayerReference" },
            { "playerpriority",              "PlayerPriority" },
            { "prefetch",                    "IsPrefetchEnabled" },
            { "releasepriorityfixed",        "ReleasePriorityFixed" },
            { "sequencesoundfiletype",       "SequenceSoundFileType" },
            { "soundlimit",                  "PlayerSoundLimit" },
            { "soundsetbank",                "SoundSetBankReference" },
            { "soundsetbank1",               "SoundSetBankReference1" },
            { "soundsetbank2",               "SoundSetBankReference2" },
            { "soundsetbank3",               "SoundSetBankReference3" },
            { "surroundpan",                 "SurroundPan" },
            { "userparameter1",              "UserParameter1" },
            { "userparameter2",              "UserParameter2" },
            { "userparameter3",              "UserParameter3" },
            { "userparameter",               "UserParameter" },
            { "volume",                      "Volume" },
            { "wavearchive",                 "WaveArchiveReference" },
            { "waveencoding",                "WaveEncoding" },
        };

        // Color
        private readonly Dictionary<string, int> _colorIndices = new Dictionary<string, int>()
        {
            { "none", 0 },
            { "red", 1 },
            { "orange", 2 },
            { "yellow", 3 },
            { "green", 4 },
            { "cyan", 5 },
            { "blue", 6 },
            { "purple", 7 },
            { "pink", 8 },
        };

        // BiquadType
        private readonly Dictionary<string, string> _biquadTypes = new Dictionary<string, string>()
        {
            { "none",    "None" },
            { "inherit", "INHERIT" },
            { "lpf",     "LPF" },
            { "hpf",     "HPF" },
            { "bpf512",  "BPF512" },
            { "bpf1024", "BPF1024" },
            { "bpf2048", "BPF2048" },
            { "usr0",    "USR0" },
            { "usr1",    "USR1" },
            { "usr2",    "USR2" },
            { "usr3",    "USR3" },
            { "usr4",    "USR4" },
            { "usr5",    "USR5" },
            { "usr6",    "USR6" },
            { "usr7",    "USR7" },
            { "usr8",    "USR8" },
            { "usr9",    "USR9" },
            { "usr10",   "USR10" },
            { "usr11",   "USR11" },
            { "usr12",   "USR12" },
            { "usr13",   "USR13" },
            { "usr14",   "USR14" },
            { "usr15",   "USR15" },
            { "usr16",   "USR16" },
            { "usr17",   "USR17" },
            { "usr18",   "USR18" },
            { "usr19",   "USR19" },
            { "usr20",   "USR20" },
            { "usr21",   "USR21" },
            { "usr22",   "USR22" },
            { "usr23",   "USR23" },
            { "usr24",   "USR24" },
            { "usr25",   "USR25" },
            { "usr26",   "USR26" },
            { "usr27",   "USR27" },
            { "usr28",   "USR28" },
            { "usr29",   "USR29" },
            { "usr30",   "USR30" },
            { "usr31",   "USR31" },
            { "usr32",   "USR32" },
            { "usr33",   "USR33" },
            { "usr34",   "USR34" },
            { "usr35",   "USR35" },
            { "usr36",   "USR36" },
            { "usr37",   "USR37" },
            { "usr38",   "USR38" },
            { "usr39",   "USR39" },
            { "usr40",   "USR40" },
            { "usr41",   "USR41" },
            { "usr42",   "USR42" },
            { "usr43",   "USR43" },
            { "usr44",   "USR44" },
            { "usr45",   "USR45" },
            { "usr46",   "USR46" },
            { "usr47",   "USR47" },
            { "usr48",   "USR48" },
            { "usr49",   "USR49" },
            { "usr50",   "USR50" },
            { "usr51",   "USR51" },
            { "usr52",   "USR52" },
            { "usr53",   "USR53" },
            { "usr54",   "USR54" },
            { "usr55",   "USR55" },
            { "usr56",   "USR56" },
            { "usr57",   "USR57" },
            { "usr58",   "USR58" },
            { "usr59",   "USR59" },
            { "usr60",   "USR60" },
            { "usr61",   "USR61" },
            { "usr62",   "USR62" },
            { "usr63",   "USR63" },
        };

        // DecayCurve3D
        private readonly Dictionary<string, string> _decayCurve3Ds = new Dictionary<string, string>()
        {
            { "log", "Log" },
            { "linear", "Linear" },
        };

        // PanMode
        private readonly Dictionary<string, string> _panModes = new Dictionary<string, string>()
        {
            { "dual",    "Dual" },
            { "balance", "Balance" },
        };

        // PanCurve
        private readonly Dictionary<string, string> _panCurves = new Dictionary<string, string>()
        {
            { "linear(-6db)",       "Linear(-6dB)" },
            { "linear(0db)",        "Linear(0dB)" },
            { "linear(0db,clamp)",  "Linear(0dB,clamp)" },
            { "sincos(-3db)",       "Sin/Cos(-3dB)" },
            { "sincos(0db)",        "Sin/Cos(0dB)" },
            { "sincos(0db,clamp)",  "Sin/Cos(0dB,clamp)" },
            { "sqrt(-3db)",         "Sqrt(-3dB)" },
            { "sqrt(0db)",          "Sqrt(0dB)" },
            { "sqrt(0db,clamp)",    "Sqrt(0dB,clamp)" },
        };

        // SequenceSoundFileType
        private readonly Dictionary<string, string> _sequenceSoundFileTypes = new Dictionary<string, string>()
        {
            { "text", "Text" },
            { "smf",  "Smf" },
        };

        // GroupOutputType
        private readonly Dictionary<string, string> _groupOutputTypes = new Dictionary<string, string>()
        {
            { "embedding", "Embedding" },
            { "link",      "Link" },
            { "user",      "UserManagement" },
            { "none",      "None" },
        };

        // InstrumentEnvelopeMode
        private readonly Dictionary<string, string> _instrumentEnvelopeModes = new Dictionary<string, string>()
        {
            { "instrument", "Instrument" },
            { "region", "VelocityRegion" },
        };

        // WaveArchiveLoadType
        private readonly Dictionary<string, string> _waveArchiveLoadTypes = new Dictionary<string, string>()
        {
            { "true", "Individual" },
            { "false", "Whole" },
        };


        // WaveEncoding
        private readonly Dictionary<string, string> _waveEncodings = new Dictionary<string, string>()
        {
            { "adpcm", "ADPCM" },
            { "pcm16", "PCM16" },
        };

        // WaveArchiveReference
        private readonly Dictionary<string, string> _waveArchiveReferences = new Dictionary<string, string>()
        {
            { "autoshared", "(AutoShared)" },
            { "autoindividual", "(AutoIndividual)" },
        };

    }
}
