﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System.ComponentModel;
    using Core.Parameters;
    using ToolDevelopmentKit;

    /// <summary>
    /// コンポーネントへの参照を管理します。
    /// </summary>
    public class ComponentReference : IParameterProvider, INotifyPropertyChanged
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ComponentReference()
        {
            this.Parameters = new ParameterDictionary();
            this.InitializeParameters();
        }

        /// <summary>
        /// プロパティ値が変更されたときに発生します。
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// パラメータの辞書を取得します。
        /// </summary>
        public IParameterDictionary Parameters { get; private set; }

        /// <summary>
        /// 対象コンポーネントの名前を取得または設定します。
        /// </summary>
        public string TargetName
        {
            get { return this.GetParameterValue<string>(ProjectParameterNames.TargetName); }
            set
            {
                if (this.TargetName == value)
                {
                    return;
                }

                this.SetParameterValue<string>(ProjectParameterNames.TargetName, value);

                this.OnPropertyChanged(
                    new PropertyChangedEventArgs(ProjectParameterNames.TargetName));
            }
        }

        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected virtual void InitializeParameters()
        {
            this.Parameters.AddValue(
                ProjectParameterNames.TargetName, new TextParameterValue(string.Empty));
            this.Parameters.ParameterValueChanged += OnParameterValueChanged;
        }

        /// <summary>
        /// パラメータ値が変更されたときに発生します。
        /// </summary>
        protected void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            if (e.Key == ProjectParameterNames.TargetName)
            {
                OnPropertyChanged(new PropertyChangedEventArgs(ProjectParameterNames.TargetName));
            }
        }


        /// <summary>
        /// プロパティ値が変更されたときに発生します。
        /// </summary>
        protected virtual void OnPropertyChanged(PropertyChangedEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.PropertyChanged != null)
            {
                this.PropertyChanged(this, e);
            }
        }

        /// <summary>
        /// パラメータの値を取得します。
        /// </summary>
        /// <typeparam name="TValue">値の型を指定します。</typeparam>
        /// <param name="key">パラメータのキーを指定します。</param>
        /// <returns>パラメータの値を返します。</returns>
        private TValue GetParameterValue<TValue>(string key)
        {
            Assertion.Argument.NotNull(key);
            return (TValue)this.Parameters.GetValue(ProjectParameterNames.TargetName).Value;
        }

        /// <summary>
        /// パラメータの値を設定します。
        /// </summary>
        /// <typeparam name="TValue">値の型を指定します。</typeparam>
        /// <param name="key">パラメータのキーを指定します。</param>
        /// <param name="value">パラメータの値を指定します。</param>
        private void SetParameterValue<TValue>(string key, TValue value)
        {
            Assertion.Argument.NotNull(key);
            this.Parameters.GetValue(ProjectParameterNames.TargetName).Value = value;
        }
    }
}
