﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Core.Parameters;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// コンポーネント生成とパラメータの初期値に関するサービスです。
    /// </summary>
    public class CreateComponentService
    {
        private readonly DefaultParamManager _defaultParamManager;
        private SoundComponentTraits _componentTraits;

        public CreateComponentService(SoundComponentTraits traits)
        {
            _componentTraits = traits;
            _defaultParamManager = new DefaultParamManager(this.CreateComponent);
        }

        // <summary>
        // コンポーネントのパラメータの初期値設定を有効にします。
        // </summary>
        public bool EnabledDefaultParameter
        {
            get
            {
                return _defaultParamManager.Enabled;
            }
            set
            {
                _defaultParamManager.Enabled = value;
            }
        }

        /// <summary>
        /// パラメータ初期値設定ファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">パラメータ初期値設定ファイルのパス</param>
        public void LoadDefaultParametersFile(string filePath)
        {
            _defaultParamManager.LoadFile(filePath);
        }

        /// <summary>
        /// パラメータ初期値設定ファイルを出力します。
        /// </summary>
        /// <param name="filePath">パラメータ初期値設定ファイルのパス</param>
        public void OutputDefaultParametersFile(string filePath)
        {
            _defaultParamManager.OutputFile(filePath);
        }

        // <summary>
        // パラメータの初期値を設定を反映したコンポーネントを生成します。
        // </summary>
        public T Create<T>() where T : Component
        {
            Debug.Assert(_componentTraits != null);

            Component component = this.CreateComponent(typeof(T));

            _defaultParamManager.ApplyDefaultParameters(component);

            return (T)component;
        }

        // <summary>
        // パラメータの値の検証に使用する Component を生成します。
        // </summary>
        private Component CreateComponent(Type type)
        {
            Component component = null;

            if (type == typeof(Instrument))
            {
                component = new Instrument(); // Instrument は現時点では Factory を使用していない。
            }
            else
            {
                component = _componentTraits.SoundSetComponentFactory.Create(type);
            }

            return component;
        }
    }
}
