﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;

    /// <summary>
    /// コンポーネント管理機能を提供する基本クラスです。
    /// </summary>
    public abstract class ComponentService : IDisposable
    {
        private DocumentService _documentService;
        private ComponentManager _componentManager;

        private bool _opened = false;
        private bool delayComponentsAddedEvent = false;
        private List<Component> componentsAddedList;

        private bool delayComponentsRemovedEvent = false;
        private List<Component> componentsRemovedList;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="documentService">ドキュメントサービス。</param>
        /// <param name="componentManager">コンポーネントマネージャ。</param>
        protected ComponentService(DocumentService documentService, ComponentManager componentManager)
        {
            if (null == documentService) { throw new ArgumentNullException("documentService"); }
            if (null == componentManager) { throw new ArgumentNullException("componentManager"); }

            _documentService = documentService;
            _componentManager = componentManager;

            this.componentsAddedList = new List<Component>();
            this.componentsRemovedList = new List<Component>();

            componentManager.ComponentsAdded += OnComponentsAddedInternal;
            componentManager.ComponentsRemoved += OnComponentsRemovedInternal;
            componentManager.ComponentsResetted += OnComponentsResettedInternal;
            componentManager.ComponentNameChanged += OnComponentNameChangedInternal;
        }

        ~ComponentService()
        {
            this.Close();
        }

        /// <summary>
        /// リソースが開かれると発生します。
        /// </summary>
        public event EventHandler Opened;

        /// <summary>
        /// リソースに保存されると発生します。
        /// </summary>
        public event EventHandler Saved;

        /// <summary>
        /// リソースが閉じられる前に発生します。
        /// </summary>
        public event EventHandler Closing;

        /// <summary>
        /// リソースが閉じられると発生します。
        /// </summary>
        public event EventHandler Closed;

        /// <summary>
        /// コンポーネントが追加されると発生します。
        /// </summary>
        public event ComponentEventHandler ComponentsAdded;

        /// <summary>
        /// コンポーネントが削除されると発生します。
        /// </summary>
        public event ComponentEventHandler ComponentsRemoved;

        /// <summary>
        ///
        /// </summary>
        public event ComponentEventHandler ComponentsResetted;

        /// <summary>
        /// コンポーネント名が変更されると発生します。
        /// </summary>
        public event ComponentNameChangedEventHandler ComponentNameChanged;

        /// <summary>
        /// リソースが開かれているかどうか調べます。
        /// </summary>
        public bool IsOpened
        {
            get { return this._opened; }
        }

        /// <summary>
        /// ドキュメントの一覧を取得します。
        /// </summary>
        public IReadOnlyDictionary<string, SoundDocument> Documents
        {
            get { return _componentManager.Documents; }
        }

        /// <summary>
        /// コンポーネントディクショナリを取得します。
        /// </summary>
        public ComponentDictionary ComponentDictionary
        {
            get { return _componentManager.ComponentDictionary; }
        }

        /// <summary>
        /// ドキュメントサービスを取得します。
        /// </summary>
        protected DocumentService DocumentService
        {
            get { return _documentService; }
        }

        /// <summary>
        /// コンポーネントマネージャを取得します。
        /// </summary>
        protected ComponentManager ComponentManager
        {
            get { return _componentManager; }
        }

        /// <summary>
        /// リソースを作成します。
        /// </summary>
        /// <param name="resource">リソース。</param>
        public void Create(IStreamResource resource)
        {
            if (null == resource) { throw new ArgumentNullException("resource"); }

            CreateInternal(resource);
            this._opened = true;

            try
            {
                OnOpened(EventArgs.Empty);
            }
            catch
            {
                Close();
                throw;
            }
        }

        /// <summary>
        /// リソースを開きます。
        /// </summary>
        /// <param name="resource">リソース。</param>
        public void Open(IStreamResource resource, bool readOnly = false)
        {
            if (null == resource) { throw new ArgumentNullException("resource"); }


            this.delayComponentsAddedEvent = true;
            this.componentsAddedList.Clear();
            try
            {
                if (this._opened == false)
                {

                    if (OpenInternal(resource, readOnly) == false)
                    {
                        return;
                    }
                }

                this._opened = true;

                try
                {
                    if (this.delayComponentsAddedEvent == true &&
                        0 < this.componentsAddedList.Count())
                    {
                        OnComponentsAdded(new ComponentEventArgs(this.componentsAddedList));
                    }
                    OnOpened(EventArgs.Empty);
                }
                catch
                {
                    Close();
                    throw;
                }
            }
            finally
            {
                this.delayComponentsAddedEvent = false;
                this.componentsAddedList.Clear();
            }
        }

        /// <summary>
        /// リソースに保存します。
        /// </summary>
        /// <param name="resource">リソース。</param>
        public void Save()
        {
            if (!this._opened) { return; }

            SaveInternal();

            foreach (SoundDocument document in Documents.Values)
            {
                document.UpdateComponentsDirty();
            }

            OnSaved(EventArgs.Empty);
        }

        /// <summary>
        /// リソースを閉じます。
        /// </summary>
        public void Close()
        {
            if (this._opened == false)
            {
                return;
            }

            this.delayComponentsRemovedEvent = true;
            this.componentsRemovedList.Clear();
            try
            {
                try
                {
                    OnClosing(EventArgs.Empty);
                }
                catch
                {
                }

                CloseInternal();
                this._opened = false;

                if (this.delayComponentsRemovedEvent == true &&
                    0 < this.componentsRemovedList.Count())
                {
                    OnComponentsRemoved(new ComponentEventArgs(this.componentsRemovedList));
                }

                try
                {
                    OnClosed(EventArgs.Empty);
                }
                catch
                {
                }
            }
            finally
            {
                this.delayComponentsRemovedEvent = false;
                this.componentsRemovedList.Clear();
            }
        }

        /// <summary>
        /// リソースを閉じます。
        /// </summary>
        void IDisposable.Dispose()
        {
            this.Close();
        }

        /// <summary>
        /// リソースを作成します。
        /// </summary>
        /// <param name="resource">リソース。</param>
        protected abstract void CreateInternal(IStreamResource resource);

        /// <summary>
        /// リソースを開きます。
        /// </summary>
        /// <param name="resource">リソース。</param>
        /// <returns>
        /// 成功した場合は true、キャンセルした場合は false を返します。
        /// </returns>
        protected abstract bool OpenInternal(IStreamResource resource, bool readOnly = false);

        /// <summary>
        /// リソースに保存します。
        /// </summary>
        protected abstract void SaveInternal();

        /// <summary>
        /// リソースを閉じます。
        /// </summary>
        protected abstract void CloseInternal();

        protected void BeginDelayComponentsAddEvent()
        {
            this.delayComponentsAddedEvent = true;
            this.componentsAddedList.Clear();
        }

        protected void EndDelayComponentsAddEvent()
        {
            if (this.delayComponentsAddedEvent == true &&
                0 < this.componentsAddedList.Count())
            {
                OnComponentsAdded(new ComponentEventArgs(componentsAddedList));
            }
            this.delayComponentsAddedEvent = false;
            this.componentsAddedList.Clear();
        }

        protected void CancelDelayComponentsAddEvent()
        {
            this.delayComponentsAddedEvent = false;
            this.componentsAddedList.Clear();
        }

        protected void BeginDelayComponentsRemoveEvent()
        {
            this.delayComponentsRemovedEvent = true;
            this.componentsRemovedList.Clear();
        }

        protected void EndDelayComponentsRemoveEvent()
        {
            if (this.delayComponentsRemovedEvent == true &&
                0 < this.componentsRemovedList.Count())
            {
                OnComponentsRemoved(new ComponentEventArgs(componentsRemovedList));
            }
            this.delayComponentsRemovedEvent = false;
            this.componentsRemovedList.Clear();
        }

        protected void CancelDelayComponentsRemoveEvent()
        {
            this.delayComponentsRemovedEvent = false;
            this.componentsRemovedList.Clear();
        }

        /// <summary>
        /// ドキュメントがコンポーネントサービスに含まれるかどうか調べます。
        /// </summary>
        /// <param name="filePath">ドキュメントのファイルパス。</param>
        /// <returns>含まれる場合は true、含まれない場合は false。</returns>
        protected bool ContainsDocument(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            return ((from SoundDocument document in _componentManager.Documents
                     where document.Resource.Key == filePath
                     select document).FirstOrDefault() != null);
        }

        /// <summary>
        /// リソースが開かれると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnOpened(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != Opened)
            {
                Opened(this, e);
            }
        }

        /// <summary>
        /// リソースに保存されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnSaved(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != Saved)
            {
                Saved(this, e);
            }
        }

        /// <summary>
        /// リソースが閉じられる前に発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnClosing(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != Closing)
            {
                Closing(this, e);
            }
        }

        /// <summary>
        /// リソースが閉じられると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnClosed(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != Closed)
            {
                Closed(this, e);
            }
        }

        /// <summary>
        /// コンポーネントが追加されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        protected virtual void OnComponentsAdded(ComponentEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentsAdded)
            {
                ComponentsAdded(this, e);
            }
        }

        /// <summary>
        /// コンポーネントが削除されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        protected virtual void OnComponentsRemoved(ComponentEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentsRemoved)
            {
                ComponentsRemoved(this, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void OnComponentsResetted(ComponentEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentsResetted)
            {
                ComponentsResetted(this, e);
            }
        }

        /// <summary>
        /// コンポーネント名が変更されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        protected virtual void OnComponentNameChanged(ComponentNameChangedEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ComponentNameChanged)
            {
                ComponentNameChanged(this, e);
            }
        }

        /// <summary>
        /// コンポーネントが追加されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        private void OnComponentsAddedInternal(object sender, ComponentEventArgs e)
        {
            if (this.delayComponentsAddedEvent == true)
            {
                Debug.Assert(this.delayComponentsRemovedEvent == false, "Illegal delayComponentsAddedEvent == true && delayComponentsRemovedEvent == true");
                this.componentsAddedList.AddRange(e.Components);
            }
            else
            {
                OnComponentsAdded(e);
            }
        }

        /// <summary>
        /// コンポーネントが削除されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        private void OnComponentsRemovedInternal(object sender, ComponentEventArgs e)
        {
            if (this.delayComponentsRemovedEvent == true)
            {
                Debug.Assert(this.delayComponentsAddedEvent == false, "Illegal delayComponentsRemovedEvent == true && delayComponentsAddedEvent == true");
                this.componentsRemovedList.AddRange(e.Components);
            }
            else
            {
                OnComponentsRemoved(e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnComponentsResettedInternal(object sender, ComponentEventArgs e)
        {
            OnComponentsResetted(e);
        }

        /// <summary>
        /// コンポーネント名が変更されると発生します。
        /// </summary>
        /// <param name="e">コンポーネントイベントデータ。</param>
        private void OnComponentNameChangedInternal(object sender, ComponentNameChangedEventArgs e)
        {
            OnComponentNameChanged(e);
        }
    }
}
