﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NintendoWare.SoundFoundation.Core.Collections;
using NintendoWare.SoundFoundation.Documents;

namespace NintendoWare.SoundFoundation.Projects
{
    public partial class ComponentManager
    {
        /// <summary>
        /// ドキュメント参照ディクショナリ。
        /// </summary>
        private class DocumentReferenceDictionary<TDocument> :
            Dictionary<string, DocumentReference>,
            IReadOnlyDictionary<string, TDocument>
            where TDocument : Document
        {
            IEnumerable<string> IReadOnlyDictionary<string, TDocument>.Keys
            {
                get { return base.Keys; }
            }

            /// <summary>
            /// 値の一覧を取得します。
            /// </summary>
            IEnumerable<TDocument> IReadOnlyDictionary<string, TDocument>.Values
            {
                get { return base.Values.Select(value => value.Document).Cast<TDocument>(); }
            }

            /// <summary>
            /// 指定キーに関連付けられた値を取得します。
            /// </summary>
            /// <param name="key">キー。</param>
            /// <returns>指定キーに関連付けられた値。</returns>
            TDocument IReadOnlyDictionary<string, TDocument>.this[string key]
            {
                get { return base[key].Document as TDocument; }
            }

            IEnumerator<KeyValuePair<string, TDocument>> IEnumerable<KeyValuePair<string, TDocument>>.GetEnumerator()
            {
                foreach (KeyValuePair<string, DocumentReference> item in this)
                {
                    yield return new KeyValuePair<string, TDocument>(item.Key, item.Value.Document as TDocument);
                }
            }

            public void Add(DocumentReference item)
            {
                if (null == item.Document)
                {
                    throw new ArgumentException("item.Document must not be null.");
                }

                if (!(item.Document is TDocument))
                {
                    throw new ArgumentException("invalid document type.");
                }

                if (null == item.Document.Resource)
                {
                    throw new ArgumentException("item.Document.Resource must not be null.");
                }

                base.Add(item.Document.Resource.Key, item);
            }

            bool IReadOnlyDictionary<string, TDocument>.TryGetValue(string key, out TDocument value)
            {
                DocumentReference documentRef;

                if (base.TryGetValue(key, out documentRef))
                {
                    value = (TDocument)documentRef.Document;
                    return true;
                }
                else
                {
                    value = null;
                }

                return false;
            }
        }
    }
}
