﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.Wave
{
    using System;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core.IO;

    public class WaveFileWavReader : WaveFileReader
    {
        private FileStream strm;
        private string filePath;
        private BinaryReader reader;
        private long dataBegin;
        private long dataEnd;
        private long curSample;
        private long numSampleCount;
        private int sampleByte;

        public override void Close()
        {
            strm.Close();
            this.filePath = null;
        }

        public override byte[] Read()
        {
            long readSamples = numSampleCount - curSample;
            long readSize = readSamples * sampleByte;

            byte[] buffer = reader.ReadBytes(Convert.ToInt32(readSize));

            curSample += readSamples;

            return buffer;
        }

        protected override void DisposeInternal()
        {
            if (strm != null)
            {
                strm.Close();
            }
            this.filePath = null;
        }

        protected override WaveFile OpenInternal(string filePath)
        {
            return OpenWav(filePath);
        }

        protected override WaveDataStream CreateDataStream()
        {
            if (this.strm == null)
            {
                throw new InvalidOperationException();
            }

            var fileStream = File.OpenRead(this.filePath);
            fileStream.Position = this.dataBegin;
            return new WaveDataStream(fileStream, this.dataEnd - this.dataBegin);
        }

        private WaveFileWav OpenWav(string filePath)
        {
            WaveFileWav waveFile = new WaveFileWav();

            strm = File.OpenRead(filePath);
            reader = LittleEndianBinaryReader.Create(strm, new System.Text.ASCIIEncoding());

            string fileChunk = new string(reader.ReadChars(4));
            if (fileChunk != "RIFF")
            {
                throw new WaveFileReaderException("Invalid format");
            }

            Int32 fileSize = reader.ReadInt32();
            string format = new string(reader.ReadChars(4));
            if (format != "WAVE")
            {
                throw new WaveFileReaderException("Invalid format");
            }

            int chunkTypeLength = 4; // chars[4] : 4byte
            int chunkSizeLength = 4; // uint32   : 4byte
            while (strm.Position < strm.Length)
            {
                if (strm.Length - strm.Position < chunkTypeLength) break;
                string chunkType = new string(reader.ReadChars(4));

                if (strm.Length - strm.Position < chunkSizeLength) break;
                UInt32 chunkSize = reader.ReadUInt32();

                if (strm.Length - strm.Position < chunkSize) break;
                long chunkEnd = strm.Position + chunkSize;
                long nextChunkPos = ((chunkEnd + 0x01) & ~0x01); // 2バイトアライン

                if (chunkType == "fmt ")
                {
                    waveFile.FormatChunk.Read(reader);
                    if (waveFile.FormatChunk.IsBigEndianPcm())
                    {
                        throw new WaveFileReaderException("Wave format must not be big-endian pcm.");
                    }
                }
                else if (chunkType == "smpl")
                {
                    waveFile.SamplerChunk.Read(reader);
                }
                else if (chunkType == "cue ")
                {
                    waveFile.CueChunk.Read(reader, chunkSize);
                }
                else if (chunkType == "LIST" || chunkType == "list")
                {
                    try
                    {
                        waveFile.ListChunk.Read(reader, chunkSize);
                    }
                    catch
                    {
                        throw new WaveFileReaderException("Invalid format");
                    }
                }
                else if (chunkType == "data")
                {
                    dataBegin = strm.Position;
                    dataEnd = chunkEnd;
                }
                else
                {
                    // unknown chunk
                }

                strm.Position = nextChunkPos;
            }

            sampleByte = (waveFile.SampleBit + 7) / 8;
            numSampleCount = (dataEnd - dataBegin) / sampleByte;

            waveFile.FrameCount = numSampleCount / waveFile.ChannelCount;

            curSample = 0;
            strm.Position = dataBegin;

            this.PrepareMarkers(waveFile);
            this.PrepareRegions(waveFile);

            this.filePath = filePath;
            return waveFile;
        }

        private void PrepareMarkers(WaveFileWav waveFile)
        {
            for (int labelIndex = 0; labelIndex < waveFile.ListChunk.LabelChunks.Count; ++labelIndex)
            {
                var labelChunk = waveFile.ListChunk.LabelChunks[labelIndex];

                // リージョン用ラベルでないかを確認
                bool isLabelForRegion = false;
                foreach (var labeledTextChunk in waveFile.ListChunk.LabeledTextChunks)
                {
                    if (labeledTextChunk.CuePointID == labelChunk.CuePointID)
                    {
                        if (new string(labeledTextChunk.PurposeID) == "rgn ")
                        {
                            isLabelForRegion = true;
                            break;
                        }
                    }
                }

                if (!isLabelForRegion)
                {
                    foreach (var cuePoint in waveFile.CueChunk.CuePoints)
                    {
                        if (cuePoint.ID == labelChunk.CuePointID)
                        {
                            var marker = new WaveFileWav.MarkerInfo();
                            marker.Name = labelChunk.Text;
                            marker.Position = cuePoint.Position;
                            waveFile.Markers.Add(marker);

                            break;
                        }
                    }
                }
            }
        }

        private void PrepareRegions(WaveFileWav waveFile)
        {
            for (int labeledTextIndex = 0; labeledTextIndex < waveFile.ListChunk.LabeledTextChunks.Count; ++labeledTextIndex)
            {
                var labeledTextChunk = waveFile.ListChunk.LabeledTextChunks[labeledTextIndex];

                // リージョン以外は除外
                if (new string(labeledTextChunk.PurposeID) != "rgn ")
                {
                    continue;
                }

                if (!waveFile.CueChunk.CuePoints.Contains(labeledTextChunk.CuePointID))
                {
                    continue;
                }

                var cuePoint = waveFile.CueChunk.CuePoints[labeledTextChunk.CuePointID];
                var newRegion = new WaveFileWav.RegionInfo();

                newRegion.Name = this.GetCuePointLabel(waveFile, labeledTextChunk.CuePointID);
                newRegion.StartFrame = cuePoint.Position;
                newRegion.EndFrame = cuePoint.Position + labeledTextChunk.SampleLength;

                waveFile.Regions.Add(newRegion);
            }
        }

        private string GetCuePointLabel(WaveFileWav waveFile, uint cuePointID)
        {
            if (!waveFile.CueChunk.CuePoints.Contains(cuePointID))
            {
                return string.Empty;
            }

            var targetLabel = waveFile.ListChunk.LabelChunks.Where(label => label.CuePointID == cuePointID).FirstOrDefault();

            if (targetLabel == null)
            {
                return string.Empty;
            }

            return targetLabel.Text;
        }
    }

}

