﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.Wave
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.FileFormats.Audio;

    public abstract class WaveFileReader : IDisposable
    {
        private enum Format
        {
            UnSupported,
            AIFF,
            WAV,
            AAC,
        }

        private bool disposed = false;

        protected class WaveDataStream : Stream
        {
            private readonly long offset;
            private readonly long length;
            private FileStream fileStream;

            public WaveDataStream(FileStream fileStream, long length)
            {
                this.fileStream = fileStream;
                this.offset = fileStream.Position;
                this.length = length;
            }

            public override bool CanRead
            {
                get { return true; }
            }

            public override bool CanWrite
            {
                get { return false; }
            }

            public override bool CanSeek
            {
                get { return true; }
            }

            public override long Length
            {
                get { return this.length; }
            }

            public override long Position
            {
                get
                {
                    return this.fileStream.Position - this.offset;
                }
                set
                {
                    this.fileStream.Position = value + this.offset;
                }
            }

            public override void Close()
            {
                base.Close();

                if (this.fileStream != null)
                {
                    this.fileStream.Dispose();
                    this.fileStream = null;
                }
            }

            public override void Flush()
            {
                this.fileStream.Flush();
            }

            public override int Read(byte[] buffer, int offset, int count)
            {
                int readCount = Math.Min((int)(this.length - this.Position), count);

                if (readCount <= 0)
                {
                    return 0;
                }
                else
                {
                    return this.fileStream.Read(buffer, offset, readCount);
                }
            }

            public override void Write(byte[] buffer, int offset, int count)
            {
                throw new NotImplementedException();
            }

            public override long Seek(long offset, SeekOrigin origin)
            {
                switch (origin)
                {
                    case SeekOrigin.Begin:
                        return this.fileStream.Seek(offset + this.offset, origin);

                    case SeekOrigin.End:
                        throw new NotImplementedException();

                    case SeekOrigin.Current:
                        return this.fileStream.Seek(offset, origin);
                }

                throw new NotImplementedException();
            }

            public override void SetLength(long value)
            {
                throw new NotImplementedException();
            }
        }

        ~WaveFileReader()
        {
            Dispose();
        }

        public delegate WaveFileReader CreateWaveFileOtherReaderDelegate(string filePath);
        public static CreateWaveFileOtherReaderDelegate CreateWaveFileOtherReader;

        public static WaveFileReader CreateInstance(string filePath)
        {
            if (CreateWaveFileOtherReader != null)
            {
                var reader = CreateWaveFileOtherReader(filePath);
                if (reader != null)
                {
                    return reader;
                }
            }

            Format format = GuessFileFormat(filePath);

            switch (format)
            {
                case Format.WAV:
                    return new WaveFileWavReader();
                case Format.AIFF:
                    return new WaveFileAiffReader();
                case Format.AAC:
                    return new WaveFileAACReader();
                default:
                    throw new WaveFileReaderException("Not supported file format \"" + filePath + "\"");
            }
        }

        public event LoopEndFrameModifiedEventHandler LoopEndFrameModified;
        public delegate void LoopEndFrameModifiedEventHandler(object sender,
                                                              LoopEndFrameModifiedEventArgs e);

        public WaveFile Open(string filePath)
        {
            WaveFile waveFile = OpenInternal(filePath);

            // ループエンド位置をチェックする。
            // ループエンドには、フレーム数 + 1 までを格納できる。
            if (waveFile.FrameCount + 1 < waveFile.LoopEndFrame)
            {
                LoopEndFrameModifiedEventArgs args =
                    new LoopEndFrameModifiedEventArgs()
                    {
                        FilePath = filePath,
                        LoopEndFrame = waveFile.LoopEndFrame,
                        FrameCount = waveFile.FrameCount,
                    };

                // ループエンド位置をサンプルエンド位置にする。
                waveFile.LoopEndFrame = waveFile.FrameCount;

                OnLoopEndFrameModified(this, args);
            }

            // 再生時間を求めます。
            double frameCount = (double)(waveFile.IsLoop ? waveFile.LoopEndFrame : waveFile.FrameCount);
            double time = frameCount / waveFile.SampleRate;
            waveFile.WaveTime = time * 1000; // 秒をミリ秒単位に変換

            return waveFile;
        }

        public abstract void Close();

        public abstract byte[] Read();

        public Stream OpenDataStream()
        {
            return this.CreateDataStream();
        }

        void IDisposable.Dispose()
        {
            Dispose();
        }

        protected abstract WaveFile OpenInternal(string filePath);

        protected abstract void DisposeInternal();

        protected abstract WaveDataStream CreateDataStream();

        private static Format GuessFileFormat(string filePath)
        {
            using (FileStream strm = File.OpenRead(filePath))
            {
                BinaryReader reader = new BinaryReader(strm);

                string fileChunk = new string(reader.ReadChars(4));
                reader.ReadInt32(); // skip
                string format = new string(reader.ReadChars(4));

                if (fileChunk == "RIFF")
                {
                    if (format == "WAVE") return Format.WAV;
                }
                else if (fileChunk == "FORM")
                {
                    if (format == "AIFF") return Format.AIFF;
                }
            }

            using (FileStream strm = File.OpenRead(filePath))
            {
                AdtsHeader adtsHeader = AdtsHeader.Parse(strm);
                if (adtsHeader != null && adtsHeader.IsValid() == true)
                {
                    return Format.AAC;
                }
            }

            return Format.UnSupported;
        }

        private void Dispose()
        {
            if (this.disposed) { return; }

            DisposeInternal();
            this.disposed = true;

            GC.SuppressFinalize(this);
        }

        private void OnLoopEndFrameModified(object send, LoopEndFrameModifiedEventArgs e)
        {
            if (null != LoopEndFrameModified)
            {
                LoopEndFrameModified(this, e);
            }
        }
    }

    public class LoopEndFrameModifiedEventArgs : EventArgs
    {
        public string FilePath { get; set; }
        public long LoopEndFrame { get; set; }
        public long FrameCount { get; set; }
    }

    internal class WaveFileImpl
    {
        public delegate Int16[] ToInt16Handler(byte[] buffer);

        public static Int16[] S8ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 1, i++)
            {
                array[i] = (Int16)((sbyte)buffer[pos] << 8);
            }

            return array;
        }
        public static Int16[] U8ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 1, i++)
            {
                array[i] = (Int16)(((short)buffer[pos] - 0x80) << 8);
            }

            return array;
        }
        public static Int16[] BigS16ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 2];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 2, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] LittleS16ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 2];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 2, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos + 1] << 8) | buffer[pos]);
            }

            return array;
        }
        public static Int16[] BigS24ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 3];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 3, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] LittleS24ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 3];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 3, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos + 2] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] BigS32ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 4];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 4, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] LittleS32ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 4];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 4, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos + 3] << 8) | buffer[pos + 2]);
            }

            return array;
        }
    }

    internal class WaveFileReaderException : WaveFileFormatException
    {
        public WaveFileReaderException(string s) : base(s) { }
    }
}
