﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.IO;
    using Core.Parameters;
    using Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Conversion;

    public abstract class Model2XmlTranslator
    {
        private Converter converter = new Converter();
        private PathResolver pathResolver = new PathResolver();

        private Model2XmlTranslationContext context;
        private CommonItemParamTranslator itemParamTranslator;

        protected Model2XmlTranslator(CommonItemParamTranslator itemParamTranslator)
        {
            Ensure.Argument.NotNull(itemParamTranslator);

            this.itemParamTranslator = itemParamTranslator;
            this.itemParamTranslator.PathResolver = pathResolver;
        }

        public PathResolver PathResolver
        {
            get { return this.pathResolver; }
        }

        protected CommonItemParamTranslator ItemParamTranslator
        {
            get { return this.itemParamTranslator; }
        }

        protected TDest Run<TDest>(object value)
            where TDest : class
        {
            this.context = new Model2XmlTranslationContext();

            return this.converter.Convert(value, typeof(TDest)) as TDest;
        }

        protected Model2XmlTranslationContext Context
        {
            get { return this.context; }
        }

        protected void AddConverter<TSrc, TDest>()
            where TSrc : Component
            where TDest : XmlComponent, new()
        {
            this.converter.TypeConverters.Add(
                new TypeCouple(typeof(TSrc), typeof(TDest)),
                this.ComponentToXmlComponent<TSrc, TDest>);
        }

        protected void AddConverter<TSrc, TDest>(ConvertHandler handler)
        {
            Assertion.Argument.NotNull(handler);
            this.converter.TypeConverters.Add(new TypeCouple(typeof(TSrc), typeof(TDest)), handler);
        }

        protected void AddDefaultConverter<TSrc, TDest>()
        {
            this.converter.TypeConverters.Add(
                new TypeCouple(typeof(TSrc), typeof(TDest)), DefaultConverters.ConvertClass);
        }

        protected XmlComponent TranslateComponent<TComponent, TXmlComponent>(
            TComponent src, Type dstType, Converter converter)
            where TComponent : Component
            where TXmlComponent : XmlComponent, new()
        {
            Ensure.Argument.NotNull(src);
            Ensure.Argument.NotNull(converter);

            TXmlComponent dest = new TXmlComponent();

            if (src.Name != null && src.Name.Length > 0)
            {
                dest.Name = src.Name;
            }

            if (dest is IXmlParameterProvider)
            {
                this.TranslateParameters(src, dest as IXmlParameterProvider, converter);
            }

            if (dest is IXmlComponentParent)
            {
                this.TranslateChildren(src, dest as IXmlComponentParent, converter);
            }

            return dest;
        }

        protected XmlComponent TranslateComponentWithoutName<TComponent, TXmlComponent>(
            TComponent src, Type dstType, Converter converter)
            where TComponent : Component
            where TXmlComponent : XmlComponent, new()
        {
            Ensure.Argument.NotNull(src);
            Ensure.Argument.NotNull(converter);

            TXmlComponent dest = new TXmlComponent();

            if (dest is IXmlParameterProvider)
            {
                this.TranslateParameters(src, dest as IXmlParameterProvider, converter);
            }

            if (dest is IXmlComponentParent)
            {
                this.TranslateChildren(src, dest as IXmlComponentParent, converter);
            }

            return dest;
        }

        protected void TranslateParameters(
            IParameterProvider src, IXmlParameterProvider dest, Converter converter)
        {
            this.TranslateParameters(src, src.Parameters.Keys, dest, converter);
        }

        protected void TranslateParameters(
            IParameterProvider src, string[] srcParamNames, IXmlParameterProvider dest, Converter converter)
        {
            Assertion.Argument.NotNull(src);
            Assertion.Argument.NotNull(srcParamNames);
            Assertion.Argument.NotNull(dest);
            Assertion.Argument.NotNull(converter);

            foreach (string srcParamName in srcParamNames)
            {
                if (!src.Parameters.ContainsKey(srcParamName)) { continue; }
                this.itemParamTranslator.Model2Xml(srcParamName, src.Parameters[srcParamName], dest);
            }
        }

        protected void TranslateChildren(Component src, IXmlComponentParent dest, Converter converter)
        {
            TranslateChildren(src, dest, null, converter);
        }

        protected void TranslateChildren(
            Component src, IXmlComponentParent dest, Type destType, Converter converter)
        {
            Assertion.Argument.NotNull(src);
            Assertion.Argument.NotNull(dest);
            Assertion.Argument.NotNull(converter);

            foreach (Component item in src.Children)
            {
                XmlComponent destItem = null;

                if (destType == null)
                {
                    destItem = converter.Convert(item, destType) as XmlComponent;
                }
                else
                {
                    TypeCouple typeCouple = new TypeCouple(item.GetType(), destType);

                    if (converter.TypeConverters.ContainsKey(typeCouple))
                    {
                        destItem = converter.TypeConverters[typeCouple](item, destType, converter) as XmlComponent;
                    }
                }

                if (destItem == null)
                {
                    throw new Exception("invalid type");
                }

                if (dest.Items == null)
                {
                    dest.Items = new List<XmlComponent>();
                }

                dest.Items.Add(destItem);
            }
        }

        /// <summary>
        /// Component から XmlComponent に変換します。
        /// </summary>
        /// <typeparam name="TComponent">変換元の型を指定します。</typeparam>
        /// <typeparam name="TXmlComponent">変換後の型を指定します。</typeparam>
        /// <param name="src">変換元のオブジェクトを指定します。</param>
        /// <param name="dstType">変換後の型を指定します。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>変換後のオブジェクトを返します。</returns>
        protected object ComponentToXmlComponent<TComponent, TXmlComponent>(
            object src, Type dstType, Converter converter)
            where TComponent : Component
            where TXmlComponent : XmlComponent, new()
        {
            return this.TranslateComponent<TComponent, TXmlComponent>(
                src as TComponent, dstType, converter);
        }

        /// <summary>
        /// Component から XmlComponent に変換します。
        /// </summary>
        /// <typeparam name="TComponent">変換元の型を指定します。</typeparam>
        /// <typeparam name="TXmlComponent">変換後の型を指定します。</typeparam>
        /// <param name="src">変換元のオブジェクトを指定します。</param>
        /// <param name="dstType">変換後の型を指定します。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>変換後のオブジェクトを返します。</returns>
        protected object ComponentToXmlComponentWithoutName<TComponent, TXmlComponent>(
            object src, Type dstType, Converter converter)
            where TComponent : Component
            where TXmlComponent : XmlComponent, new()
        {
            return this.TranslateComponentWithoutName<TComponent, TXmlComponent>(
                src as TComponent, dstType, converter);
        }
    }
}
