﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Documents
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class DocumentServiceTraits : IDocumentServiceTraits
    {
        private DocumentFactoryCollection _DocumentFactorys = null;
        private DocumentReaderCollection _DocumentReaders = null;
        private DocumentWriterCollection _DocumentWriters = null;
        private DocumentConverterCollection _DocumentConverters = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentServiceTraits()
        {
            _DocumentFactorys = new DocumentFactoryCollection();
            _DocumentReaders = new DocumentReaderCollection();
            _DocumentWriters = new DocumentWriterCollection();
            _DocumentConverters = new DocumentConverterCollection();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentFactoryCollection DocumentFactorys
        {
            get { return _DocumentFactorys; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentReaderCollection DocumentReaders
        {
            get { return _DocumentReaders; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentWriterCollection DocumentWriters
        {
            get { return _DocumentWriters; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentConverterCollection DocumentConverters
        {
            get { return _DocumentConverters; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string[] SupportedDocumentTypeNames
        {
            get
            {
                return (from factory in DocumentFactorys
                        select factory.TypeName).ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IDocumentFactory GetDocumentFactory(string typeName)
        {
            if (!DocumentFactorys.Contains(typeName)) { return null; }
            return DocumentFactorys[typeName];
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IDocumentReader GetDocumentReader(IStreamResource resource)
        {
            Exception exception = null;

            foreach (IDocumentReader reader in DocumentReaders)
            {
                try
                {
                    if (reader.IsSupported(resource) == true)
                    {
                        return reader;
                    }
                }
                catch (FileNotSupportedVersionException e)
                {
                    if (exception == null)
                    {
                        exception = e;
                    }
                }
                catch (Exception e)
                {
                    exception = e;
                }
            }

            if (exception != null)
            {
                FileNotSupportedVersionException e = exception as FileNotSupportedVersionException;

                if (null != e)
                {
                    throw new FileNotSupportedVersionException(resource.Key, e.Version);
                }
                else
                {
                    throw exception;
                }
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IDocumentReader GetDocumentReader(string typeName)
        {
            var readers = DocumentReaders
              .Where(r => r.DocumentTypeName == typeName);
            return readers.FirstOrDefault();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IDocumentWriter GetDocumentWriter(string typeName)
        {
            var writers = DocumentWriters
              .Where(w => w.DocumentTypeName == typeName);
            return writers.FirstOrDefault();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IDocumentConverter GetDocumentConverter(string inputDocumentTypeName, string outputDocumentTypeName)
        {
            var converters = DocumentConverters
              .Where(c =>
                     c.InputDocumentTypeName == inputDocumentTypeName &&
                     c.OutputDocumentTypeName == outputDocumentTypeName);
            return converters.FirstOrDefault();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IDocumentConverter[] FindDocumentConverter(string inputDocumentTypeName)
        {
            return DocumentConverters
              .Where(c =>
                     c.InputDocumentTypeName == inputDocumentTypeName).ToArray();
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class DocumentFactoryCollection : KeyedCollection<string, IDocumentFactory>
    {
        protected override string GetKeyForItem(IDocumentFactory item)
        {
            return item.TypeName;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class DocumentReaderCollection : List<IDocumentReader>
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class DocumentWriterCollection : List<IDocumentWriter>
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class DocumentConverterCollection : List<IDocumentConverter>
    {
    }
}
