﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//#define DOCUMENTREFERENCE_LEAKCHECK

namespace NintendoWare.SoundFoundation.Documents
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core.Resources;
    using ToolDevelopmentKit;

    public class DocumentManager
    {
        private Dictionary<string, List<DocumentReference>> _Dictionary;
        public event EventHandler DocumentAdded;
        public event EventHandler DocumentRemoved;

        public DocumentManager()
        {
            _Dictionary = new Dictionary<string, List<DocumentReference>>();
        }

        /// <summary>
        /// ドキュメントが開かれると発生します。
        /// </summary>
        public event EventHandler<DocumentEventArgs> DocumentOpened;

        /// <summary>
        /// ドキュメントの一覧を取得します。
        /// </summary>
        public Document[] Documents
        {
            get
            {
                return (from List<DocumentReference> documentList in _Dictionary.Values
                        where 0 < documentList.Count
                        select documentList[0].Document).ToArray<Document>();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentReference Add(Document document)
        {
            DocumentReference docRef;

            document.ResourceChanged += new EventHandler(this.ResourceChanged);
            docRef = _Add(document);
            OnDocumentAdded(new ResourceEventArgs(document.Resource));

            return docRef;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Remove(DocumentReference docRef)
        {
            IStreamResource resource = docRef.Document.Resource;

            if (_Remove(docRef) == true)
            {
                OnDocumentRemoved(new ResourceEventArgs(resource));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool IsDocumentExists(string resourceKey)
        {
            return _Dictionary.ContainsKey(resourceKey) && 0 < _Dictionary[resourceKey].Count;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentReference[] Find(string resourceKey)
        {
            DocumentReference[] docRefArray = new DocumentReference[0];
            List<DocumentReference> docRefList = null;

            if (_Dictionary.TryGetValue(resourceKey, out docRefList) == true)
            {
                docRefArray = docRefList.ToArray();
            }

            return docRefArray;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public DocumentReference[] this[string resourceKey]
        {
            get
            {
                return Find(resourceKey);
            }
        }

        /// <summary>
        /// ドキュメントが開かれると発生します。
        /// </summary>
        /// <param name="e">イベントデータを指定します。</param>
        protected virtual void OnDocumentOpened(DocumentEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.DocumentOpened != null)
            {
                this.DocumentOpened(this, e);
            }
        }

        private DocumentReference _Add(Document document)
        {
            DocumentReference docRef = new DocumentReference(this);
            docRef.Document = document;

            _Add(docRef);

            return docRef;
        }

        private void _Add(DocumentReference docRef)
        {
            Debug.Assert(null != docRef, "unexpected error");
            Debug.Assert(null != docRef.Document, "unexpected error");
            Debug.Assert(null != docRef.Document.Resource, "unexpected error");

            List<DocumentReference> docRefList = null;

            if (_Dictionary.TryGetValue(docRef.Document.Resource.Key, out docRefList) == false)
            {
                docRefList = new List<DocumentReference>();
                _Dictionary.Add(docRef.Document.Resource.Key, docRefList);
            }

            docRefList.Add(docRef);

#if DEBUG
#if DOCUMENTREFERENCE_LEAKCHECK
            System.Diagnostics.Debug.WriteLine(
                string.Format("[Debug/DocumentReferenceLeak] Open  : RefCount={0} FilePath={1}",
                               _Dictionary[docRef.Document.Resource.Key].Count, docRef.Document.Resource.Key));
#endif
#endif
        }

        private bool _Remove(DocumentReference docRef)
        {
            bool retValue = false;
            List<DocumentReference> docRefList = null;
            string key = docRef.Document.Resource.Key;

            if (_Dictionary.TryGetValue(key, out docRefList) == true && docRefList.Count > 0)
            {

                retValue = docRefList.Remove(docRef);

                if (retValue)
                {
                    if (0 == _Dictionary[key].Count)
                    {
                        docRef.Document.Close();
                    }

                    docRef.Document = null;
                }

            }

#if DEBUG
#if DOCUMENTREFERENCE_LEAKCHECK
            System.Diagnostics.Debug.WriteLine(
                string.Format( "[Debug/DocumentReferenceLeak] Close : RefCount={0} FilePath={1}",
                               _Dictionary.ContainsKey( key ) ? _Dictionary[ key ].Count : 0, key ) );
#endif
#endif

            return retValue;
        }

        private void ResourceChanged(object sender, EventArgs e)
        {
            UpdateDictionary((Document)sender);
        }

        private void UpdateDictionary(Document document)
        {
            List<DocumentReference> addList = new List<DocumentReference>();

            foreach (List<DocumentReference> list in _Dictionary.Values)
            {
                DocumentReference[] array = list.ToArray();
                foreach (DocumentReference docRef in array)
                {
                    if (docRef.Document == document)
                    {
                        addList.Add(docRef);
                        list.Remove(docRef);
                    }
                }
            }

            foreach (DocumentReference docRef in addList)
            {
                _Add(docRef);
            }
        }

        private void OnDocumentAdded(ResourceEventArgs e)
        {
            if (DocumentAdded != null)
            {
                DocumentAdded(this, e);
            }
        }

        private void OnDocumentRemoved(ResourceEventArgs e)
        {
            if (DocumentRemoved != null)
            {
                DocumentRemoved(this, e);
            }
        }
    }
}
