﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace NintendoWare.SoundFoundation.Core
{
    /// <summary>
    /// イベントの種類 と RoutingEventBindingList のコレクションです。
    /// </summary>
    public class RoutingEventBindingMap
    {
        private Dictionary<Type, Collection<RoutingEventBinding>> _bindingMap = new Dictionary<Type, Collection<RoutingEventBinding>>();

        /// <summary>
        /// 追加されているイベントの種類の一覧を取得します。
        /// </summary>
        public Type[] Keys
        {
            get { return _bindingMap.Keys.ToArray(); }
        }

        /// <summary>
        /// 指定されたイベントの種類に関連づいた RoutingEventBinding の一覧を取得します。
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        public RoutingEventBinding[] this[Type type]
        {
            get { return _bindingMap[type].ToArray(); }
        }

        /// <summary>
        /// 指定されたイベントの種類がマップに含まれているかどうかを調べます。
        /// </summary>
        /// <param name="type">イベントの種類。</param>
        /// <returns>含まれている場合は true、含まれていない場合は false。</returns>
        public bool ContainsKey(Type type)
        {
            return _bindingMap.ContainsKey(type);
        }

        /// <summary>
        /// 新しい RoutingEventBinding を追加します。
        /// </summary>
        /// <param name="binding">ルーティングイベントバインディング。</param>
        public void Add(RoutingEventBinding binding)
        {
            if (null == binding) { throw new ArgumentNullException("binding"); }

            if (!_bindingMap.ContainsKey(binding.EventType))
            {
                _bindingMap.Add(binding.EventType, new Collection<RoutingEventBinding>());
            }

            _bindingMap[binding.EventType].Add(binding);
        }

        /// <summary>
        /// 既存の RoutingEventBinding をマップから削除します。
        /// </summary>
        /// <param name="binding">ルーティングイベントバインディング。</param>
        public void Remove(RoutingEventBinding binding)
        {
            if (null == binding) { throw new ArgumentNullException("binding"); }

            if (!_bindingMap.ContainsKey(binding.EventType)) { return; }
            _bindingMap[binding.EventType].Remove(binding);

            if (0 == _bindingMap[binding.EventType].Count)
            {
                _bindingMap.Remove(binding.EventType);
            }
        }
    }
}
