﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core
{
    using System.Collections.Generic;
    using ToolDevelopmentKit;

    /// <summary>
    /// キーオブジェクトファクトリを集約します。
    /// </summary>
    /// <typeparam name="Parameterized">キーを指定します。</typeparam>
    /// <typeparam name="TParam">オブジェクトのインスタンス生成に使用するパラメータの型を指定します。</typeparam>
    /// <typeparam name="TValue">生成するオブジェクトの型を指定します。</typeparam>
    public class KeyedObjectAggregateFactory<TKey, TParam, TValue>
    {
        private Dictionary<TKey, IObjectFactory<TParam, TValue>> factories =
            new Dictionary<TKey, IObjectFactory<TParam, TValue>>();

        /// <summary>
        /// 内包するオブジェクトファクトリのディクショナリを取得します。
        /// </summary>
        public IDictionary<TKey, IObjectFactory<TParam, TValue>> Factories
        {
            get { return this.factories; }
        }

        /// <summary>
        /// 指定されたキーからオブジェクトのインスタンスを作成します。
        /// </summary>
        /// <param name="key">キーを指定します。</param>
        /// <param name="param">オブジェクトのインスタンス生成に使用するパラメータを指定します。</param>
        /// <returns>生成したオブジェクトのインスタンスを返します。</returns>
        public TValue Create(TKey key, TParam param)
        {
            Ensure.Argument.NotNull(key);
            Ensure.Argument.NotNull(key);

            IObjectFactory<TParam, TValue> factory = null;
            this.factories.TryGetValue(key, out factory);

            if (factory == null)
            {
                return this.CreateDefaultValue(param);
            }

            return factory.Create(param);
        }

        /// <summary>
        /// デフォルト値を作成します。
        /// 指定されたキーが見つからなかった場合に実行されます。
        /// </summary>
        /// <param name="param">オブジェクトのインスタンス生成に使用するパラメータを指定します。</param>
        /// <returns>デフォルト値を返します。</returns>
        protected virtual TValue CreateDefaultValue(TParam param)
        {
            return default(TValue);
        }
    }
}
