﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core.Collections
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;

    public class NumberedListDecorator<TItem> :
        KeyedListDecorator<TItem, TItem>, INotifyCollectionChanged
    {
        private Dictionary<TItem, int> numberDictionary = new Dictionary<TItem, int>();

        public NumberedListDecorator(IList<TItem> list)
            : base(list, GetKeyFromItem)
        {
            Ensure.Argument.NotNull(list);

            if (list is INotifyCollectionChanged)
            {
                (list as INotifyCollectionChanged).CollectionChanged += OnCollectionChangedInternal;
            }
        }

        /// <summary>
        /// 指定アイテムのインデックスを取得します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        /// <returns>
        /// アイテムがリストに存在する場合は、アイテムのインデックスを返します。
        /// それ以外の場合は、-1を返します。
        /// </returns>
        public override int IndexOf(TItem item)
        {
            if (this.Count == 0) { return -1; }

            if (this.numberDictionary.Count == 0)
            {
                RenumberItems();
            }

            if (!this.numberDictionary.ContainsKey(item))
            {
                return -1;
            }

            return this.numberDictionary[item];
        }

        /// <summary>
        /// アイテムを削除します。
        /// </summary>
        /// <param name="item">アイテム。</param>
        public override bool Remove(TItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            int index = this.IndexOf(item);
            if (-1 == index) { return false; }

            RemoveAt(index);
            return true;
        }

        /// <summary>
        /// すべてのアイテムに対して番号の再割り当てを行います。
        /// </summary>
        public void RenumberItems()
        {
            ClearNumbers();
            NumberItems(this, 0);
        }

        /// <summary>
        /// アイテムからキーを取得します。
        /// </summary>
        /// <param name="item">アイテムを指定します。</param>
        /// <returns>キーを返します。</returns>
        private static TItem GetKeyFromItem(TItem item)
        {
            return item;
        }

        /// <summary>
        /// アイテムに番号を割り当てます。
        /// </summary>
        /// <param name="items">アイテムの列挙子を指定します。</param>
        /// <param name="index">アイテムの開始番号を指定します。</param>
        private void NumberItems(IEnumerable items, int number)
        {
            Assertion.Argument.NotNull(items);

            int itemNumber = number;

            foreach (object item in items)
            {
                this.numberDictionary.Add((TItem)item, itemNumber);
                itemNumber++;
            }
        }

        /// <summary>
        /// すべての番号をクリアします。
        /// </summary>
        private void ClearNumbers()
        {
            this.numberDictionary.Clear();
        }

        /// <summary>
        /// アイテムに最後尾に追加されたかどうかを調べます。
        /// </summary>
        /// <param name="items">追加されたアイテムのリストを指定します。</param>
        /// <param name="addIndex">追加された位置を指定します。</param>
        private bool IsAddedLast(IList items, int addIndex)
        {
            Assertion.Argument.NotNull(items);
            return items.Count + addIndex < this.Count;
        }

        /// <summary>
        /// 委譲先コレクションが変更されると発生します。
        /// </summary>
        /// <param name="e">コレクション変更イベントデータ。</param>
        private void OnCollectionChangedInternal(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    if (IsAddedLast(e.NewItems, e.NewStartingIndex))
                    {
                        NumberItems(e.NewItems, e.NewStartingIndex);
                    }
                    else
                    {
                        ClearNumbers();
                    }
                    break;

                default:
                    ClearNumbers();
                    break;
            }
        }
    }
}
