﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareReport
{
    using System;
    using System.IO;
    using System.Linq;
    using System.Xml;
    using Projects;
    using ToolDevelopmentKit;

    public class SoundProjectReportExporter
    {
        public const string SoundListName = "Sounds";
        public const string GroupListName = "Groups";
        public const string PlayerListName = "Players";
        public const string SoundSetBankListName = "SoundSetBanks";

        private HtmlReportExporter exporter;

        public string Generator { get; set; }

        public void Export(TextWriter writer, SoundProjectService projectService, SoundListOutput soundListOutput)
        {
            Ensure.Argument.NotNull(writer);
            Ensure.Argument.NotNull(projectService);
            Ensure.Argument.NotNull(soundListOutput);

            try
            {
                // TODO : ★名前が決まったらTitleを変更。サウンドプロジェクトレポートにする？
                this.exporter = new HtmlReportExporter()
                {
                    Title = soundListOutput.Title,
                    Generator = this.Generator,
                };

                using (XmlTextWriter xmlWriter = new XmlTextWriter(writer))
                {
                    xmlWriter.Formatting = Formatting.Indented;

                    xmlWriter.WriteStartElement("html");
                    this.exporter.WriteHtmlHeader(xmlWriter);

                    {
                        xmlWriter.WriteStartElement("body");

                        this.exporter.WriteTitle(xmlWriter);
                        this.exporter.WriteStartBodyDiv(xmlWriter);

                        this.ExportSummary(
                            xmlWriter,
                            projectService,
                            soundListOutput.OutputType);

                        switch (soundListOutput.OutputType)
                        {
                            case SoundListOutputType.EachSoundSet:
                                xmlWriter.WriteStartElement("div");

                                this.ExportSoundSetIndex(xmlWriter, projectService);
                                this.ExportItemListsBySoundSet(xmlWriter, projectService, soundListOutput);

                                xmlWriter.WriteEndElement();
                                break;

                            case SoundListOutputType.EntireSoundProject:
                                this.ExportItemLists(xmlWriter, projectService, soundListOutput);
                                break;

                            default:
                                throw new Exception("internal error : invalid output type.");
                        }

                        this.exporter.WriteFooter(xmlWriter, this.GetProjectFilePath(projectService));

                        this.exporter.WriteEndBodyDiv(xmlWriter);
                        xmlWriter.WriteFullEndElement();    // body
                    }

                    xmlWriter.WriteFullEndElement();    // html
                    xmlWriter.Flush();
                }
            }
            finally
            {
                this.exporter = null;
            }
        }

        private void ExportSummary(
            XmlTextWriter writer,
            SoundProjectService projectService,
            SoundListOutputType outputType)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(projectService);

            {
                writer.WriteStartElement("div");
                writer.WriteAttributeString("class", "block");

                {
                    writer.WriteStartElement("h2");
                    writer.WriteAttributeString("class", "blockheader");
                    writer.WriteString("summary");
                    writer.WriteEndElement();
                }

                {
                    writer.WriteStartElement("div");
                    writer.WriteAttributeString("class", "blockbody");

                    writer.WriteStartElement("h3");
                    writer.WriteString("Number of items");
                    writer.WriteEndElement();   // h3

                    {
                        writer.WriteStartElement("table");

                        this.ExportSummaryColumnHeader(writer);

                        {
                            writer.WriteStartElement("tbody");

                            if (outputType == SoundListOutputType.EachSoundSet)
                            {
                                foreach (var soundSet in projectService.SoundSets)
                                {
                                    this.ExportSummarySoundSetItems(writer, soundSet);
                                }
                            }

                            this.ExportSummaryProjectItems(writer, projectService);

                            writer.WriteEndElement();   // tbody
                        }

                        writer.WriteEndElement();   // table
                    }

                    writer.WriteEndElement();   // div.blockbody
                }

                writer.WriteEndElement();   // div.block
            }
        }

        private void ExportSummaryColumnHeader(XmlTextWriter writer)
        {
            writer.WriteStartElement("thead");
            writer.WriteStartElement("tr");

            {
                writer.WriteStartElement("th");
                writer.WriteString("soundset");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("sound");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("STRM");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("WSD");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("SEQ");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("WSDSET");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("SEQSET");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("BANK");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("WARC");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("PLAYER");
                writer.WriteEndElement();

                writer.WriteStartElement("th");
                writer.WriteString("GROUP");
                writer.WriteEndElement();
            }

            writer.WriteEndElement();   // tr
            writer.WriteEndElement();   // thead
        }

        private void ExportSummaryProjectItems(
            XmlTextWriter writer,
            SoundProjectService projectService)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(projectService);

            writer.WriteStartElement("tr");
            writer.WriteAttributeString("class", "total");

            {
                writer.WriteStartElement("td");
                writer.WriteString("total");
                writer.WriteEndElement();

                // sound
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.Sounds.Where(item => item.IsHierarchyEnabled).Count());

                // STRM
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.StreamSounds.Where(item => item.IsEnabled).Count());

                // WSD
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.WaveSounds.Where(item => item.IsHierarchyEnabled).Count());

                // SEQ
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.SequenceSounds.Where(item => item.IsHierarchyEnabled).Count());

                // WSDSET
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.WaveSoundSets.Where(item => item.IsEnabled).Count());

                // SEQSET
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.SequenceSoundSets.Where(item => item.IsEnabled).Count());

                // BANK
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.SoundSetBanks.Where(item => item.IsEnabled).Count());

                // WARC
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.WaveArchives.Where(item => item.IsEnabled).Count());

                // PLAYER
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.Players.Where(item => item.IsEnabled).Count());

                // GROUP
                this.ExportSummaryItemNumberCell(
                    writer,
                    projectService.Groups.Where(item => item.IsEnabled).Count());
            }

            writer.WriteEndElement();   // tr
        }

        private void ExportSummarySoundSetItems(XmlTextWriter writer, SoundSet soundset)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(soundset);

            writer.WriteStartElement("tr");

            {
                writer.WriteStartElement("td");
                writer.WriteString(soundset.Name);
                writer.WriteEndElement();

                // sound
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<Sound>().Count());

                // STRM
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<StreamSoundBase>().Count());

                // WSD
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<WaveSoundBase>().Count());

                // SEQ
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<SequenceSoundBase>().Count());

                // WSDSET
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<WaveSoundSetBase>().Count());

                // SEQSET
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<SequenceSoundSetBase>().Count());

                // BANK
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<SoundSetBankBase>().Count());

                // WARC
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<WaveArchiveBase>().Count());

                // PLAYER
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<PlayerBase>().Count());

                // GROUP
                this.ExportSummaryItemNumberCell(
                    writer,
                    soundset.EnumerateItems(true).OfType<GroupBase>().Count());
            }

            writer.WriteEndElement();   // tr
        }

        private void ExportSummaryItemNumberCell(XmlTextWriter writer, int number)
        {
            Assertion.Argument.NotNull(writer);

            writer.WriteStartElement("td");
            writer.WriteAttributeString("class", "number");

            writer.WriteString(number.ToString());

            writer.WriteEndElement();
        }

        private void ExportItemListsBySoundSet(
            XmlTextWriter writer, SoundProjectService projectService, SoundListOutput soundListOutput)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(projectService);
            Assertion.Argument.NotNull(soundListOutput);

            writer.WriteStartElement("div");
            writer.WriteAttributeString("id", "soundsetlist");

            int soundSetIndex = 0;
            int listIndex = 0;

            foreach (SoundSet soundSet in projectService.SoundSets)
            {
                writer.WriteStartElement("div");
                writer.WriteAttributeString("id", "soundset" + soundSetIndex);

                {
                    writer.WriteStartElement("div");
                    writer.WriteAttributeString("class", "block");

                    {
                        writer.WriteStartElement("h2");
                        writer.WriteAttributeString("class", "blockheader");
                        writer.WriteString(soundSet.Name);
                        writer.WriteEndElement();
                    }

                    {
                        writer.WriteStartElement("div");
                        writer.WriteAttributeString("class", "blockbody");

                        foreach (ListOutput listOutput in soundListOutput.ListOutputs)
                        {
                            if (!listOutput.IsOutput)
                            {
                                continue;
                            }

                            switch (listOutput.Name)
                            {
                                case SoundListName:
                                    this.ExportItemList<Sound>(
                                       writer,
                                       new SoundSetSoundListInfoProvider(
                                           projectService.Project.UserDataStructureSettings,
                                           new SoundProjectReportTraits(projectService.Project),
                                           soundSet,
                                           soundListOutput.IsIgnoreDisableItems,
                                           listOutput),
                                       new SoundInfoProvider(projectService.Project.UserDataStructureSettings.Settings), listIndex);
                                    break;

                                case GroupListName:
                                    this.ExportItemList<GroupBase>(
                                       writer,
                                       new SoundSetGroupListInfoProvider(
                                           new SoundProjectReportTraits(projectService.Project),
                                           soundSet,
                                           soundListOutput.IsIgnoreDisableItems,
                                           listOutput),
                                       new GroupInfoProvider(), listIndex);
                                    break;

                                case PlayerListName:
                                    this.ExportItemList<PlayerBase>(
                                      writer,
                                      new SoundSetPlayerListInfoProvider(
                                          new SoundProjectReportTraits(projectService.Project),
                                          soundSet,
                                          soundListOutput.IsIgnoreDisableItems,
                                          listOutput),
                                      new PlayerInfoProvider(), listIndex);
                                    break;

                                case SoundSetBankListName:
                                    this.ExportItemList<SoundSetBankBase>(
                                        writer,
                                        new SoundSetBankListInfoProvider(
                                            new SoundProjectReportTraits(projectService.Project),
                                            soundSet,
                                            soundListOutput.IsIgnoreDisableItems,
                                            listOutput),
                                        new SoundSetBankInfoProvider(), listIndex);
                                    break;

                                default:
                                    continue;
                            }

                            listIndex++;
                        }

                        writer.WriteEndElement();   // div.blockbody
                    }

                    writer.WriteEndElement();   // div.block
                }

                writer.WriteEndElement();   // div#(soundsetname)
                soundSetIndex++;
            }

            writer.WriteEndElement();   //div#soundsetlist
        }

        private void ExportSoundSetIndex(XmlTextWriter writer, SoundProjectService projectService)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(projectService);

            writer.WriteStartElement("p");
            writer.WriteString("SoundSet : ");

            int index = 0;

            foreach (SoundSet soundSet in projectService.SoundSets)
            {
                {
                    writer.WriteStartElement("a");
                    writer.WriteAttributeString("href", "#");
                    writer.WriteAttributeString("onclick", "ShowSoundSet('soundset" + index + "')");
                    writer.WriteString(soundSet.Name);
                    writer.WriteEndElement(); // a
                }

                writer.WriteString(" | ");
                index++;
            }

            {
                writer.WriteStartElement("a");
                writer.WriteAttributeString("href", "#");
                writer.WriteAttributeString("onclick", "ShowAllSoundSet()");
                writer.WriteString("[Show All]");
                writer.WriteEndElement(); // a
            }

            writer.WriteEndElement(); // p
        }

        private void ExportItemLists(
            XmlTextWriter writer, SoundProjectService projectService, SoundListOutput soundListOutput)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(projectService);
            Assertion.Argument.NotNull(soundListOutput);

            int listIndex = 0;

            foreach (ListOutput listOutput in soundListOutput.ListOutputs)
            {
                if (!listOutput.IsOutput)
                {
                    continue;
                }

                switch (listOutput.Name)
                {
                    case SoundListName:
                        this.ExportItemList<Sound>(
                           writer,
                           new ProjectItemListInfoProvider<Sound>(
                               new SoundProjectReportTraits(projectService.Project),
                               projectService.Sounds.Where(item => !soundListOutput.IsIgnoreDisableItems || item.IsHierarchyEnabled),
                               "Sound List",
                               listOutput),
                           new SoundInfoProvider(projectService.Project.UserDataStructureSettings.Settings), listIndex);
                        break;

                    case GroupListName:
                        this.ExportItemList<GroupBase>(
                           writer,
                           new ProjectItemListInfoProvider<GroupBase>(
                               new SoundProjectReportTraits(projectService.Project),
                               projectService.Groups.Where(item => !soundListOutput.IsIgnoreDisableItems || item.IsEnabled),
                               "Group List",
                               listOutput),
                           new GroupInfoProvider(), listIndex);
                        break;

                    case PlayerListName:
                        this.ExportItemList<PlayerBase>(
                          writer,
                          new ProjectItemListInfoProvider<PlayerBase>(
                              new SoundProjectReportTraits(projectService.Project),
                               projectService.Players.Where(item => !soundListOutput.IsIgnoreDisableItems || item.IsEnabled),
                               "Player List",
                              listOutput),
                          new PlayerInfoProvider(), listIndex);
                        break;

                    case SoundSetBankListName:
                        this.ExportItemList<SoundSetBankBase>(
                            writer,
                            new ProjectItemListInfoProvider<SoundSetBankBase>(
                                new SoundProjectReportTraits(projectService.Project),
                               projectService.SoundSetBanks.Where(item => !soundListOutput.IsIgnoreDisableItems || item.IsEnabled),
                               "Bank List",
                                listOutput),
                            new SoundSetBankInfoProvider(),
                            listIndex);
                        break;

                    default:
                        continue;
                }

                listIndex++;
            }
        }

        private void ExportItemList<TItem>(
            XmlTextWriter writer, IListInfoProvider<TItem> listInfoProvider,
            IListItemInfoProvider<TItem> itemInfoProvider, int listIndex)
            where TItem : Component
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(listInfoProvider);
            Assertion.Argument.NotNull(itemInfoProvider);

            // アイテムがない場合は、テーブルを作成しません。
            if (listInfoProvider.Items.FirstOrDefault() == null)
            {
                return;
            }

            {
                writer.WriteStartElement("h3");
                this.exporter.WriteShowButton(writer, "display", listIndex.ToString(), listInfoProvider.Caption);
                writer.WriteEndElement();
            }

            {
                writer.WriteStartElement("div");
                writer.WriteAttributeString("id", "display" + listIndex.ToString());

                this.exporter.WriteItemList<TItem>(writer, listInfoProvider, itemInfoProvider);

                writer.WriteEndElement();
            }
        }

        private string GetProjectFilePath(SoundProjectService projectService)
        {
            Assertion.Argument.NotNull(projectService);

            if (projectService.ProjectDocument == null)
            {
                return string.Empty;
            }

            if (projectService.ProjectDocument.Resource == null)
            {
                return string.Empty;
            }

            return projectService.ProjectDocument.Resource.Key;
        }
    }
}
