﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using NintendoWare.SoundFoundation.Commands;

namespace NintendoWare.SoundFoundation.CommandHandlers
{
    /// <summary>
    /// コマンドとコマンドハンドラを関連付けます。
    /// </summary>
    public class CommandBinding
    {
        private ICommandTarget _target;
        private string _commandID;

        private QueryStatusHandler _queryStatusHandler;
        private ExecuteHandler _executeHandler;

        public delegate CommandStatus QueryStatusHandler(Command command);
        public delegate bool ExecuteHandler(Command command);

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">コマンドを関連付けるコマンドターゲット。</param>
        /// <param name="commandID">関連付けるコマンドID。</param>
        /// <param name="queryStatusHandler">関連付ける QueryStatusHandler ハンドラ。</param>
        /// <param name="executeHandler">関連付ける Execute ハンドラ。</param>
        public CommandBinding(ICommandTarget target, string commandID,
                               QueryStatusHandler queryStatusHandler, ExecuteHandler executeHandler)
            : this(target, commandID, queryStatusHandler, executeHandler, null) { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">コマンドを関連付けるコマンドターゲット。</param>
        /// <param name="commandID">関連付けるコマンドID。</param>
        /// <param name="queryStatusHandler">関連付ける QueryStatusHandler ハンドラ。</param>
        /// <param name="executeHandler">関連付ける Execute ハンドラ。</param>
        /// <param name="executedHandler">対象コマンドが実行されると呼び出されるデリゲート。</param>
        public CommandBinding(ICommandTarget target, string commandID, QueryStatusHandler queryStatusHandler,
                               ExecuteHandler executeHandler, CommandEventHandler executedHandler)
        {
            Initialize(target, commandID, queryStatusHandler, executeHandler, executedHandler);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">コマンドを関連付けるコマンドターゲット。</param>
        /// <param name="handler">関連付けるコマンドハンドラ。</param>
        public CommandBinding(ICommandTarget target, CommandHandler handler)
            : this(target, handler, null) { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">コマンドを関連付けるコマンドターゲット。</param>
        /// <param name="handler">関連付けるコマンドハンドラ。</param>
        /// <param name="executedHandler">対象コマンドが実行されると呼び出されるデリゲート。</param>
        public CommandBinding(ICommandTarget target, CommandHandler handler,
                               CommandEventHandler executedHandler)
        {
            if (null == target) { throw new ArgumentNullException("target"); }
            if (null == handler) { throw new ArgumentNullException("handler"); }
            if (null == handler.TargetCommandID) { throw new ArgumentNullException("handler.TargetCommand"); }

            Initialize(target, handler.TargetCommandID, handler.QueryStatus, handler.Execute, executedHandler);
        }

        /// <summary>
        /// 対象コマンドターゲットを取得します。
        /// </summary>
        public ICommandTarget CommandTarget
        {
            get { return _target; }
        }

        /// <summary>
        /// 対象コマンドIDを取得します。
        /// </summary>
        public string CommandID
        {
            get { return _commandID; }
        }

        /// <summary>
        /// 対象コマンドが実行されると発生します。
        /// </summary>
        public event CommandEventHandler CommandExecuted;

        /// <summary>
        /// 対象コマンドターゲットが指定コマンドIDを実行対象とするかどうかを調べます。
        /// </summary>
        /// <param name="commandID">コマンドID。</param>
        /// <returns>実行対象である場合は 対象コマンドターゲット、実行対象でない場合は null。</returns>
        public ICommandTarget FindTarget(string commandID)
        {
            if (null == commandID) { throw new ArgumentNullException("commandID"); }
            return (commandID == _commandID) ? _target : null;
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうかを調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドの状態。</returns>
        public CommandStatus QueryStatus(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }
            if (!(command is Command)) { throw new ArgumentException("invalid command type."); }

            return _queryStatusHandler(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        public bool Execute(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }
            if (!(command is Command)) { throw new ArgumentException("invalid command type."); }

            bool result = _executeHandler(command);

            OnExecuted(new CommandEventArgs(command, result));

            return result;
        }

        /// <summary>
        /// 対象コマンドが実行されると発生します。
        /// </summary>
        /// <param name="e">コマンドイベントデータ。</param>
        protected virtual void OnExecuted(CommandEventArgs e)
        {
            if (null != CommandExecuted)
            {
                CommandExecuted(this, e);
            }
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="target">コマンドを関連付けるコマンドターゲット。</param>
        /// <param name="commandID">関連付けるコマンドID。</param>
        /// <param name="queryStatusHandler">関連付ける QueryStatusHandler ハンドラ。</param>
        /// <param name="executeHandler">関連付ける Execute ハンドラ。</param>
        /// <param name="executedHandler">対象コマンドが実行されると呼び出されるデリゲート。</param>
        private void Initialize(ICommandTarget target, string commandID, QueryStatusHandler queryStatusHandler,
                                 ExecuteHandler executeHandler, CommandEventHandler executedHandler)
        {
            if (null == target) { throw new ArgumentNullException("target"); }
            if (null == commandID) { throw new ArgumentNullException("commandID"); }
            if (null == queryStatusHandler) { throw new ArgumentNullException("queryStatusHandler"); }
            if (null == executeHandler) { throw new ArgumentNullException("executeHandler"); }

            _target = target;
            _commandID = commandID;
            _queryStatusHandler = queryStatusHandler;
            _executeHandler = executeHandler;

            if (null != executedHandler)
            {
                CommandExecuted += executedHandler;
            }
        }
    }
}
