﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// パケット通信をサポートする接続ポイントです。
    /// </summary>
    public abstract class ConnectionPoint
    {
        private WaveStreamInformation streamInformation;
        private ConnectionPoint connectedPoint;

        private bool isEndOfStream = false;

        /// <summary>
        /// 接続すると発生します。
        /// </summary>
        public event EventHandler Connected;

        /// <summary>
        /// パケットを受信すると発生します。
        /// </summary>
        public event EventHandler<PacketEventArgs> PacketReceived;

        /// <summary>
        /// ストリームの情報を取得します。
        /// </summary>
        public WaveStreamInformation StreamInformation
        {
            get { return this.streamInformation; }
            protected set { this.streamInformation = value; }
        }

        /// <summary>
        /// 接続先のポイントを取得します。
        /// </summary>
        public ConnectionPoint ConnectedPoint
        {
            get { return this.connectedPoint; }
        }

        /// <summary>
        /// 接続中かどうかを調べます。
        /// </summary>
        public bool IsConnected
        {
            get { return this.connectedPoint != null; }
        }

        /// <summary>
        /// ストリーム終端に達したかどうかを調べます。
        /// </summary>
        public bool IsEndOfStream
        {
            get { return this.isEndOfStream; }
        }

        /// <summary>
        /// 指定接続ポイントと接続します。
        /// </summary>
        /// <param name="point">接続先のポイントを指定します。</param>
        protected void ConnectToInternal(ConnectionPoint point)
        {
            Ensure.Argument.NotNull(point);

            if (this.IsConnected || point.IsConnected)
            {
                throw new Exception("point is already connected.");
            }

            this.connectedPoint = point;
            point.connectedPoint = this;
            point.streamInformation = this.streamInformation;

            this.OnConnected(EventArgs.Empty);
            point.OnConnected(EventArgs.Empty);
        }

        /// <summary>
        /// パケットを送信します。
        /// </summary>
        /// <param name="packet">送信するパケットを指定します。</param>
        protected void SendInternal(Packet packet)
        {
            Assertion.Argument.NotNull(packet);
            Ensure.Operation.True(!this.IsEndOfStream);

            if (this.connectedPoint == null)
            {
                throw new Exception("point not connected.");
            }

            if (packet.IsEndOfStream)
            {
                this.isEndOfStream = true;
            }

            this.connectedPoint.ReceiveInternal(packet);
        }

        /// <summary>
        /// パケットを受信します。
        /// </summary>
        /// <param name="packet"></param>
        protected void ReceiveInternal(Packet packet)
        {
            Assertion.Argument.NotNull(packet);
            Ensure.Operation.True(!this.IsEndOfStream);

            if (packet.IsEndOfStream)
            {
                this.isEndOfStream = true;
            }

            this.OnPacketReceived(new PacketEventArgs(packet));
        }

        /// <summary>
        /// 接続すると発生します。
        /// </summary>
        /// <param name="e">イベントデータを指定します。</param>
        protected virtual void OnConnected(EventArgs e)
        {
            if (this.Connected != null)
            {
                this.Connected(this, e);
            }
        }

        /// <summary>
        /// パケットを受信すると発生します。
        /// </summary>
        /// <param name="e">イベントデータを指定します。</param>
        protected virtual void OnPacketReceived(PacketEventArgs e)
        {
            if (this.PacketReceived != null)
            {
                this.PacketReceived(this, e);
            }
        }
    }

    /// <summary>
    /// パケット通信をサポートする接続ポイントです。
    /// </summary>
    /// <typeparam name="TAccessPoint">接続先ポイントの型を指定します。</typeparam>
    public class ConnectionPoint<TAccessPoint> : ConnectionPoint
        where TAccessPoint : ConnectionPoint
    {
        /// <summary>
        /// 接続先のポイントを取得します。
        /// </summary>
        public new TAccessPoint ConnectedPoint
        {
            get { return base.ConnectedPoint as TAccessPoint; }
        }

        /// <summary>
        /// 指定接続ポイントと接続します。
        /// </summary>
        /// <param name="point">接続先のポイントを指定します。</param>
        public void ConnectTo(TAccessPoint point)
        {
            this.ConnectToInternal(point);
        }
    }
}
