﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Binarization
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Linq;
    using Core;
    using ToolDevelopmentKit;

    public abstract class DomObject : INamedTreeObject
    {
        private string[] tags = new string[0];
        private object value;

        // プロセッサ
        private List<Attribute> attributes = new List<Attribute>();

        // 依存関係
        private string name;
        private DomObject parent;
        private DomObjectCollection children;
        private DomObject dependedObject;

        public DomObject(object value, bool hasChild)
        {
            this.value = value;

            // HACK : 百万単位でインスタンシングされる可能性があり、メモリ使用量に大きく影響するので、ここで最適化
            if (hasChild)
            {
                this.children = new DomObjectCollection(this);
                this.children.CollectionChanged += OnChildrenChanged;
            }
            else
            {
                this.children = DomObjectCollection.Empty;
            }
        }

        public event NameChangedEventHandler NameChanged;

        public event EventHandler ParentChanged;

        public string Name
        {
            get { return this.name; }
            set
            {
                if (value == this.name) { return; }

                string oldName = this.name;
                this.name = value;
                OnNameChanged(new NameChangedEventArgs(oldName, value));
            }
        }

        public string[] Tags
        {
            get { return this.tags; }
            set
            {
                Ensure.Argument.NotNull(value);
                this.tags = value;
            }
        }

        public virtual object Value
        {
            get { return this.value; }
        }

        public IList<Attribute> Attributes
        {
            get { return this.attributes; }
        }

        public DomObject Root
        {
            get
            {
                if (null == parent) { return this; }
                return parent.Root;
            }
        }

        public virtual DomObject Parent
        {
            get
            {
                return this.parent;
            }
            set
            {
                if (value == this.parent) { return; }

                this.parent = value;

                OnParentChanged(EventArgs.Empty);
            }
        }

        public virtual DomObjectCollection Children
        {
            get { return this.children; }
        }

        public bool IsDepended
        {
            get { return null != dependedObject; }
        }

        ITreeObject ITreeObject.Parent
        {
            get { return Parent; }
            set
            {
                if (null != value && !(value is DomObject))
                {
                    throw new ArgumentException("value must be DomObject.");
                }

                Parent = value as DomObject;
            }
        }

        ITreeObjectCollection ITreeObject.Children
        {
            get { return Children; }
        }

        protected DomObject DependedObject
        {
            get { return this.dependedObject; }
            set
            {
                if (value == this.dependedObject) { return; }

                this.dependedObject = value;
                value.DependedObject = this;
            }
        }

        public bool HasAttribute<TAttribute>()
            where TAttribute : Attribute
        {
            foreach (Attribute attr in this.attributes)
            {
                if (attr is TAttribute)
                {
                    return true;
                }
            }

            return false;
        }

        public TAttribute GetAttribute<TAttribute>()
            where TAttribute : Attribute
        {
            foreach (Attribute attr in this.attributes)
            {
                if (attr is TAttribute)
                {
                    return attr as TAttribute;
                }
            }

            return null;
        }

        public IEnumerable<TAttribute> GetAttributes<TAttribute>()
            where TAttribute : Attribute
        {
            foreach (Attribute attr in this.attributes)
            {
                if (attr is TAttribute)
                {
                    yield return attr as TAttribute;
                }
            }
        }

        protected virtual void OnNameChanged(NameChangedEventArgs e)
        {
            if (null != NameChanged)
            {
                NameChanged(this, e);
            }
        }

        protected virtual void OnParentChanged(EventArgs e)
        {
            if (null != ParentChanged)
            {
                ParentChanged(this, e);
            }
        }

        private void SetObjectsParent(IEnumerable<DomObject> objects, DomObject parent)
        {
            foreach (DomObject obj in objects)
            {
                obj.Parent = parent;
            }
        }

        private void OnChildrenChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                case NotifyCollectionChangedAction.Replace:
                    SetObjectsParent(e.NewItems.Cast<DomObject>(), this);
                    break;

                case NotifyCollectionChangedAction.Remove:
                    SetObjectsParent(e.NewItems.Cast<DomObject>(), null);
                    break;

                case NotifyCollectionChangedAction.Reset:
                    throw new Exception("unexpected event");
            }
        }
    }
}
