﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using Nintendo.ToolFoundation.IO;
using NintendoWare.SoundAdditionalInfoConverter.Logs;
using NintendoWare.SoundAdditionalInfoConverter.Resources;
using NintendoWare.SoundAdditionalInfoConverter.FileFormat;
using NintendoWare.SoundAdditionalInfoConverter.Binaries;
using YamlDotNet.RepresentationModel;
using YamlDotNet.Serialization;
using YamlDotNet.Serialization.NamingConventions;

namespace NintendoWare.SoundAdditionalInfoConverter
{
    internal class AdditionalInfoConverter
    {
        private readonly string CurrentFileVersion = "1.0.0.0";

        internal bool CheckFileVersion(StringReader reader)
        {
            var yaml = new YamlStream();
            yaml.Load(reader);

            if (yaml.Documents.Count <= 0)
            {
                Logger.GetInstance().OutputError(Messages.Error_Common_InvalidFileFormat);
                return false;
            }

            var rootNode = (YamlMappingNode)yaml.Documents[0].RootNode;

            try
            {
                var fileVersionNode = (YamlScalarNode)rootNode.Children[new YamlScalarNode("FileVersion")];
                if (fileVersionNode == null)
                {
                    Logger.GetInstance().OutputError(Messages.Error_Common_FileVersionNotFound);
                    return false;
                }

                if (fileVersionNode.Value != this.CurrentFileVersion)
                {
                    Logger.GetInstance().OutputError(Messages.Error_Common_InvalidFileVersion);
                    return false;
                }
            }
            catch (KeyNotFoundException)
            {
                Logger.GetInstance().OutputError(Messages.Error_Common_FileVersionNotFound);
                return false;
            }

            return true;
        }

        internal bool Convert(string filePath, string outputDir)
        {
            if (!File.Exists(filePath))
            {
                Logger.GetInstance().OutputError(Messages.Error_Common_InputFileNotFound);
                throw new FileNotFoundException();
            }

            // 入力ファイルの読み込み
            string yamlFile = string.Empty;
            using (var reader = new StreamReader(filePath))
            {
                yamlFile = reader.ReadToEnd();
            }

            // ファイルバージョンチェック
            if (!CheckFileVersion(new StringReader(yamlFile)))
            {
                return false;
            }

            // 入力ファイルのデシリアライズ
            DeserializerBuilder builder = new DeserializerBuilder();
            //builder.WithNamingConvention(new UnderscoredNamingConvention());
            builder.WithNamingConvention(new PascalCaseNamingConvention());

            AddtionalInfoFile additionalInfoFile = null;
            try
            {
                Deserializer deserializer = builder.Build();
                additionalInfoFile = deserializer.Deserialize<AddtionalInfoFile>(new StringReader(yamlFile));
            }
            catch (Exception e)
            {
                Logger.GetInstance().OutputError(Messages.Error_Common_InvalidFileFormat);
                Logger.GetInstance().OutputError(e.Message);
                return false;
            }

            // 出力ディレクトリの確認
            if (!Directory.Exists(outputDir))
            {
                Directory.CreateDirectory(outputDir);
            }

            // ストリームジャンプデータを出力
            StreamJumpDataFile streamJumpDataFile = additionalInfoFile.StreamJumpData;
            if (additionalInfoFile.StreamJumpData != null)
            {
                StreamJumpDataBinary binary = StreamJumpDataBinaryBuilder.Build(streamJumpDataFile);

                string inputFileName = Path.GetFileNameWithoutExtension(filePath);
                string outputFilePath = Path.Combine(outputDir, inputFileName) + binary.GetExtension();

                using (FileStream fileStream = new FileStream(outputFilePath, FileMode.Create))
                {
                    using (var binaryWriter = LittleEndianBinaryWriter.Create(fileStream))
                    {
                        binary.Output(binaryWriter);

                        binaryWriter.Close();
                    }

                    fileStream.Close();
                }
            }

            return true;
        }
    }
}
