﻿namespace Opal.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// 指定クラスのプロパティ値が記録可能なビューモデルクラスです。
    /// </summary>
    /// <typeparam name="TTarget">編集対象となるプロパティを保持するテンプレート型です。</typeparam>
    /// <typeparam name="TValue">編集対象となるプロパティのテンプレート型です。</typeparam>
    public class PropertyValueRecordableViewModel<TTarget, TValue> : ObservableValueViewModel
        where TTarget : class
    {
        private readonly Tuple<WeakReference<TTarget>, PropertyInfo> targetAndProperty;
        private TValue initValue;
        private TValue value;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="propertyInfo">編集対象の PropertyInfo です。</param>
        /// <param name="initValue">設定する初期値です。</param>
        public PropertyValueRecordableViewModel(
            TTarget target,
            PropertyInfo propertyInfo,
            TValue initValue)
        {
            Debug.Assert(target != null);
            Debug.Assert(propertyInfo != null);

            this.targetAndProperty =
                Tuple.Create(new WeakReference<TTarget>(target), propertyInfo);
            this.initValue = initValue;
            this.value = initValue;
        }

        /// <summary>
        /// 値を取得設定します。
        /// </summary>
        public TValue Value
        {
            get
            {
                return this.value;
            }

            set
            {
                this.SetProperty(ref this.value, value, () => this.ChangeValue());
            }
        }

        /// <summary>
        /// 初期値と編集値をを比較します。
        /// </summary>
        /// <returns>同一の場合は、true を返します。</returns>
        protected virtual bool CompareValues()
        {
            return this.initValue.Equals(this.value);
        }

        private void ChangeValue()
        {
            var weakTarget = this.targetAndProperty.Item1;
            TTarget target = null;
            if (weakTarget.TryGetTarget(out target))
            {
                this.targetAndProperty.Item2.SetValue(target, this.value, null);
                this.IsValueChanged = this.CompareValues();
            }

            target = null;
        }
    }
}
