﻿namespace Opal.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Input;
    using Opal.App;
    using Opal.Commands;
    using Opal.Menus;
    using Opal.Services;

    /// <summary>
    /// メインウィンドウビューモデルクラスです。
    /// </summary>
    public class MainWindowViewModel : ViewModel, IAdmin
    {
        private readonly ICommand undoCommand;
        private readonly ICommand redoCommand;
        private readonly ICommand closeCommand;
        private readonly ICommand removeHistoryCommand;

        private ICollectionView fileNewMenus = null;
        private ICollectionView fileOpenMenus = null;
        private ICollectionView fileSaveMenus = null;
        private ICollectionView displayMenus = null;
        private ICollectionView toolMenus = null;

        private ICollectionView documents = null;
        private ICollectionView tools = null;

        private DocumentViewModel activeDocument = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MainWindowViewModel()
        {
            this.undoCommand =
                new RelayCommand(
                    param => this.ExecuteUndo(param),
                    param => { return this.CanExecuteUndo(param); });

            this.redoCommand =
                new RelayCommand(
                    param => this.ExecuteRedo(param),
                    param => { return this.CanExecuteRedo(param); });

            this.removeHistoryCommand =
                new RelayCommand(
                    param => this.ExecuteRemoveHistory(param),
                    param => { return this.CanExecuteRemoveHistory(param); });

            this.closeCommand =
                new RelayCommand(
                    param => this.Close(),
                    param => { return this.CanClose(); });
        }

        /// <summary>
        /// ファイル新規作成メニューを取得します。
        /// </summary>
        public ICollectionView FileNewMenus
        {
            get
            {
                if (this.fileNewMenus == null)
                {
                    var menuManager = AppManager.GetMenuManager();
                    if (menuManager != null)
                    {
                        this.fileNewMenus = menuManager.GetMenus(MenuCategory.FileNew);
                    }
                }

                return this.fileNewMenus;
            }
        }

        /// <summary>
        /// ファイル開くメニューを取得します。
        /// </summary>
        public ICollectionView FileOpenMenus
        {
            get
            {
                if (this.fileOpenMenus == null)
                {
                    var menuManager = AppManager.GetMenuManager();
                    if (menuManager != null)
                    {
                        this.fileOpenMenus = menuManager.GetMenus(MenuCategory.FileOpen);
                    }
                }

                return this.fileOpenMenus;
            }
        }

        /// <summary>
        /// ファイル保存メニューを取得します。
        /// </summary>
        public ICollectionView FileSaveMenus
        {
            get
            {
                if (this.fileSaveMenus == null)
                {
                    var menuManager = AppManager.GetMenuManager();
                    if (menuManager != null)
                    {
                        this.fileSaveMenus = menuManager.GetMenus(MenuCategory.FileSave);
                    }
                }

                return this.fileSaveMenus;
            }
        }

        /// <summary>
        /// 表示メニューを取得します。
        /// </summary>
        public ICollectionView DisplayMenus
        {
            get
            {
                if (this.displayMenus == null)
                {
                    var menuManager = AppManager.GetMenuManager();
                    if (menuManager != null)
                    {
                        this.displayMenus = menuManager.GetDisplayMenus();
                    }
                }

                return this.displayMenus;
            }
        }

        /// <summary>
        /// ツールメニューを取得します。
        /// </summary>
        public ICollectionView ToolMenus
        {
            get
            {
                if (this.toolMenus == null)
                {
                    var menuManager = AppManager.GetMenuManager();
                    if (menuManager != null)
                    {
                        this.toolMenus = menuManager.GetMenus(MenuCategory.Tool);
                    }
                }

                return this.toolMenus;
            }
        }

        /// <summary>
        /// ドキュメントを取得します。
        /// </summary>
        public ICollectionView Documents
        {
            get
            {
                if (this.documents == null)
                {
                    var manager = AppManager.GetPaneManager();
                    if (manager != null)
                    {
                        this.documents = manager.GetDocuments();
                    }
                }

                return this.documents;
            }
        }

        /// <summary>
        /// ツールを取得します。
        /// </summary>
        public ICollectionView Tools
        {
            get
            {
                if (this.tools == null)
                {
                    var manager = AppManager.GetPaneManager();
                    if (manager != null)
                    {
                        this.tools = manager.GetTools();
                    }
                }

                return this.tools;
            }
        }

        /// <summary>
        /// アクティブなドキュメントを取得します。
        /// </summary>
        public DocumentViewModel ActiveDocument
        {
            get
            {
                return this.activeDocument;
            }

            internal set
            {
                if (this.activeDocument != value)
                {
                    this.SetProperty(ref this.activeDocument, value);
                }
            }
        }

        /// <summary>
        /// アンドゥコマンドを取得します。
        /// </summary>
        public ICommand UndoCommand
        {
            get
            {
                return this.undoCommand;
            }
        }

        /// <summary>
        /// リドゥコマンドを取得します。
        /// </summary>
        public ICommand RedoCommand
        {
            get
            {
                return this.redoCommand;
            }
        }

        /// <summary>
        /// 履歴削除コマンドを取得します。
        /// </summary>
        public ICommand RemoveHistoryCommand
        {
            get
            {
                return this.removeHistoryCommand;
            }
        }

        /// <summary>
        /// ウィンドウを閉じるコマンドを取得します。
        /// </summary>
        public ICommand CloseCommand
        {
            get
            {
                return this.closeCommand;
            }
        }

        /// <summary>
        /// 閉じる処理を行います。
        /// </summary>
        public virtual void Close()
        {
            AppManager.CloseOwnerWindow();
        }

        /// <summary>
        /// ウィンドウが閉じられるときに呼び出されます。
        /// </summary>
        /// <returns>閉じる処理をキャンセルする場合に true を返します。</returns>
        internal bool ClosingWindow()
        {
            return this.ClosingWindowInternal();
        }

        /// <summary>
        /// ウィンドウが閉じられるときに呼び出されます。
        /// </summary>
        /// <returns>閉じる処理をキャンセルする場合に true を返します。</returns>
        protected virtual bool ClosingWindowInternal()
        {
            return false;
        }

        /// <summary>
        /// 閉じる事が可能か判定します。
        /// </summary>
        /// <returns>閉じる事が可能な場合は、true を返します。</returns>
        protected virtual bool CanClose()
        {
            return true;
        }

        /// <summary>
        /// Undoを実行します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        protected virtual void ExecuteUndo(object parameter)
        {
        }

        /// <summary>
        /// Undoが実行可能かどうかを返します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        /// <returns>結果を返します。</returns>
        protected virtual bool CanExecuteUndo(object parameter)
        {
            return false;
        }

        /// <summary>
        /// Redoを実行します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        protected virtual void ExecuteRedo(object parameter)
        {
        }

        /// <summary>
        /// Redoが実行可能かどうかを返します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        /// <returns>結果を返します。</returns>
        protected virtual bool CanExecuteRedo(object parameter)
        {
            return false;
        }

        /// <summary>
        /// 履歴を消去します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        protected virtual void ExecuteRemoveHistory(object parameter)
        {
        }

        /// <summary>
        /// 履歴消去が実行可能かどうかを返します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        /// <returns>結果を返します。</returns>
        protected virtual bool CanExecuteRemoveHistory(object parameter)
        {
            return false;
        }
    }
}
