﻿namespace Opal.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Input;
    using Opal.Commands;
    using Opal.Services;

    /// <summary>
    /// ドキュメントペインビューモデルクラスです。
    /// </summary>
    public abstract class DocumentViewModel : PaneViewModel, IDocument
    {
        private readonly ICommand closeCommand;
        private bool isForceClose = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        protected DocumentViewModel()
        {
            this.closeCommand =
                new RelayCommand(param => this.Close(), param => { return true; });
        }

        /// <summary>
        /// アクティブ状態が変更された時のイベントです。
        /// </summary>
        internal event EventHandler<ActiveChangedEventArgs> ActiveChanged;

        /// <summary>
        /// ドキュメントが閉じられた時のイベントです。
        /// </summary>
        internal event EventHandler<DocumentClosedEventArgs> DocumentClosed;

        /// <summary>
        /// 閉じる処理のコマンドを取得します。
        /// </summary>
        public ICommand CloseCommand
        {
            get
            {
                return this.closeCommand;
            }
        }

        /// <summary>
        /// 強制で閉じる状態かを設定します。
        /// </summary>
        internal bool IsForceClose
        {
            set
            {
                this.isForceClose = value;
            }
        }

        /// <summary>
        /// 設定処理を実行します。
        /// </summary>
        /// <param name="parameter">設定用のパラメーターです。</param>
        public virtual void Setup(object parameter)
        {
        }

        /// <summary>
        /// 閉じる処理を実行します。
        /// </summary>
        public void Close()
        {
            if (this.isForceClose || this.TryClose())
            {
                if (this.DocumentClosed != null)
                {
                    this.DocumentClosed(this, new DocumentClosedEventArgs(this));
                }
            }
        }

        /// <summary>
        /// アクティブ状態が変更された時に実行される処理です。
        /// </summary>
        /// <param name="e">イベント引数です。</param>
        internal override void OnActiveChanged(ActiveChangedEventArgs e)
        {
            if (this.ActiveChanged != null)
            {
                this.ActiveChanged(this, e);
            }

            this.OnActiveChangedInternal(e);
        }

        /// <summary>
        /// 閉じる処理を試みます。
        /// </summary>
        /// <returns>閉じる場合は、true を返します。</returns>
        protected virtual bool TryClose()
        {
            return true;
        }

        /// <summary>
        /// アクティブ状態が変更された時に実行される内部処理です。
        /// </summary>
        /// <param name="e">イベント引数です。</param>
        protected virtual void OnActiveChangedInternal(ActiveChangedEventArgs e)
        {
        }
    }
}
