﻿namespace Opal.Utilities
{
    using System;
    using System.Diagnostics;
    using System.Linq.Expressions;
    using System.Reflection;
    using Opal.Events;

    /// <summary>
    /// イベントハンドラのユーティリティクラスです。
    /// </summary>
    public static class EventHandlerUtility
    {
        /// <summary>
        /// 弱参照イベントハンドラを作成します。
        /// </summary>
        /// <typeparam name="TEventArgs">イベント引数のテンプレートの型です。</typeparam>
        /// <param name="handler">イベントハンドラです。</param>
        /// <param name="unregisterCallback">イベントハンドラの登録を取り消すコールバックです。</param>
        /// <returns>イベントハンドラを返します。</returns>
        public static EventHandler<TEventArgs> MakeWeakEventHandler<TEventArgs>(this EventHandler<TEventArgs> handler, UnregisterCallback<TEventArgs> unregisterCallback)
            where TEventArgs : EventArgs
        {
            Debug.Assert(handler != null);
            Debug.Assert(!handler.Method.IsStatic && handler.Target != null);

            Type type = typeof(WeakEventHandler<,>).MakeGenericType(handler.Method.DeclaringType, typeof(TEventArgs));
            ConstructorInfo constructor = type.GetConstructor(new Type[] { typeof(EventHandler<TEventArgs>), typeof(UnregisterCallback<TEventArgs>) });

            IWeakEventHandler<TEventArgs> weakHandler = (IWeakEventHandler<TEventArgs>)constructor.Invoke(new object[] { handler, unregisterCallback });
            return weakHandler.Handler;
        }

        /// <summary>
        /// イベント名を取得します。
        /// </summary>
        /// <typeparam name="T">取得対象のテンプレートの型です。</typeparam>
        /// <param name="expression">イベント名取得のエクスプレッションです。</param>
        /// <returns>イベント名を返します。</returns>
        internal static string GetEventName<T>(Expression<Func<T>> expression)
        {
            return (expression.Body as MemberExpression).Member.Name;
        }
    }
}
