﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// オペレーションを束ねるクラスです。
    /// </summary>
    public sealed class OperationSet : Operation
    {
        private readonly List<Operation> operations = new List<Operation>();
        private readonly object syncRoot = new object();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OperationSet()
            : this(string.Empty)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="operations">生成時に登録したオペレーションの可変配列です。</param>
        public OperationSet(params Operation[] operations)
            : this(string.Empty, operations)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="label">設定するラベルです。</param>
        /// <param name="operations">生成時に登録したオペレーションの可変配列です。</param>
        public OperationSet(string label, params Operation[] operations)
        {
            Debug.Assert(label != null);

            if (operations != null)
            {
                this.operations.AddRange(operations);
            }
        }

        /// <summary>
        /// オペレーション数を取得します。
        /// </summary>
        /// <returns>オペレーション数を返します。</returns>
        public int Count
        {
            get
            {
                return this.operations.Count;
            }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>実行時に返される IOperation です。</returns>
        public override Operation Execute()
        {
            lock (this.syncRoot)
            {
#if false
                // TODO: 元のコードでは一度コピーして返しているが、コピーしない方が管理も楽なので、
                // しばらく運用して問題なければここは消す
                OperationSet operationSet = new OperationSet();
                foreach (var opration in this.operations)
                {
                    operationSet.Add(opration.Execute());
                }

                operationSet.Reverse();
                return operationSet;
#endif
                foreach (var opration in this.operations)
                {
                    opration.Execute();
                }

                this.Reverse();
                return this;
            }
        }

        /// <summary>
        /// アンドゥ可能か判定します。
        /// </summary>
        /// <returns>true ならばアンドゥ対象です。</returns>
        public override bool CanUndoable()
        {
            lock (this.syncRoot)
            {
                // Undoableが一つでも false なら Undo対象から除外
                if (this.operations.Any(obj => obj.CanUndoable() == false))
                {
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// 追加します。
        /// </summary>
        /// <param name="operation">追加する IOperation です。</param>
        public void Add(Operation operation)
        {
            Debug.Assert(operation != null);
            lock (this.syncRoot)
            {
                this.operations.Add(operation);
            }
        }

        /// <summary>
        /// リストを反転します。
        /// </summary>
        public void Reverse()
        {
            lock (this.syncRoot)
            {
                this.operations.Reverse();
            }
        }

        /// <summary>
        /// コレクション指定でオペレーションを追加します。
        /// </summary>
        /// <param name="operations">追加するオペレーションのコレクションです。</param>
        internal void AddRange(ICollection<Operation> operations)
        {
            Debug.Assert(operations != null);
            lock (this.syncRoot)
            {
                this.operations.AddRange(operations);
            }
        }
    }
}
