﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// リスト削除オペレーションです。
    /// </summary>
    /// <typeparam name="TObject">リストアイテムのテンプレート型です。</typeparam>
    public sealed class ListItemRemoveOperation<TObject> : ListItemOperation<TObject>
    {
        private int removeIndex = -1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        public ListItemRemoveOperation(IList<TObject> target, TObject item)
            : base(target, item)
        {
            Debug.Assert(target.Contains(item));
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        /// <param name="removeIndex">削除対象のアイテムのインデックスです。同一の値またはインスタンスが存在する場合に使用します。</param>
        public ListItemRemoveOperation(IList<TObject> target, TObject item, int removeIndex)
            : base(target, item)
        {
            Debug.Assert(target.Contains(item));
            Debug.Assert(target.Count > removeIndex && removeIndex >= 0);

            if (this.IsValueType)
            {
                Debug.Assert(object.Equals(target[removeIndex], item));
            }
            else
            {
                Debug.Assert(object.ReferenceEquals(target[removeIndex], item));
            }

            this.removeIndex = removeIndex;
        }

        /// <summary>
        /// Undo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        protected override void Undo(IList<TObject> target, TObject item)
        {
            if (this.removeIndex < 0)
            {
                return;
            }

            target.Insert(this.removeIndex, item);
        }

        /// <summary>
        /// Redo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        protected override void Redo(IList<TObject> target, TObject item)
        {
            this.removeIndex = target.IndexOf(item);
            if (this.removeIndex < 0)
            {
                // 対象が見つからない場合は何もしないこととする。
                // 例えば、あるアイテム追加する処理の中に存在する別の種類のアイテムをすべて取り除く処理があるときに
                // 複数アイテムを追加しようとすると複数回同じアイテムが取り除かれる。
                return;
            }

            target.RemoveAt(this.removeIndex);
        }
    }
}
