﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// リスト要素移動オペレーションです。
    /// </summary>
    /// <typeparam name="TObject">リストアイテムのテンプレート型です。</typeparam>
    public sealed class ListItemMoveOperation<TObject> : ListItemOperation<TObject>
    {
        private int moveIndex = -1;
        private int itemIndex = -1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        /// <param name="itemIndex">移動対象のアイテムのインデックスです。</param>
        /// <param name="moveIndex">移動後のアイテムのインデックスです。</param>
        public ListItemMoveOperation(IList<TObject> target, TObject item, int itemIndex, int moveIndex)
            : base(target, item)
        {
            Debug.Assert(target.Contains(item));
            Debug.Assert(target.Count > moveIndex && moveIndex >= 0);
            Debug.Assert(target.Count > itemIndex && itemIndex >= 0);

            this.moveIndex = moveIndex;
            this.itemIndex = itemIndex;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        /// <param name="moveIndex">移動後のアイテムのインデックスです。</param>
        public ListItemMoveOperation(IList<TObject> target, TObject item, int moveIndex)
            : base(target, item)
        {
            Debug.Assert(target.Contains(item));
            Debug.Assert(target.Count > moveIndex && moveIndex >= 0);

            this.moveIndex = moveIndex;
        }

        /// <summary>
        /// Undo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        protected override void Undo(IList<TObject> target, TObject item)
        {
            Debug.Assert(target.Contains(item));
            Debug.Assert(target.Count > this.moveIndex && this.moveIndex >= 0);

            if (this.IsValueType)
            {
                Debug.Assert(object.Equals(target[this.moveIndex], item));
            }
            else
            {
                Debug.Assert(object.ReferenceEquals(target[this.moveIndex], item));
            }

            target.RemoveAt(this.moveIndex);
            target.Insert(this.itemIndex, item);
        }

        /// <summary>
        /// Redo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="item">対象のアイテムです。</param>
        protected override void Redo(IList<TObject> target, TObject item)
        {
            if (this.itemIndex != -1)
            {
                if (this.IsValueType)
                {
                    Debug.Assert(object.Equals(target[this.itemIndex], item));
                }
                else
                {
                    Debug.Assert(object.ReferenceEquals(target[this.itemIndex], item));
                }
            }
            else
            {
                // インデックス指定がない場合はリストから対象のインデックスを探す
                this.itemIndex = target.IndexOf(item);
            }

            target.RemoveAt(this.itemIndex);
            target.Insert(this.moveIndex, item);
        }
    }
}
