﻿namespace Opal.Operations
{
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;

    /// <summary>
    /// キーが値に埋め込まれているコレクションの削除オペレーションです。
    /// </summary>
    /// <typeparam name="TKey">コレクションアイテムキーのテンプレート型です。</typeparam>
    /// <typeparam name="TItem">コレクションアイテムのテンプレート型です。</typeparam>
    public sealed class KeyedCollectionItemRemoveOperation<TKey, TItem> : KeyedCollectionItemOperation<TKey, TItem>
    {
        private TItem removedItem;
        private int removeIndex = -1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        public KeyedCollectionItemRemoveOperation(KeyedCollection<TKey, TItem> target, TKey key)
            : base(target, key)
        {
            Debug.Assert(key != null);
        }

        /// <summary>
        /// Undo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        protected override void Undo(KeyedCollection<TKey, TItem> target, TKey key)
        {
            if (this.removeIndex == -1)
            {
                return;
            }

            target.Insert(this.removeIndex, this.removedItem);
        }

        /// <summary>
        /// Redo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        protected override void Redo(KeyedCollection<TKey, TItem> target, TKey key)
        {
            try
            {
                this.removedItem = target[key];
            }
            catch (KeyNotFoundException)
            {
                // 対象が見つからない場合は何もしないこととする。
                // 例えば、あるアイテム追加する処理の中に存在する別の種類のアイテムをすべて取り除く処理があるときに
                // 複数アイテムを追加しようとすると複数回同じアイテムが取り除かれる。
                return;
            }

            this.removeIndex = target.IndexOf(this.removedItem);
            Debug.Assert(this.removeIndex >= 0);
            target.RemoveAt(removeIndex);
        }
    }
}
