﻿namespace Opal.Operations
{
    using System.Collections.ObjectModel;
    using System.Diagnostics;

    /// <summary>
    /// キーが値に埋め込まれているコレクションのオペレーションの基底クラスです。
    /// </summary>
    /// <typeparam name="TKey">コレクションアイテムキーのテンプレート型です。</typeparam>
    /// <typeparam name="TItem">コレクションアイテムのテンプレート型です。</typeparam>
    public abstract class KeyedCollectionItemOperation<TKey, TItem> : Operation
    {
        private readonly KeyedCollection<TKey, TItem> target;
        private readonly TKey key;
        private bool undoFlag = false;
        private bool isValueType = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        protected KeyedCollectionItemOperation(KeyedCollection<TKey, TItem> target)
        {
            Debug.Assert(target != null);

            this.isValueType = typeof(TItem).IsValueType;
            this.target = target;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        protected KeyedCollectionItemOperation(KeyedCollection<TKey, TItem> target, TKey key)
            : this(target)
        {
            Debug.Assert(key != null);
            this.key = key;
        }

        /// <summary>
        /// マージ可能か判定します。
        /// </summary>
        public override bool IsMergeable
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// キーを取得設定します。
        /// </summary>
        internal TKey Key
        {
            get;
            set;
        }

        /// <summary>
        /// 値型かどうかの状態を取得します。
        /// </summary>
        protected bool IsValueType
        {
            get
            {
                return this.isValueType;
            }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>実行時に返される Operation です。</returns>
        public override Operation Execute()
        {
            if (this.undoFlag)
            {
                this.Undo(this.target, this.key);
            }
            else
            {
                this.Redo(this.target, this.key);
            }

            this.undoFlag = !this.undoFlag;

            return this;
        }

        /// <summary>
        /// Undo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        protected abstract void Undo(KeyedCollection<TKey, TItem> target, TKey key);

        /// <summary>
        /// Redo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        protected abstract void Redo(KeyedCollection<TKey, TItem> target, TKey key);
    }
}
