﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class TargetShader : ObservableEntity<target_shaderType>, IDeepCopyable<TargetShader>, IDeepCopyFrom<TargetShader>
    {
        private readonly ObservableList<ShaderProgram> @shaderPrograms = new ObservableList<ShaderProgram>();
        private string @shadingModelName = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TargetShader()
        {
            this.@shaderPrograms.CollectionChanged += OnCollectionChanged<ShaderProgram>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public TargetShader(TargetShader source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public TargetShader(target_shaderType source)
        {
            this.@shaderPrograms.CollectionChanged += OnCollectionChanged<ShaderProgram>;
            if (source.@shader_program_array != null)
            {
                this.@shaderPrograms.Clear();
                foreach (var elem in source.@shader_program_array.Items)
                {
                    this.@shaderPrograms.Add(new ShaderProgram(elem));
                }
            }
            this.@shadingModelName = source.@shading_model_name;
        }

        public ObservableList<ShaderProgram> ShaderPrograms
        {
            get
            {
                return this.@shaderPrograms;
            }
        }

        public string ShadingModelName
        {
            get
            {
                return this.@shadingModelName;
            }

            set
            {
                if (this.@shadingModelName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@shadingModelName, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override target_shaderType CreateSerializableData()
        {
            Ensure.Operation.Range(this.ShaderPrograms.Count, 1, int.MaxValue);
            var writeData = new target_shaderType();
            if (this.@ShaderPrograms.Count > 0)
            {
                writeData.@shader_program_array = new shader_program_arrayType();
                writeData.@shader_program_array.Items = this.@shaderPrograms.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shader_program_array.Items.Length == 0)
                {
                    writeData.@shader_program_array = null;
                }
            }
            writeData.@shading_model_name = this.ShadingModelName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        TargetShader IDeepCopyable<TargetShader>.DeepCopy()
        {
            return new TargetShader(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(TargetShader source)
        {
            CopyElements(source.@shaderPrograms, this.@shaderPrograms);
            this.@shadingModelName = source.@shadingModelName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ShaderPrograms)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShadingModelName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@shaderPrograms)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@shaderPrograms)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@shaderPrograms)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
